package io.github.fishstiz.minecraftcursor;

import io.github.fishstiz.minecraftcursor.api.CursorType;
import io.github.fishstiz.minecraftcursor.compat.ExternalCursorTracker;
import io.github.fishstiz.minecraftcursor.config.Config;
import io.github.fishstiz.minecraftcursor.config.ConfigLoader;
import io.github.fishstiz.minecraftcursor.impl.CursorControllerImpl;
import io.github.fishstiz.minecraftcursor.impl.MinecraftCursorInitializerImpl;
import io.github.fishstiz.minecraftcursor.provider.CursorControllerProvider;
import io.github.fishstiz.minecraftcursor.platform.Services;
import io.github.fishstiz.minecraftcursor.util.CursorTypeUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Optional;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_364;
import net.minecraft.class_437;

public final class MinecraftCursor {
    public static final String MOD_ID = "minecraft-cursor";
    public static final Logger LOGGER = LoggerFactory.getLogger(MOD_ID);
    public static final Config CONFIG = ConfigLoader.load(Services.PLATFORM.getConfigDir().resolve(MOD_ID + ".json").toFile());
    private static final CursorTypeResolver RESOLVER = new CursorTypeResolver();
    private static final CursorControllerImpl CONTROLLER = new CursorControllerImpl();
    private static class_437 visibleHudScreen;

    private MinecraftCursor() {
    }

    static void init() {
        new MinecraftCursorInitializerImpl().init(CursorManager.INSTANCE, RESOLVER);

        Services.PLATFORM.getEntrypoints().forEach(entrypoint -> {
            try {
                entrypoint.init(CursorManager.INSTANCE, RESOLVER);
            } catch (LinkageError | Exception e) {
                LOGGER.error("[minecraft-cursor] Skipping invalid implementation of MinecraftCursorInitializer");
            }
        });

        CursorControllerProvider.init(CONTROLLER);
    }

    static void onScreenInit(class_310 minecraft, class_437 screen) {
        RESOLVER.lastFailedElement = "";

        if (minecraft.field_1755 == null) {
            CursorManager.INSTANCE.setCurrentCursor(CursorType.DEFAULT);
            visibleHudScreen = screen;
            return;
        }

        visibleHudScreen = null;
    }

    static void onScreenRender(class_310 minecraft, class_437 screen, class_332 guiGraphics, int mouseX, int mouseY) {
        visibleHudScreen = null;

        RESOLVER.getInspector().render(minecraft, screen, guiGraphics, mouseX, mouseY);

        if (ExternalCursorTracker.get().isCustom()) return;

        CursorManager.INSTANCE.setCurrentCursor(resolveCursorType(screen, mouseX, mouseY));
    }

    static void onClientTick(class_310 minecraft) {
        if (ExternalCursorTracker.get().isCustom()) return;

        if (minecraft.field_1755 == null && visibleHudScreen != null && !minecraft.field_1729.method_1613()) {
            double scale = minecraft.method_22683().method_4495();
            double mouseX = minecraft.field_1729.method_1603() / scale;
            double mouseY = minecraft.field_1729.method_1604() / scale;
            CursorManager.INSTANCE.setCurrentCursor(resolveCursorType(visibleHudScreen, mouseX, mouseY));
        } else if (minecraft.field_1755 == null && visibleHudScreen == null) {
            CursorManager.INSTANCE.setCurrentCursor(ExternalCursorTracker.get().getCursorOrDefault());
        }
    }

    private static CursorType resolveCursorType(class_437 screen, double mouseX, double mouseY) {
        if (!CursorManager.INSTANCE.isAdaptive()) {
            return CursorType.DEFAULT;
        }

        if (CONTROLLER.hasTransientCursor()) {
            return CONTROLLER.consumeTransientCursor();
        }

        CursorType externalCursor = ExternalCursorTracker.get().getCursorOrDefault();
        if (!externalCursor.isDefault()) {
            return externalCursor;
        }

        if (CursorTypeUtil.isGrabbing()) {
            return CursorType.GRABBING;
        }

        CursorType cursorType = RESOLVER.resolve(screen, mouseX, mouseY);

        if (!cursorType.isDefault()) {
            return cursorType;
        }

        Optional<class_364> child = screen.method_19355(mouseX, mouseY);

        if (child.isPresent()) {
            return RESOLVER.resolve(child.get(), mouseX, mouseY);
        }

        return CursorType.DEFAULT;
    }

    public static void toggleInspect() {
        RESOLVER.toggleInspector();
    }
}
