package io.github.fishstiz.minecraftcursor.cursor;

import io.github.fishstiz.minecraftcursor.CursorLoader;
import io.github.fishstiz.minecraftcursor.MinecraftCursor;
import io.github.fishstiz.minecraftcursor.api.CursorType;
import io.github.fishstiz.minecraftcursor.compat.ExternalCursorTracker;
import io.github.fishstiz.minecraftcursor.config.Config;
import io.github.fishstiz.minecraftcursor.util.NativeImageUtil;
import org.jetbrains.annotations.NotNull;
import org.lwjgl.glfw.GLFW;
import org.lwjgl.glfw.GLFWImage;
import org.lwjgl.system.MemoryUtil;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.function.Consumer;
import net.minecraft.class_1011;
import net.minecraft.class_2561;
import net.minecraft.class_2960;

import static io.github.fishstiz.minecraftcursor.util.SettingsUtil.*;

public class Cursor {
    private static final String IMG_TYPE = ".png";
    protected final Consumer<Cursor> onLoad;
    private final CursorType type;
    private final class_2960 location;
    private class_2561 text;
    private String base64Image;
    private double scale;
    private int xhot;
    private int yhot;
    private boolean enabled;
    private boolean loaded;
    private int textureWidth;
    private int textureHeight;
    private long id = MemoryUtil.NULL;

    public Cursor(CursorType type, Consumer<Cursor> onLoad) {
        this.type = type;
        this.onLoad = onLoad;
        this.location = CursorLoader.getDirectory().method_48331(type.getKey() + IMG_TYPE);
    }

    public void loadImage(@NotNull class_1011 image, Config.Settings settings) throws IOException {
        this.textureWidth = image.method_4307();
        this.textureHeight = image.method_4323();

        if (!SUPPORTED_SIZES.contains(textureWidth) || textureHeight % textureWidth != 0) {
            throw new IOException("Invalid cursor size. Width must be one of " + SUPPORTED_SIZES + ", and height must be a multiple of width.");
        }

        boolean cropped = false;
        class_1011 croppedImage = image;

        try {
            int size = this.textureWidth;
            if (image.method_4323() > size) {
                croppedImage = NativeImageUtil.cropImage(image, 0, 0, size, size);
                cropped = true;
            }

            this.base64Image = NativeImageUtil.toBase64String(croppedImage);
            this.enabled = settings.isEnabled();

            create(croppedImage, settings.getScale(), settings.getXHot(), settings.getYHot());
        } finally {
            if (cropped) {
                croppedImage.close();
            }
        }
    }

    protected void updateImage(double scale, int xhot, int yhot) {
        if (!this.isLoaded()) {
            return;
        }

        try (class_1011 image = NativeImageUtil.fromBase64String(base64Image)) {
            create(image, scale, xhot, yhot);
        } catch (IOException e) {
            MinecraftCursor.LOGGER.error("Error updating image of {}: {}", type, e);
        }
    }

    private void create(class_1011 image, double scale, int xhot, int yhot) {
        scale = sanitizeScale(scale);
        xhot = sanitizeHotspot(xhot, this);
        yhot = sanitizeHotspot(yhot, this);

        long glfwImageAddress = MemoryUtil.NULL;
        long previousId = this.id;
        ByteBuffer pixels = null;

        double autoScaled = getAutoScale(scale);
        try (class_1011 scaledImage = scale == 1 ? image : NativeImageUtil.scaleImage(image, autoScaled)) {
            int scaledXHot = scale == 1 ? xhot : (int) Math.round(xhot * autoScaled);
            int scaledYHot = scale == 1 ? yhot : (int) Math.round(yhot * autoScaled);
            int scaledWidth = scaledImage.method_4307();
            int scaledHeight = scaledImage.method_4323();

            GLFWImage glfwImage = GLFWImage.create();
            pixels = MemoryUtil.memAlloc(scaledWidth * scaledHeight * 4);
            NativeImageUtil.writePixelsRGBA(scaledImage, pixels);
            glfwImage.set(scaledWidth, scaledHeight, pixels);

            glfwImageAddress = glfwImage.address();
            ExternalCursorTracker.get().claimAddress(glfwImageAddress);
            this.id = GLFW.glfwCreateCursor(glfwImage, scaledXHot, scaledYHot);

            if (this.id == MemoryUtil.NULL) {
                MinecraftCursor.LOGGER.error("[minecraft-cursor] Error creating cursor '{}'. ", this.type.getKey());
                return;
            }

            loaded = true;
            this.scale = scale;
            this.xhot = xhot;
            this.yhot = yhot;

            if (this.onLoad != null) {
                this.onLoad.accept(this);
            }
        } finally {
            if (pixels != null) {
                MemoryUtil.memFree(pixels);
            }
            if (previousId != MemoryUtil.NULL && this.id != previousId) {
                GLFW.glfwDestroyCursor(previousId);
            }
            if (glfwImageAddress != MemoryUtil.NULL) {
                ExternalCursorTracker.get().unclaimAddress(glfwImageAddress);
            }
        }
    }

    public void destroy() {
        if (this.id != MemoryUtil.NULL) {
            GLFW.glfwDestroyCursor(this.id);
            this.id = MemoryUtil.NULL;
        }
    }

    public void reload() {
        if (this.isLoaded()) {
            this.updateImage(this.getScale(), this.getXHot(), this.getYHot());
        }
    }

    public void applySettings(Config.Settings settings) {
        this.enableWithoutLoading(settings.isEnabled());
        this.updateImage(settings.getScale(), settings.getXHot(), settings.getYHot());
    }

    protected void enableWithoutLoading(boolean enabled) {
        this.enabled = enabled;
        if (this.onLoad != null) this.onLoad.accept(this);
    }

    public boolean enable(boolean enabled) {
        if (!this.isLoaded() && !this.enabled && enabled && !CursorLoader.loadCursorTexture(this)) {
            return false;
        }

        boolean previous = this.enabled;
        this.enabled = enabled;

        if (previous != this.enabled && this.onLoad != null) {
            this.onLoad.accept(this);
        }
        return true;
    }

    public class_2960 getLocation() {
        return this.location;
    }

    public long getId() {
        return enabled ? id : MemoryUtil.NULL;
    }

    public @NotNull CursorType getType() {
        return type;
    }

    public @NotNull String getTypeKey() {
        return type.getKey();
    }

    public @NotNull class_2561 getText() {
        if (this.text == null) {
            this.text = class_2561.method_43471("minecraft-cursor.options.cursor-type." + type.getKey());
        }
        return this.text;
    }

    public double getScale() {
        return this.scale;
    }

    public void setScale(double scale) {
        updateImage(scale, this.xhot, this.yhot);
    }

    public int getXHot() {
        return this.xhot;
    }

    public void setXHot(double xhot) {
        setXHot((int) xhot);
    }

    public void setXHot(int xhot) {
        updateImage(this.scale, xhot, this.yhot);
    }

    public int getYHot() {
        return this.yhot;
    }

    public void setYHot(double yhot) {
        setYHot((int) yhot);
    }

    public void setYHot(int yhot) {
        updateImage(this.scale, this.xhot, yhot);
    }

    public void setHotspots(int xhot, int yhot) {
        updateImage(this.scale, xhot, yhot);
    }

    public boolean isEnabled() {
        return enabled;
    }

    public boolean isLoaded() {
        return loaded;
    }

    public int getTextureWidth() throws IllegalStateException {
        assertLoaded();
        return textureWidth;
    }

    public int getTextureHeight() throws IllegalStateException {
        assertLoaded();
        return textureHeight;
    }

    private void assertLoaded() {
        if (!this.isLoaded()) {
            throw new IllegalStateException("Cursor has not been loaded");
        }
    }
}
