package io.github.fishstiz.cursors_extended.gui.screen.panel;

import com.mojang.datafixers.util.Pair;
import io.github.fishstiz.cursors_extended.CursorsExtended;
import io.github.fishstiz.cursors_extended.config.Config;
import io.github.fishstiz.cursors_extended.cursor.Cursor;
import io.github.fishstiz.cursors_extended.cursor.CursorRegistry;
import io.github.fishstiz.cursors_extended.gui.MouseEvent;
import io.github.fishstiz.cursors_extended.gui.widget.ButtonWidget;
import io.github.fishstiz.cursors_extended.gui.widget.CursorPreviewWidget;
import io.github.fishstiz.cursors_extended.gui.widget.OptionsListWidget;
import io.github.fishstiz.cursors_extended.gui.widget.SliderWidget;
import io.github.fishstiz.cursors_extended.resource.texture.AnimatedCursorTexture;
import io.github.fishstiz.cursors_extended.resource.texture.CursorTexture;
import io.github.fishstiz.cursors_extended.util.CursorTypeUtil;
import io.github.fishstiz.cursors_extended.util.SettingsUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Iterator;
import java.util.function.Consumer;
import net.minecraft.class_11875;
import net.minecraft.class_2561;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_4185;
import net.minecraft.class_5244;
import net.minecraft.class_7919;

import static io.github.fishstiz.cursors_extended.CursorsExtended.CONFIG;

public class GlobalOptionsPanel extends AbstractOptionsPanel {
    private static final class_2561 TITLE = class_2561.method_43471("cursors_extended.options.global.title");
    private static final class_7919 SCALE_TOOLTIP = createGlobalTooltip(SCALE_TEXT);
    private static final class_7919 XHOT_TOOLTIP = createGlobalTooltip(XHOT_TEXT);
    private static final class_7919 YHOT_TOOLTIP = createGlobalTooltip(YHOT_TEXT);
    private static final class_2561 ANIMATIONS_TEXT = class_2561.method_43471("cursors_extended.options.global.animation");
    private static final class_7919 ANIMATIONS_INFO = class_7919.method_47407(class_2561.method_43471("cursors_extended.options.global.animation.tooltip"));
    private static final class_2561 RESET_TEXT = class_2561.method_43471("cursors_extended.options.resource_pack.reset");
    private static final class_7919 RESET_INFO = class_7919.method_47407(class_2561.method_43471("cursors_extended.options.resource_pack.reset.tooltip"));
    private static final int PREVIEW_BUTTON_SIZE = 20;
    private final Runnable refreshCursors;
    private @NotNull Iterator<Cursor> cursors = cursorIterator();
    private @NotNull Cursor currentCursor = getDefaultCursor();
    private OptionsListWidget optionList;
    private GlobalPreviewWidget previewWidget;
    private boolean scaling = false;

    public GlobalOptionsPanel(Runnable refreshCursors) {
        super(TITLE);
        this.refreshCursors = refreshCursors;
    }

    @Override
    protected void initContents() {
        this.recycleCursors();
        final Pair<Integer, Integer> maxHotspot = getMaxHotspots();

        this.previewWidget = new GlobalPreviewWidget(
                this.currentCursor,
                this.getFont(),
                new ButtonWidget(class_5244.field_39003, this::cycleOnPreviewPress).withSize(PREVIEW_BUTTON_SIZE)
        );

        this.optionList = new OptionsListWidget(this.getMinecraft(), this.getFont(), class_4185.field_39501, this.getSpacing());
        this.optionList.addToggleableSlider(
                new SliderWidget(
                        CONFIG.getGlobal().scale(),
                        SettingsUtil.SCALE_MIN,
                        SettingsUtil.SCALE_MAX,
                        SettingsUtil.SCALE_STEP,
                        this::onChangeScale,
                        this.index(SCALE_TEXT),
                        class_5244.field_39003,
                        SettingsUtil::getAutoText,
                        this::onSliderMouseEvent
                ),
                CONFIG.getGlobal().isScaleActive(),
                applyGlobalOnToggle(CONFIG.getGlobal()::setScaleActive),
                SCALE_TOOLTIP
        );
        this.optionList.addToggleableSlider(
                new SliderWidget(
                        CONFIG.getGlobal().xhot(),
                        SettingsUtil.HOT_MIN,
                        maxHotspot.getFirst(),
                        SettingsUtil.HOT_STEP,
                        this::onChangeXHot,
                        this.index(XHOT_TEXT),
                        HOTSPOT_SUFFIX,
                        null,
                        this::onSliderMouseEvent
                ),
                CONFIG.getGlobal().isXHotActive(),
                applyGlobalOnToggle(CONFIG.getGlobal()::setXHotActive),
                XHOT_TOOLTIP
        );
        this.optionList.addToggleableSlider(
                new SliderWidget(
                        CONFIG.getGlobal().yhot(),
                        SettingsUtil.HOT_MIN,
                        maxHotspot.getSecond(),
                        SettingsUtil.HOT_STEP,
                        this::onChangeYHot,
                        this.index(YHOT_TEXT),
                        HOTSPOT_SUFFIX,
                        null,
                        this::onSliderMouseEvent
                ),
                CONFIG.getGlobal().isYHotActive(),
                applyGlobalOnToggle(CONFIG.getGlobal()::setYHotActive),
                YHOT_TOOLTIP
        );
        this.optionList.addToggle(
                this.isAnimatedAny(),
                this::toggleCursorAnimations,
                this.index(ANIMATIONS_TEXT),
                ANIMATIONS_INFO,
                this.hasAnimationAny()
        );
        this.optionList.addWidget(
                new ButtonWidget(
                        this.index(RESET_TEXT),
                        this::resetCursorSettings
                ).withTooltip(RESET_INFO)
        );

        this.optionList.search(this.getSearch());

        this.addRenderableWidget(this.previewWidget);
        this.addRenderableWidget(this.optionList);
    }

    @Override
    protected void repositionContents(int x, int y) {
        if (this.previewWidget != null && this.optionList != null) {
            this.previewWidget.method_25358(this.method_25368());
            this.previewWidget.method_48229(x, y);

            this.optionList.method_55445(this.method_25368(), this.computeMaxHeight(y) - this.previewWidget.method_25364() - this.getSpacing());
            this.optionList.method_48229(x, this.previewWidget.method_55443() + this.getSpacing());
        }
    }

    private void onChangeScale(double scale) {
        CONFIG.getGlobal().setScale((float) scale);
        if (CONFIG.getGlobal().isScaleActive()) {
            setScale(currentCursor, (float) scale);
        }
    }

    private void onChangeXHot(double xhot) {
        CONFIG.getGlobal().setXHot(xhot);
        if (CONFIG.getGlobal().isXHotActive()) {
            setXHot(currentCursor, (int) xhot);
        }
    }

    private void onChangeYHot(double yhot) {
        CONFIG.getGlobal().setYHot(yhot);
        if (CONFIG.getGlobal().isYHotActive()) {
            setYHot(currentCursor, (int) yhot);
        }
    }

    private void onSliderMouseEvent(SliderWidget target, MouseEvent mouseEvent, double scale) {
        if (target.getPrefix().equals(SCALE_TEXT)) {
            this.scaling = (mouseEvent.clicked() || mouseEvent.dragged()) && currentCursor.isTextureEnabled();
        }

        if (mouseEvent.released()) {
            CursorRegistry registry = CursorsExtended.getInstance().getRegistry();
            if (target.getPrefix().equals(SCALE_TEXT) && CONFIG.getGlobal().isScaleActive()) {
                registry.getInternalCursors().forEach(cursor -> setScale(cursor, CONFIG.getGlobal().scale()));
            } else if (target.getPrefix().equals(XHOT_TEXT) && CONFIG.getGlobal().isXHotActive()) {
                registry.getInternalCursors().forEach(cursor -> setXHot(cursor, CONFIG.getGlobal().xhot()));
            } else if (target.getPrefix().equals(YHOT_TEXT) && CONFIG.getGlobal().isYHotActive()) {
                registry.getInternalCursors().forEach(cursor -> setYHot(cursor, CONFIG.getGlobal().yhot()));
            }
        }
    }

    private void cycleOnPreviewPress(class_4185 target) {
        target.method_25365(false);
        this.cycleCurrentCursor();
    }

    private void cycleCurrentCursor() {
        if (!this.cursors.hasNext()) {
            this.cursors = cursorIterator();
        }
        this.currentCursor = this.cursors.hasNext() ? this.cursors.next() : getDefaultCursor();

        if (this.previewWidget != null) {
            this.previewWidget.setCursor(this.currentCursor);
        }
    }

    private void recycleCursors() {
        this.cursors = cursorIterator();
        this.cycleCurrentCursor();
    }

    private void toggleCursorAnimations(boolean animated) {
        for (Cursor cursor : CursorsExtended.getInstance().getRegistry().getInternalCursors()) {
            if (cursor.getTexture() instanceof AnimatedCursorTexture animatedCursor) {
                animatedCursor.setAnimated(animated);
                CONFIG.getOrCreateSettings(cursor).setAnimated(animated);
            }
        }
    }

    private void resetCursorSettings() {
        for (Cursor cursor : CursorsExtended.getInstance().getRegistry().getCursors()) {
            CursorTexture texture = cursor.getTexture();
            if (texture != null) {
                Config.CursorSettings settings = CONFIG.getOrCreateSettings(cursor);
                settings.mergeAll(texture.metadata().cursor());
                cursor.setEnabled(settings.enabled());
                CONFIG.putCursorSettings(cursor, settings);
                CursorsExtended.getInstance().getLoader().updateTexture(cursor, CONFIG.getGlobal().apply(settings));
            }
        }

        this.refreshCursors.run();
        this.refreshWidgets();
        this.repositionElements();
    }

    @Override
    protected void searched(@NotNull String search, @Nullable class_2561 matched) {
        if (this.optionList != null) {
            this.optionList.search(search);
        }
    }

    public boolean hasAnimationAny() {
        for (Cursor cursor : CursorsExtended.getInstance().getRegistry().getInternalCursors()) {
            if (cursor.getTexture() instanceof AnimatedCursorTexture) {
                return true;
            }
        }
        return false;
    }

    public boolean isAnimatedAny() {
        for (Cursor cursor : CursorsExtended.getInstance().getRegistry().getInternalCursors()) {
            if (cursor.getTexture() instanceof AnimatedCursorTexture animatedCursor && animatedCursor.animated()) {
                return true;
            }
        }
        return false;
    }


    private static Consumer<Boolean> applyGlobalOnToggle(Consumer<Boolean> onToggle) {
        return value -> {
            onToggle.accept(value);
            CursorsExtended.getInstance().getRegistry().getInternalCursors().forEach(cursor ->
                    CursorsExtended.getInstance().getLoader().updateTexture(cursor, CONFIG.getGlobal().apply(CONFIG.getOrCreateSettings(cursor)))
            );
        };
    }

    private static class_7919 createGlobalTooltip(class_2561 message) {
        return class_7919.method_47407(class_2561.method_43469("cursors_extended.options.global.tooltip", message));
    }

    private static Iterator<Cursor> cursorIterator() {
        return CursorsExtended.getInstance().getRegistry().getInternalCursors()
                .stream()
                .filter(Cursor::isTextureEnabled)
                .iterator();
    }

    private static @NotNull Cursor getDefaultCursor() {
        return CursorsExtended.getInstance().getRegistry().get(class_11875.field_62449);
    }

    private static Pair<Integer, Integer> getMaxHotspots() {
        int currentMaxX = -1;
        int currentMaxY = -1;

        for (Cursor cursor : CursorsExtended.getInstance().getRegistry().getInternalCursors()) {
            if (cursor.hasTexture()) {
                int maxXHot = SettingsUtil.getMaxXHot(cursor);
                if (maxXHot > currentMaxX) {
                    currentMaxX = maxXHot;
                }

                int maxYHot = SettingsUtil.getMaxYHot(cursor);
                if (maxYHot > currentMaxY) {
                    currentMaxY = maxYHot;
                }
            }
        }
        return new Pair<>(currentMaxX != -1 ? currentMaxX : 0, currentMaxY != -1 ? currentMaxY : 0);
    }

    @Override
    public void method_25394(@NotNull class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
        super.method_25394(guiGraphics, mouseX, mouseY, partialTick);

        if (this.scaling) {
            guiGraphics.method_74037(CursorTypeUtil.arrowIfDefault(this.currentCursor.cursorType()));
        }
    }

    private static class GlobalPreviewWidget extends CursorPreviewWidget {
        private static final float CELL_DIVISOR = 32;
        private @NotNull Cursor cursor;

        public GlobalPreviewWidget(@NotNull Cursor cursor, @NotNull class_327 font, @Nullable class_4185 button) {
            super(cursor, font, button);

            this.cursor = cursor;
        }

        @Override
        protected void method_48579(@NotNull class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
            this.renderBackground(guiGraphics);
            this.renderPreviewText(guiGraphics);
            this.renderTestButton(guiGraphics, mouseX, mouseY, partialTick);
            this.renderRuler(guiGraphics, mouseX, mouseY);
            this.renderBorder(guiGraphics);

            if (this.method_49606()) guiGraphics.method_74037(this.cursors_extended$cursorType(mouseX, mouseY));
        }

        @Override
        protected float getCellWidth() {
            return this.method_25368() / CELL_DIVISOR;
        }

        @Override
        protected float getCellHeight() {
            return this.getCellWidth();
        }

        @Override
        protected boolean isOverflowing() {
            return false;
        }

        public void setCursor(@NotNull Cursor cursor) {
            this.cursor = cursor;
        }

        @Override
        public @NotNull Cursor getCursor() {
            return this.cursor;
        }
    }
}
