package io.github.fishstiz.cursors_extended.resource;

import com.google.common.hash.Hashing;
import io.github.fishstiz.cursors_extended.CursorsExtended;
import io.github.fishstiz.cursors_extended.config.Config;
import io.github.fishstiz.cursors_extended.config.CursorMetadata;
import io.github.fishstiz.cursors_extended.config.JsonLoader;
import io.github.fishstiz.cursors_extended.config.CursorProperties;
import io.github.fishstiz.cursors_extended.resource.texture.AnimationState;
import io.github.fishstiz.cursors_extended.cursor.CursorRegistry;
import io.github.fishstiz.cursors_extended.cursor.Cursor;
import io.github.fishstiz.cursors_extended.lifecycle.ClientStartedListener;
import io.github.fishstiz.cursors_extended.resource.texture.AnimatedCursorTexture;
import io.github.fishstiz.cursors_extended.resource.texture.BasicCursorTexture;
import io.github.fishstiz.cursors_extended.resource.texture.CursorTexture;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import org.jetbrains.annotations.NotNull;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import net.minecraft.class_1011;
import net.minecraft.class_11875;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3300;
import net.minecraft.class_3302;

import static io.github.fishstiz.cursors_extended.CursorsExtended.*;
import static io.github.fishstiz.cursors_extended.util.SettingsUtil.*;

public class CursorTextureLoader implements class_3302, ClientStartedListener {
    private static final class_2960 DIRECTORY = CursorsExtended.loc("textures/gui/sprites/cursors");
    private final Map<String, CursorMetadata> preparedMetadata = new Object2ObjectOpenHashMap<>();
    private final CursorRegistry registry;
    private class_310 minecraft;
    private boolean prepared;

    public CursorTextureLoader(CursorRegistry registry) {
        this.registry = registry;
    }

    @Override
    public void onClientStarted(class_310 minecraft) {
        this.minecraft = minecraft;
    }

    @Override
    public @NotNull CompletableFuture<Void> method_25931(
            class_11558 sharedState,
            Executor backgroundExecutor,
            class_4045 preparationBarrier,
            Executor gameExecutor
    ) {
        prepared = false;

        return CompletableFuture.runAsync(() -> prepare(sharedState.method_72361()), backgroundExecutor)
                .thenCompose(preparationBarrier::method_18352);
    }

    public void reload() {
        prepare(minecraft.method_1478());
        loadTextures(registry.getCursors());
    }

    private Optional<String> prepareMetadataHash(class_3300 manager, Iterable<Cursor> hashableCursors) {
        ByteArrayOutputStream out = new ByteArrayOutputStream();

        for (Cursor cursor : hashableCursors) {
            class_2960 path = getExpectedPath(cursor.cursorType());
            manager.method_14486(path.method_48331(CursorMetadata.FILE_TYPE)).ifPresentOrElse(resource -> {
                try {
                    // this is bugged, the resource should be the image, not the metadata, so that the image source and metadata source are the same.
                    // fixing this will reset configs
                    CursorMetadata metadata = loadMetadata(manager, path, resource.method_14480());
                    preparedMetadata.put(cursor.name(), metadata);

                    out.write(resource.method_14480().getBytes(StandardCharsets.UTF_8));
                    writeBytes(out, metadata);
                } catch (Exception ignore) {
                }
            }, () -> preparedMetadata.put(cursor.name(), new CursorMetadata()));
        }

        return out.size() == 0
                ? Optional.empty()
                : Optional.of(Hashing.murmur3_32_fixed().hashBytes(out.toByteArray()).toString());
    }


    private void prepare(class_3300 manager) {
        preparedMetadata.clear();

        prepareMetadataHash(manager, registry.getInternalCursors()).ifPresentOrElse(hash -> {
            if (!Objects.equals(CONFIG.getHash(), hash)) {
                LOGGER.info("[cursors_extended] Resource pack hash has changed, updating config...");
                CONFIG.setHash(hash);
                CONFIG.getGlobal().setActiveAll(false);
                CONFIG.markSettingsStale();
            }
        }, () -> {
            LOGGER.info("[cursors_extended] No resource pack detected.");
            CONFIG.setHash("");
        });

        for (Cursor cursor : registry.getCursors()) {
            cursor.prepareReload();

            CursorMetadata metadata = preparedMetadata.computeIfAbsent(cursor.name(), type -> {
                class_2960 path = getExpectedPath(cursor.cursorType());
                return manager.method_14486(path.method_48331(CursorMetadata.FILE_TYPE))
                        .map(resource -> loadMetadata(manager, path, resource.method_14480()))
                        .orElse(new CursorMetadata());
            });

            if (CONFIG.isStale(cursor)) {
                CONFIG.getOrCreateSettings(cursor).mergeSelective(metadata.cursor());
            }
        }

        prepared = true;
        CONFIG.save();
    }

    public void releaseTexture(Cursor cursor) {
        CursorTexture texture = cursor.getTexture();

        if (texture != null) {
            texture.close();
            cursor.setTexture(null);
            minecraft.execute(() -> minecraft.method_1531().method_4615(texture.comp_3627()));
        }
    }

    private boolean loadTexture(class_3300 manager, Cursor cursor) {
        if (!prepared) return false;

        class_2960 path = getExpectedPath(cursor.cursorType());
        boolean loaded = false;

        try {
            var resource = manager.method_14486(path).orElse(null);
            if (resource != null) {
                try (InputStream in = resource.method_14482(); class_1011 image = class_1011.method_4309(in)) {
                    assertImageSize(image.method_4307(), image.method_4323());

                    CursorMetadata metadata = preparedMetadata.getOrDefault(cursor.name(), loadMetadata(manager, path, resource.method_14480()));
                    CursorProperties settings = CONFIG.getGlobal().apply(CONFIG.getOrCreateSettings(cursor));
                    CursorTexture texture = metadata.animation() != null
                            ? new AnimatedCursorTexture(AnimationState.of(metadata.animation().mode()), image, path, metadata, settings)
                            : new BasicCursorTexture(image, path, metadata, settings);

                    cursor.setTexture(texture);
                    minecraft.execute(() -> minecraft.method_1531().method_4615(path));
                    loaded = true;
                }
            }
        } catch (Exception e) {
            LOGGER.error("[cursors_extended] Failed to load cursor texture for '{}'. ", cursor.cursorType(), e);
        }

        if (!loaded) releaseTexture(cursor);
        cursor.reloaded();
        return loaded;
    }

    public boolean loadTexture(Cursor cursor) {
        return loadTexture(minecraft.method_1478(), cursor);
    }

    private void loadTextures(class_3300 manager, Collection<Cursor> cursors) {
        for (Cursor cursor : cursors) {
            loadTexture(manager, cursor);
        }
    }

    public void loadTextures(Collection<Cursor> cursors) {
        loadTextures(minecraft.method_1478(), cursors);
    }

    public void updateTexture(Cursor cursor, float scale, int xhot, int yhot) {
        CursorTexture texture = cursor.getTexture();
        if (texture == null) return;

        Config.CursorSettings settings = CONFIG.getOrCreateSettings(cursor).copy();
        settings.setScale(scale);
        settings.setXHot(cursor, xhot);
        settings.setYHot(cursor, yhot);

        try {
            CursorTexture updatedTexture = texture.recreate(settings);
            cursor.setTexture(updatedTexture);
            texture.close();
        } catch (Exception e) {
            LOGGER.error("[cursors_extended] Failed to update texture of cursor '{}'.", cursor.cursorType(), e);
            cursor.setTexture(texture);
        }
    }

    public void updateTexture(Cursor cursor, CursorProperties settings) {
        updateTexture(cursor, settings.scale(), settings.xhot(), settings.yhot());
    }

    private CursorMetadata loadMetadata(class_3300 manager, class_2960 location, String source) {
        return manager.method_14489(location.method_48331(CursorMetadata.FILE_TYPE))
                .stream()
                .filter(metadata -> metadata.method_14480().equals(source))
                .findFirst()
                .map(metadata -> JsonLoader.fromResource(CursorMetadata.class, metadata))
                .orElse(new CursorMetadata());
    }

    private static void writeBytes(ByteArrayOutputStream out, CursorMetadata metadata) throws IOException {
        CursorMetadata.CursorSettings cs = metadata.cursor();
        out.write(Float.toString(cs.scale()).getBytes(StandardCharsets.UTF_8));
        out.write(Integer.toString(cs.xhot()).getBytes(StandardCharsets.UTF_8));
        out.write(Integer.toString(cs.yhot()).getBytes(StandardCharsets.UTF_8));
        out.write(Boolean.toString(cs.enabled()).getBytes(StandardCharsets.UTF_8));
        if (cs.animated() != null) {
            out.write(Boolean.toString(cs.animated()).getBytes(StandardCharsets.UTF_8));
        }

        CursorMetadata.Animation anim = metadata.animation();
        if (anim != null) {
            out.write(anim.mode().name().getBytes(StandardCharsets.UTF_8));
            out.write(Integer.toString(anim.frametime()).getBytes(StandardCharsets.UTF_8));
            if (anim.width() != null) {
                out.write(Integer.toString(anim.width()).getBytes(StandardCharsets.UTF_8));
            }
            if (anim.height() != null) {
                out.write(Integer.toString(anim.height()).getBytes(StandardCharsets.UTF_8));
            }
            for (CursorMetadata.Animation.Frame f : anim.frames()) {
                out.write(Integer.toString(f.index()).getBytes(StandardCharsets.UTF_8));
                out.write(Integer.toString(f.clampedTime(anim)).getBytes(StandardCharsets.UTF_8));
            }
        }
    }

    private static class_2960 getExpectedPath(class_11875 cursorType) {
        return DIRECTORY.method_48331("/" + cursorType.toString() + ".png");
    }

    public static class_2960 getDir() {
        return DIRECTORY;
    }
}
