/*
 * Copyright (c) 2019-2025 Wurst-Imperium and contributors.
 *
 * This source code is subject to the terms of the GNU General Public
 * License, version 3. If a copy of the GPL was not distributed with this
 * file, You can obtain one at: https://www.gnu.org/licenses/gpl-3.0.txt
 */
package net.wurstclient.zoom.test;

import java.io.File;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;

import org.lwjgl.glfw.GLFW;

import com.mojang.brigadier.ParseResults;
import com.mojang.brigadier.exceptions.CommandSyntaxException;

import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1074;
import net.minecraft.class_1157;
import net.minecraft.class_11908;
import net.minecraft.class_11910;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_310;
import net.minecraft.class_318;
import net.minecraft.class_339;
import net.minecraft.class_342;
import net.minecraft.class_3928;
import net.minecraft.class_4068;
import net.minecraft.class_4185;
import net.minecraft.class_433;
import net.minecraft.class_437;
import net.minecraft.class_442;
import net.minecraft.class_490;
import net.minecraft.class_5498;
import net.minecraft.class_5676;
import net.minecraft.class_634;

public enum WiModsTestHelper
{
	;
	
	private static final AtomicInteger screenshotCounter = new AtomicInteger(0);
	
	/**
	 * Runs the given consumer on Minecraft's main thread and waits for it to
	 * complete.
	 */
	public static void submitAndWait(Consumer<class_310> consumer)
	{
		class_310 mc = class_310.method_1551();
		mc.method_20493(() -> consumer.accept(mc)).join();
	}
	
	/**
	 * Runs the given function on Minecraft's main thread, waits for it to
	 * complete, and returns the result.
	 */
	public static <T> T submitAndGet(Function<class_310, T> function)
	{
		class_310 mc = class_310.method_1551();
		return mc.method_5385(() -> function.apply(mc)).join();
	}
	
	/**
	 * Waits for the given duration.
	 */
	public static void waitFor(Duration duration)
	{
		try
		{
			Thread.sleep(duration.toMillis());
			
		}catch(InterruptedException e)
		{
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * Waits until the given condition is true, or fails if the timeout is
	 * reached.
	 */
	public static void waitUntil(String event,
		Predicate<class_310> condition, Duration maxDuration)
	{
		LocalDateTime startTime = LocalDateTime.now();
		LocalDateTime timeout = startTime.plus(maxDuration);
		System.out.println("Waiting until " + event);
		
		while(true)
		{
			if(submitAndGet(condition::test))
			{
				double seconds =
					Duration.between(startTime, LocalDateTime.now()).toMillis()
						/ 1000.0;
				System.out.println(
					"Waiting until " + event + " took " + seconds + "s");
				break;
			}
			
			if(LocalDateTime.now().isAfter(timeout))
				throw new RuntimeException(
					"Waiting until " + event + " took too long");
			
			waitFor(Duration.ofMillis(50));
		}
	}
	
	/**
	 * Waits until the given condition is true, or fails after 10 seconds.
	 */
	public static void waitUntil(String event,
		Predicate<class_310> condition)
	{
		waitUntil(event, condition, Duration.ofSeconds(10));
	}
	
	/**
	 * Waits until the given screen is open, or fails after 10 seconds.
	 */
	public static void waitForScreen(Class<? extends class_437> screenClass)
	{
		waitUntil("screen " + screenClass.getName() + " is open",
			mc -> screenClass.isInstance(mc.field_1755));
	}
	
	/**
	 * Waits for the fading animation of the title screen to finish, or fails
	 * after 10 seconds.
	 */
	public static void waitForTitleScreenFade()
	{
		waitUntil("title screen fade is complete", mc -> {
			if(!(mc.field_1755 instanceof class_442 titleScreen))
				return false;
			
			return !titleScreen.field_18222;
		});
	}
	
	/**
	 * Waits until the red overlay with the Mojang logo and progress bar goes
	 * away, or fails after 5 minutes.
	 */
	public static void waitForResourceLoading()
	{
		waitUntil("loading is complete", mc -> mc.method_18506() == null,
			Duration.ofMinutes(5));
	}
	
	public static void waitForWorldLoad()
	{
		waitUntil("world is loaded",
			mc -> mc.field_1687 != null
				&& !(mc.field_1755 instanceof class_3928),
			Duration.ofMinutes(30));
	}
	
	public static void waitForWorldTicks(int ticks)
	{
		long startTicks = submitAndGet(mc -> mc.field_1687.method_75260());
		waitUntil(ticks + " world ticks have passed",
			mc -> mc.field_1687.method_75260() >= startTicks + ticks,
			Duration.ofMillis(ticks * 100).plusMinutes(5));
	}
	
	public static void waitForBlock(int relX, int relY, int relZ, class_2248 block)
	{
		class_2338 pos =
			submitAndGet(mc -> mc.field_1724.method_24515().method_10069(relX, relY, relZ));
		waitUntil(
			"block at ~" + relX + " ~" + relY + " ~" + relZ + " ("
				+ pos.method_23854() + ") is " + block,
			mc -> mc.field_1687.method_8320(pos).method_26204() == block);
	}
	
	/**
	 * Waits for 50ms and then takes a screenshot with the given name.
	 */
	public static void takeScreenshot(String name)
	{
		takeScreenshot(name, Duration.ofMillis(50));
	}
	
	/**
	 * Waits for the given delay and then takes a screenshot with the given
	 * name.
	 */
	public static void takeScreenshot(String name, Duration delay)
	{
		waitFor(delay);
		
		String count =
			String.format("%02d", screenshotCounter.incrementAndGet());
		String filename = count + "_" + name + ".png";
		File gameDir = FabricLoader.getInstance().getGameDir().toFile();
		
		submitAndWait(mc -> class_318.method_22690(gameDir, filename,
			mc.method_1522(), 1, message -> {}));
	}
	
	/**
	 * Returns the first button on the current screen that has the given
	 * translation key, or fails if not found.
	 *
	 * <p>
	 * For non-translated buttons, the translationKey parameter should be the
	 * raw button text instead.
	 */
	public static class_4185 findButton(class_310 mc,
		String translationKey)
	{
		String message = class_1074.method_4662(translationKey);
		
		for(class_4068 drawable : mc.field_1755.field_33816)
			if(drawable instanceof class_4185 button
				&& button.method_25369().getString().equals(message))
				return button;
			
		throw new RuntimeException(message + " button could not be found");
	}
	
	/**
	 * Looks for the given button at the given coordinates and fails if it is
	 * not there.
	 */
	public static void checkButtonPosition(class_4185 button, int expectedX,
		int expectedY)
	{
		String buttonName = button.method_25369().getString();
		
		if(button.method_46426() != expectedX)
			throw new RuntimeException(buttonName
				+ " button is at the wrong X coordinate. Expected X: "
				+ expectedX + ", actual X: " + button.method_46426());
		
		if(button.method_46427() != expectedY)
			throw new RuntimeException(buttonName
				+ " button is at the wrong Y coordinate. Expected Y: "
				+ expectedY + ", actual Y: " + button.method_46427());
	}
	
	/**
	 * Clicks the button with the given translation key, or fails after 10
	 * seconds.
	 *
	 * <p>
	 * For non-translated buttons, the translationKey parameter should be the
	 * raw button text instead.
	 */
	public static void clickButton(String translationKey)
	{
		String buttonText = class_1074.method_4662(translationKey);
		
		waitUntil("button saying " + buttonText + " is visible", mc -> {
			class_437 screen = mc.field_1755;
			if(screen == null)
				return false;
			
			for(class_4068 drawable : screen.field_33816)
			{
				if(drawable instanceof class_339 widget)
					if(clickButtonInWidget(widget, buttonText))
						return true;
			}
			
			return false;
		});
	}
	
	private static boolean clickButtonInWidget(class_339 widget,
		String buttonText)
	{
		class_11910 pressContext = new class_11910(GLFW.GLFW_KEY_UNKNOWN, 0);
		
		if(widget instanceof class_4185 button
			&& buttonText.equals(button.method_25369().getString()))
		{
			button.method_25306(pressContext);
			return true;
		}
		
		if(widget instanceof class_5676<?> button
			&& buttonText.equals(button.field_27963.getString()))
		{
			button.method_25306(pressContext);
			return true;
		}
		
		return false;
	}
	
	/**
	 * Types the given text into the nth text field on the current screen, or
	 * fails after 10 seconds.
	 */
	public static void setTextFieldText(int index, String text)
	{
		waitUntil("text field #" + index + " is visible", mc -> {
			class_437 screen = mc.field_1755;
			if(screen == null)
				return false;
			
			int i = 0;
			for(class_4068 drawable : screen.field_33816)
			{
				if(!(drawable instanceof class_342 textField))
					continue;
				
				if(i == index)
				{
					textField.method_1852(text);
					return true;
				}
				
				i++;
			}
			
			return false;
		});
	}
	
	public static void setKeyPressState(int key, boolean pressed)
	{
		setKeyPressState(key, pressed, 0);
	}
	
	public static void setKeyPressState(int key, boolean pressed, int modifiers)
	{
		submitAndWait(mc -> {
			long window = mc.method_22683().method_4490();
			int action = pressed ? 1 : 0;
			int scancode = 0;
			class_11908 context = new class_11908(key, scancode, modifiers);
			mc.field_1774.method_1466(window, action, context);
		});
	}
	
	public static void scrollMouse(int horizontal, int vertical)
	{
		submitAndWait(mc -> mc.field_1729.method_1598(mc.method_22683().method_4490(),
			horizontal, vertical));
	}
	
	public static void closeScreen()
	{
		submitAndWait(mc -> mc.method_1507(null));
	}
	
	public static void openGameMenu()
	{
		submitAndWait(mc -> mc.method_1507(new class_433(true)));
	}
	
	public static void openInventory()
	{
		submitAndWait(mc -> mc.method_1507(new class_490(mc.field_1724)));
	}
	
	public static void toggleDebugHud()
	{
		submitAndWait(mc -> mc.field_61504.method_72774());
	}
	
	public static void setPerspective(class_5498 perspective)
	{
		submitAndWait(mc -> mc.field_1690.method_31043(perspective));
	}
	
	public static void dismissTutorialToasts()
	{
		submitAndWait(mc -> mc.method_1577().method_4910(class_1157.field_5653));
	}
	
	public static void clearChat()
	{
		submitAndWait(mc -> mc.field_1705.method_1743().method_1808(true));
	}
	
	/**
	 * Runs the given chat command and waits one tick for the action to
	 * complete.
	 *
	 * <p>
	 * Do not put a / at the start of the command.
	 */
	public static void runChatCommand(String command)
	{
		System.out.println("Running command: /" + command);
		submitAndWait(mc -> {
			class_634 netHandler = mc.method_1562();
			
			// Validate command using client-side command dispatcher
			ParseResults<?> results = netHandler.method_2886()
				.parse(command, netHandler.method_2875());
			
			// Command is invalid, fail the test
			if(!results.getExceptions().isEmpty())
			{
				StringBuilder errors =
					new StringBuilder("Invalid command: " + command);
				for(CommandSyntaxException e : results.getExceptions().values())
					errors.append("\n").append(e.getMessage());
				
				throw new RuntimeException(errors.toString());
			}
			
			// Command is valid, send it
			netHandler.method_45730(command);
		});
		waitForWorldTicks(1);
	}
	
	public static void assertOneItemInSlot(int slot, class_1792 item)
	{
		submitAndWait(mc -> {
			class_1799 stack = mc.field_1724.method_31548().method_5438(slot);
			if(!stack.method_31574(item) || stack.method_7947() != 1)
				throw new RuntimeException(
					"Expected 1 " + item.method_63680().getString() + " at slot "
						+ slot + ", found " + stack.method_7947() + " "
						+ stack.method_7909().method_63680().getString() + " instead");
		});
	}
	
	public static void assertNoItemInSlot(int slot)
	{
		submitAndWait(mc -> {
			class_1799 stack = mc.field_1724.method_31548().method_5438(slot);
			if(!stack.method_7960())
				throw new RuntimeException("Expected no item in slot " + slot
					+ ", found " + stack.method_7947() + " "
					+ stack.method_7909().method_63680().getString() + " instead");
		});
	}
}
