/**
 * ═══════════════════════════════════════════════════════════════════════════════════════
 *                   LUMINARA SPOOKY SHADER - CORE COLOR SYSTEM
 *                    Enhanced Horror & Eerie Atmosphere Color Suite
 * ═══════════════════════════════════════════════════════════════════════════════════════
 * 
 * Modified color management system for spooky atmospheric rendering with
 * dark, desaturated colors and eerie environmental effects.
 * 
 * Spooky Enhancements:
 * - Darkened and desaturated color palette
 * - Eerie greenish/purple tints
 * - Enhanced fog and mist effects
 * - Oppressive atmospheric lighting
 * - Creepy time-of-day transitions
 * 
 * VcorA Spooky Edition - Part of Luminara Shader Pack
 * ═══════════════════════════════════════════════════════════════════════════════════════
 */

// Include spooky configuration
#include "/lib/../horror_config.glsl"


// ═══════════════════════════════════════════════════════════════════════════════════════
//                           DYNAMIC SKY COLOR RENDERING SYSTEM
// ═══════════════════════════════════════════════════════════════════════════════════════

#ifndef INCLUDE_SKY_COLORS
    #define INCLUDE_SKY_COLORS

    #if defined OVERWORLD
        /**
         * Advanced Sky Color Processing Pipeline
         * 
         * Processes raw sky color data through gamma correction and environmental
         * modifiers to create realistic sky appearance under various conditions.
         */
        
        // Foundation: Gamma-corrected sky color base
        vec3 skyColorSqrt = sqrt(skyColor);
        
        /**
         * Thunderstorm Sky Color Protection
         * Prevents sky color from reaching complete darkness during severe weather
         */
        float invRainStrength2 = (1.0 - rainStrength) * (1.0 - rainStrength);
        vec3 skyColorM = mix(max(skyColorSqrt, vec3(0.63, 0.67, 0.73)), skyColorSqrt, invRainStrength2);
        vec3 skyColorM2 = mix(max(skyColor, sunFactor * vec3(0.265, 0.295, 0.35)), skyColor, invRainStrength2);

        /**
         * Biome-Specific Weather Modifiers
         * Adjusts sky colors based on local environmental conditions
         */
        #ifdef SPECIAL_BIOME_WEATHER
            vec3 nmscSnowM = inSnowy * vec3(-0.3, 0.05, 0.2);    // Snow biome: Cool blue shift
            vec3 nmscDryM = inDry * vec3(-0.3);                  // Desert biome: Warm desaturation
            vec3 ndscSnowM = inSnowy * vec3(-0.25, -0.01, 0.25); // Snow biome lower sky
            vec3 ndscDryM = inDry * vec3(-0.05, -0.09, -0.1);    // Desert biome lower sky
        #else
            vec3 nmscSnowM = vec3(0.0), nmscDryM = vec3(0.0), ndscSnowM = vec3(0.0), ndscDryM = vec3(0.0);
        #endif
        
        /**
         * Advanced Rain Style Color Modifications
         * Style 2: Enhanced precipitation color effects
         */
        #if RAIN_STYLE == 2
            vec3 nmscRainMP = vec3(-0.15, 0.025, 0.1);    // Rain color shift for upper sky
            vec3 ndscRainMP = vec3(-0.125, -0.005, 0.125); // Rain color shift for lower sky
            #ifdef SPECIAL_BIOME_WEATHER
                vec3 nmscRainM = inRainy * nmscRainMP;        // Biome-specific rain upper sky
                vec3 ndscRainM = inRainy * ndscRainMP;        // Biome-specific rain lower sky
            #else
                vec3 nmscRainM = nmscRainMP;                  // Global rain upper sky
                vec3 ndscRainM = ndscRainMP;                  // Global rain lower sky
            #endif
        #else
            vec3 nmscRainM = vec3(0.0), ndscRainM = vec3(0.0);
        #endif
        
        /**
         * Weather Color Modifiers
         * Base weather influence calculations for sky color temperature
         */
        vec3 nmscWeatherM = vec3(-0.1, -0.4, -0.6) + vec3(0.0, 0.06, 0.12) * noonFactor;
        vec3 ndscWeatherM = vec3(-0.15, -0.3, -0.42) + vec3(0.0, 0.02, 0.08) * noonFactor;

        // Time-Based Sky Color Calculations - Spooky Edition
        // Creates dark color progression throughout the day and night cycle
        
        // Noon Sky Colors: Dark and gloomy with greenish tint
        vec3 noonUpSkyColor = pow(skyColorM, vec3(3.2)) * 0.15 * 0.65;
        
        vec3 noonMiddleSkyColor = (skyColorM * (vec3(0.20, 0.22, 0.25) + rainFactor * (nmscWeatherM + nmscRainM + nmscSnowM + nmscDryM))
                                + noonUpSkyColor * 0.15) * 0.65 + vec3(0.0, 0.006, 0.0);
        
        vec3 noonDownSkyColor = (skyColorM * (vec3(0.15, 0.17, 0.20) + rainFactor * (ndscWeatherM + ndscRainM + ndscSnowM + ndscDryM))
                              + noonUpSkyColor * 0.08) * 0.65;

        // Sunset/Sunrise Sky Colors: Dark and muted
        vec3 sunsetUpSkyColor = (skyColorM2 * (vec3(0.18, 0.12, 0.18) + vec3(0.02, 0.04, 0.06) * rainFactor2)) * 0.65 + vec3(0.003, 0.0, 0.0);
        
        vec3 sunsetMiddleSkyColor = (skyColorM2 * (vec3(0.30, 0.25, 0.28) + vec3(0.03, 0.05, 0.0) * rainFactor2)) * 0.65;
        
        vec3 sunsetDownSkyColorP = vec3(0.25, 0.20, 0.15) - vec3(0.12, 0.06, 0.0) * rainFactor;
        vec3 sunsetDownSkyColor = (sunsetDownSkyColorP * 0.25 + 0.12 * sunsetMiddleSkyColor) * 0.65;

        // Day Sky Colors: Oppressive gray atmosphere
        vec3 dayUpSkyColor = mix(noonUpSkyColor, sunsetUpSkyColor, invNoonFactor2);
        vec3 dayMiddleSkyColor = mix(noonMiddleSkyColor, sunsetMiddleSkyColor, invNoonFactor2);
        vec3 dayDownSkyColor = mix(noonDownSkyColor, sunsetDownSkyColor, invNoonFactor2);

        // Night Sky Colors: Nearly black with eerie tint
        vec3 nightColFactor = vec3(0.008, 0.010, 0.020) * (1.0 - 0.6 * rainFactor) + skyColor * 0.10;
        vec3 nightUpSkyColor = pow(nightColFactor, vec3(0.85)) * 0.05 * 0.65 + vec3(0.004, 0.0, 0.006);
        
        vec3 nightMiddleSkyColor = sqrt(nightUpSkyColor) * 0.20 * 0.65 + vec3(0.0, 0.0, 0.0016);
        
        vec3 nightDownSkyColor = nightMiddleSkyColor * vec3(0.4, 0.45, 0.55) * 0.65;
    #endif

#endif //INCLUDE_SKY_COLORS

// Advanced Lighting and Ambient Color System - Spooky Edition
// Enhanced with horror atmosphere

#ifndef INCLUDE_LIGHT_AND_AMBIENT_COLORS
    #define INCLUDE_LIGHT_AND_AMBIENT_COLORS

    #if defined OVERWORLD
        
        // Noon Clear Weather Lighting - Darkened
        #ifndef COMPOSITE
            vec3 noonClearLightColor = vec3(0.5, 0.45, 0.35) * 1.3 * 0.65;
        #else
            vec3 noonClearLightColor = vec3(0.3, 0.5, 1.0) * 0.65;
        #endif
        vec3 noonClearAmbientColor = pow(skyColor, vec3(0.70)) * 0.60 * 0.60;

        // Sunset Clear Weather Lighting
        #ifndef COMPOSITE
            vec3 sunsetClearLightColor = pow(vec3(0.64, 0.45, 0.3), vec3(1.5 + invNoonFactor)) * 5.0;
        #else
            vec3 sunsetClearLightColor = pow(vec3(0.62, 0.39, 0.24), vec3(1.5 + invNoonFactor)) * 6.8;
        #endif
        vec3 sunsetClearAmbientColor = noonClearAmbientColor * vec3(1.21, 0.92, 0.76) * 0.95;

        /**
         * Night Clear Weather Lighting
         * Provides subtle moonlight illumination with brightness adjustment
         */
        #if !defined COMPOSITE && !defined DEFERRED1
            vec3 nightClearLightColor = vec3(0.20, 0.10, 0.15) * (0.9 + vsBrightness * 1.1); // إضاءة ليل أصلية
        #elif defined DEFERRED1
            vec3 nightClearLightColor = vec3(0.16, 0.10, 0.15); // إضاءة ليل أصلية
        #else
            vec3 nightClearLightColor = vec3(0.10, 0.08, 0.20); // إضاءة ليل أصلية
        #endif
        vec3 nightClearAmbientColor = vec3(0.15, 0.15, 0.21) * (1.0 + vsBrightness * 1.0); // إضاءة محيطة أصلية

        /**
         * Biome-Specific Lighting Modifiers
         * Fine-tunes lighting based on local environmental conditions
         */
        #ifdef SPECIAL_BIOME_WEATHER
            vec3 drlcSnowM = inSnowy * vec3(-0.06, 0.0, 0.04);    // Snow biome: Cool light shift
            vec3 drlcDryM = inDry * vec3(0.0, -0.03, -0.05);      // Desert biome: Warm light shift
        #else
            vec3 drlcSnowM = vec3(0.0), drlcDryM = vec3(0.0);
        #endif
        
        /**
         * Advanced Rain Lighting System (Style 2)
         * Provides enhanced precipitation lighting effects
         */
        #if RAIN_STYLE == 2
            vec3 drlcRainMP = vec3(-0.03, 0.0, 0.02);
            #ifdef SPECIAL_BIOME_WEATHER
                vec3 drlcRainM = inRainy * drlcRainMP;             // Biome-specific rain lighting
            #else
                vec3 drlcRainM = drlcRainMP;                       // Global rain lighting
            #endif
        #else
            vec3 drlcRainM = vec3(0.0);
        #endif

        /**
         * Rainy Weather Lighting Conditions
         * Creates overcast, diffused lighting during precipitation
         */
        vec3 dayRainLightColor = vec3(0.21, 0.16, 0.13) * 1.5 + noonFactor * vec3(0.0, 0.02, 0.06)
                               + rainFactor * (drlcRainM + drlcSnowM + drlcDryM); // إضاءة أصلية
        vec3 dayRainAmbientColor = vec3(0.2, 0.2, 0.25) * (2.5 + 1.0 * vsBrightness); // إضاءة محيطة أصلية

        vec3 nightRainLightColor = vec3(0.03, 0.035, 0.05) * (0.9 + 0.9 * vsBrightness); // إضاءة أصلية
        vec3 nightRainAmbientColor = vec3(0.16, 0.20, 0.3) * (1.1 + 0.9 * vsBrightness); // إضاءة محيطة أصلية

        /**
         * Dynamic Color Blending & Interpolation
         * Seamlessly transitions between lighting conditions
         */
        #ifndef COMPOSITE
            float noonFactorDM = noonFactor;                      // Ground and cloud factor
        #else
            float noonFactorDM = noonFactor * noonFactor;         // Light shaft factor (squared for emphasis)
        #endif
        
        // Day-time color interpolation
        vec3 dayLightColor = mix(sunsetClearLightColor, noonClearLightColor, noonFactorDM);
        vec3 dayAmbientColor = mix(sunsetClearAmbientColor, noonClearAmbientColor, noonFactorDM);

        // Clear weather color blending
        vec3 clearLightColor = mix(nightClearLightColor, dayLightColor, sunVisibility2);
        vec3 clearAmbientColor = mix(nightClearAmbientColor, dayAmbientColor, sunVisibility2);

        // Rainy weather color blending
        vec3 rainLightColor = mix(nightRainLightColor, dayRainLightColor, sunVisibility2) * 3.5; // إضاءة أصلية
        vec3 rainAmbientColor = mix(nightRainAmbientColor, dayRainAmbientColor, sunVisibility2);

        // Final lighting color calculation
        vec3 lightColor = mix(clearLightColor, rainLightColor, rainFactor);
        vec3 ambientColor = mix(clearAmbientColor, rainAmbientColor, rainFactor);
        
    #elif defined NETHER
        /**
         * Nether Dimension Lighting - Horror Mode with visible objects
         * Provides hellish lighting with better object visibility
         */
        vec3 lightColor = (netherColor + lavaLightColor) * 0.7; // إضاءة مباشرة قوية للأجسام
        vec3 ambientColor = (netherColor + 0.3 * lavaLightColor) * (0.55 + 0.35 * vsBrightness); // إضاءة محيطة قوية
        
    #elif defined END
        /**
         * End Dimension Lighting - Horror Mode with visible objects
         * Creates void-like atmosphere with better object visibility
         */
        vec3 endLightColor = vec3(0.68, 0.51, 1.07); // أرجواني قوي للأجسام
        float endLightBalancer = 0.2 * vsBrightness;
        
        vec3 lightColor = endLightColor * (0.40 + endLightBalancer); // إضاءة أجسام قوية
        vec3 ambientColor = endLightColor * (0.28 + endLightBalancer); // إضاءة محيطة قوية
    #endif

#endif //INCLUDE_LIGHT_AND_AMBIENT_COLORS

// ═══════════════════════════════════════════════════════════════════════════════════════
//                           ADVANCED CLOUD COLOR RENDERING SYSTEM
//                      (Depends on lightColor & ambientColor calculations)
// ═══════════════════════════════════════════════════════════════════════════════════════

#ifndef INCLUDE_CLOUD_COLORS
    #define INCLUDE_CLOUD_COLORS

    #if defined OVERWORLD
        // Weather-Responsive Cloud Color System
        // Creates dynamic spooky cloud appearance
        
        // Rain Cloud Color: Dark gray clouds
        vec3 cloudRainColor = mix(nightMiddleSkyColor, dayMiddleSkyColor, sunFactor) * vec3(0.5, 0.5, 0.55);

        // Red moon effect factor
        float nightMoonFactor = (1.0 - sunVisibility) * (1.0 - sunFactor);
        vec3 redMoonTint = vec3(1.5, 0.2, 0.2);

        // Ambient Cloud Color - Darker but still visible
        vec3 cloudAmbientColor = mix(
            ambientColor * (sunVisibility2 * (0.3 + 0.08 * noonFactor) + 0.15),
            cloudRainColor * 0.4,
            rainFactor
        ) * vec3(0.6, 0.6, 0.65) * mix(vec3(1.0), redMoonTint, nightMoonFactor * 0.7);

        // Direct Cloud Light Color - Darker but visible
        vec3 cloudLightColor = mix(
            lightColor * (0.35 + 0.15 * noonFactor),
            cloudRainColor * 0.25,
            noonFactor * rainFactor
        ) * vec3(0.55, 0.55, 0.6) * mix(vec3(1.0), redMoonTint * 1.2, nightMoonFactor * 0.8);
    #else
        // Non-Overworld Dimension Cloud Colors
        vec3 cloudRainColor = vec3(0.5, 0.4, 0.4);
        vec3 cloudAmbientColor = ambientColor * 0.8;
        vec3 cloudLightColor = lightColor * 0.9;
    #endif

#endif //INCLUDE_CLOUD_COLORS

/**
 * ═══════════════════════════════════════════════════════════════════════════════════════
 *                            END OF CORE COLOR SYSTEM
 *                         Enhanced Luminara Shader Pack Component
 * ═══════════════════════════════════════════════════════════════════════════════════════
 * 
 * This comprehensive core color system provides:
 * 
 * Sky Color Management:
 * - Realistic day/night progression with smooth transitions
 * - Weather-responsive color adaptation (rain, storms, clear)
 * - Biome-specific environmental modifiers
 * - Advanced gamma correction and color processing
 * 
 * Lighting System:
 * - Time-based color temperature adjustments
 * - Dimensional lighting characteristics (Overworld, Nether, End)
 * - Weather-dependent light quality changes
 * - Performance-optimized calculation paths
 * 
 * Cloud Rendering:
 * - Dynamic cloud color based on atmospheric conditions
 * - Separate ambient and direct lighting for realistic depth
 * - Weather-responsive cloud appearance
 * - Cross-dimensional compatibility
 * 
 * Key Features:
 * - Seamless environmental transitions
 * - Realistic atmospheric simulation
 * - Optimized performance through efficient algorithms
 * - Modular design for easy maintenance and expansion
 * - Comprehensive weather and time-of-day support
 * 
 * Maintained by VcorA - Luminara Shader Development Team
 * ═══════════════════════════════════════════════════════════════════════════════════════
 */
