#version 120

uniform vec3 sunPosition;
uniform vec3 moonPosition;
uniform int worldTime;
uniform float rainStrength;
uniform float frameTimeCounter;

varying vec3 viewDir;
varying vec4 glcolor;

const float PI = 3.14159265359;

float hash(vec2 p) {
    return fract(sin(dot(p, vec2(12.9898, 78.233))) * 43758.5453);
}

float noise(vec2 p) {
    vec2 i = floor(p);
    vec2 f = fract(p);
    
    float a = hash(i);
    float b = hash(i + vec2(1.0, 0.0));
    float c = hash(i + vec2(0.0, 1.0));
    float d = hash(i + vec2(1.0, 1.0));
    
    vec2 u = f * f * (3.0 - 2.0 * f);
    
    return mix(a, b, u.x) + (c - a) * u.y * (1.0 - u.x) + (d - b) * u.x * u.y;
}

float fbm(vec2 p) {
    float value = 0.0;
    float amplitude = 0.5;
    float frequency = 1.0;
    
    for (int i = 0; i < 4; i++) {
        value += amplitude * noise(p * frequency);
        amplitude *= 0.5;
        frequency *= 2.0;
    }
    
    return value;
}

// Smooth time-based factor that wraps properly from 23999 to 0
float getTimePhase(float targetTime, float width) {
    float normalizedTime = float(worldTime) / 24000.0;
    float normalizedTarget = targetTime / 24000.0;
    
    // Calculate shortest distance on the circle
    float diff = abs(normalizedTime - normalizedTarget);
    if (diff > 0.5) diff = 1.0 - diff;
    
    // Smooth transition
    return smoothstep(width, 0.0, diff);
}

// Get smooth day/night cycle value
float getDayNightCycle() {
    float normalizedTime = float(worldTime) / 24000.0;
    // Peak day at 6000 (0.25), peak night at 18000 (0.75)
    float cycle = cos((normalizedTime - 0.25) * 2.0 * PI);
    return (cycle + 1.0) * 0.5; // normalize to 0-1
}

vec3 getSmoothDayNightColor() {
    vec3 nightColor   = vec3(0.05, 0.08, 0.16);
    vec3 dayColor     = vec3(0.95, 0.92, 0.86);
    vec3 sunriseColor = vec3(1.00, 0.52, 0.22);
    vec3 twilightGlow = vec3(0.45, 0.55, 0.85);

    // Base day/night cycle
    float dayIntensity = getDayNightCycle();
    dayIntensity = pow(dayIntensity, 1.15);

    // Pre-dawn boost at worldTime = 0
    float preDawnBoost = getTimePhase(0.0, 0.05) * 0.45;
    dayIntensity = clamp(dayIntensity + preDawnBoost * (1.0 - dayIntensity), 0.0, 1.0);

    // Sunrise warm at ~23000
    float dawnPeak = getTimePhase(23000.0, 0.08) * 0.35;

    // Base color with effects
    vec3 baseColor = mix(nightColor, dayColor, dayIntensity);
    vec3 warm = sunriseColor * dawnPeak;
    vec3 glow = twilightGlow * (dawnPeak * 0.15 + 0.15);

    vec3 color = baseColor + warm + glow;
    color = pow(clamp(color, 0.0, 1.0), vec3(0.45));
    return clamp(color, 0.0, 1.0);
}

vec3 renderClouds(vec3 skyColor, vec3 dir) {
    if (dir.y < 0.01) return skyColor;
    
    float cloudHeight = smoothstep(0.05, 0.4, dir.y) * smoothstep(0.7, 0.5, dir.y);
    
    float time = frameTimeCounter * 0.015;
    vec2 cloudUV = (dir.xz / max(dir.y, 0.01)) * 0.6;
    
    vec2 cloudPos = cloudUV * 1.2 + vec2(time * 0.2, time * 0.1);
    float clouds = fbm(cloudPos);
    clouds = smoothstep(0.5, 0.85, clouds);
    clouds *= cloudHeight;
    
    if (clouds < 0.02) return skyColor;
    
    // Smooth cloud coloring based on time phases
    float dayFactor = getTimePhase(12000.0, 0.25);      // Day (8000-16000)
    float sunsetFactor = getTimePhase(20000.0, 0.125);   // Sunset (18000-22000) 
    float sunriseFactor = getTimePhase(6000.0, 0.125);   // Sunrise (4000-8000)
    float nightFactor = 1.0 - max(max(dayFactor, sunsetFactor), sunriseFactor);
    
    // Cloud colors for different times
    vec3 dayCloudColor = mix(vec3(0.55, 0.55, 0.6), vec3(0.85, 0.85, 0.85), 1.0 - pow(clouds, 1.3));
    vec3 sunsetCloudColor = mix(vec3(0.3, 0.25, 0.3), vec3(0.7, 0.5, 0.4), 1.0 - pow(clouds, 1.0));
    vec3 sunriseCloudColor = mix(vec3(0.4, 0.35, 0.4), vec3(0.8, 0.7, 0.6), 1.0 - pow(clouds, 1.2));
    vec3 nightCloudColor = mix(vec3(0.1, 0.12, 0.15), vec3(0.2, 0.22, 0.25), 1.0 - pow(clouds, 0.9));
    
    // Blend cloud colors smoothly
    vec3 cloudColor = dayCloudColor * dayFactor + 
                     sunsetCloudColor * sunsetFactor + 
                     sunriseCloudColor * sunriseFactor + 
                     nightCloudColor * nightFactor;
    
    skyColor *= (1.0 - clouds * 0.15);
    return mix(skyColor, cloudColor, clouds);
}

vec3 calculateSkyGradient(vec3 dir) {
    vec3 baseColor = getSmoothDayNightColor();
    float altitude = max(dir.y, 0.0);
    
    // Smooth gradient transitions based on time
    float morningFactor = getTimePhase(6000.0, 0.25);    // 0-12000 range
    float eveningFactor = getTimePhase(18000.0, 0.25);   // 12000-24000 range
    float nightFactor = 1.0 - max(morningFactor, eveningFactor);
    
    // Different gradient styles for different times
    vec3 morningGradient = mix(baseColor * vec3(1.1, 1.05, 1.0), 
                              baseColor * vec3(0.4, 0.6, 0.9), 
                              pow(altitude, 0.7));
    
    float sunsetBlend = 1.0 - smoothstep(17000.0, 19000.0, float(worldTime));
    vec3 eveningGradient = mix(baseColor * mix(vec3(0.7, 0.5, 0.4), vec3(1.0, 0.7, 0.5), sunsetBlend),
                              baseColor * mix(vec3(0.3, 0.4, 0.6), vec3(0.5, 0.6, 0.8), sunsetBlend),
                              pow(altitude, 0.8));
    
    vec3 nightGradient = mix(baseColor * vec3(0.9, 1.0, 1.1),
                            baseColor * vec3(0.7, 0.8, 1.0),
                            pow(altitude, 0.9));
    
    // Blend gradients smoothly
    return morningGradient * morningFactor + 
           eveningGradient * eveningFactor + 
           nightGradient * nightFactor;
}

void main() {
    vec3 dir = normalize(viewDir);
    
    vec3 skyColor = calculateSkyGradient(dir);
    skyColor = renderClouds(skyColor, dir);
    
    if (rainStrength > 0.0) {
        skyColor *= mix(1.0, 0.7, rainStrength);
        skyColor = mix(skyColor, vec3(0.4, 0.4, 0.45), rainStrength * 0.2);
    }
    
    gl_FragColor = vec4(skyColor, 1.0);
}