#version 120

uniform sampler2D texture;
uniform sampler2D lightmap;
uniform sampler2D normals;
uniform sampler2D specular;

uniform vec3 sunPosition;
uniform vec3 moonPosition;
uniform vec3 cameraPosition;
uniform float frameTimeCounter;
uniform int worldTime;
uniform float rainStrength;
uniform float wetness;
uniform int isEyeInWater;

uniform ivec2 eyeBrightnessSmooth;
uniform float near;
uniform float far;

varying vec2 texcoord;
varying vec2 lmcoord;
varying vec4 glcolor;
varying vec3 normal;
varying vec3 worldPos;
varying vec3 viewPos;
varying float blockId;
varying float waveEffect;

const float PI = 3.14159265359;

vec3 saturation(vec3 color, float sat) {
    float gray = dot(color, vec3(0.299, 0.587, 0.114));
    return mix(vec3(gray), color, sat);
}

vec3 contrast(vec3 color, float cont) {
    return (color - 0.5) * cont + 0.5;
}

// Smooth Day/Night Transition Function
vec3 getSmoothDayNightColor() {
    float dayTime = float(worldTime) / 24000.0;
    
    // Define colors for different times
    vec3 morningColor = vec3(0.8, 0.7, 0.8);
    vec3 dayColor = vec3(0.9, 0.8, 0.9);
    vec3 nightColor = vec3(0.6, 0.75, 0.7);
    
    if (worldTime <= 6000) {
        float t = smoothstep(1000.0, 6000.0, float(worldTime));
        return mix(morningColor, dayColor, t);
    } else if (worldTime <= 13000) {
        float t = smoothstep(6000.0, 13000.0, float(worldTime));
        return mix(dayColor, nightColor, t);
    } else if (worldTime <= 24000) {
        float t = smoothstep(13000.0, 16000.0, float(worldTime));
        return mix(nightColor, morningColor, t);
    }
}

// Dynamic Wind Effect
vec3 dynamicWind(vec3 color, vec3 worldPos, float waveEffect) {
    if (waveEffect < 0.1) return color;
    
    float windTime = frameTimeCounter * 0.8;
    float windX = sin(windTime + worldPos.x * 0.1) * 0.3;
    float windZ = cos(windTime * 0.7 + worldPos.z * 0.15) * 0.25;
    float windCombined = (windX + windZ) * waveEffect;
    
    vec3 windTint = vec3(0.05, 0.1, 0.02) * windCombined;
    return color + windTint;
}

// Depth-Based Color Grading
vec3 depthColorGrading(vec3 color, vec3 viewPos, float skyLight) {
    float distance = length(viewPos);
    float depthFactor = clamp(distance / 16.0, 0.0, 1.0);
    
    vec3 distantColor = mix(color, vec3(0.6, 0.7, 0.9), depthFactor * 0.1);
    
    if (skyLight < 0.3) {
        vec3 warmTint = vec3(1.1, 0.95, 0.8);
        distantColor *= mix(vec3(1.0), warmTint, depthFactor * 0.2);
    }
    
    return distantColor;
}

// FIXED: Enhanced PBR with Proper Direction Calculations
vec3 dynamicPBR(vec3 color, vec3 normal, vec3 worldPos, vec3 viewDir, float skyLight, float blockLight) {
    vec3 normalMap = texture2D(normals, texcoord).rgb;
    vec3 enhancedNormal = normalize(normal);
    
    if (length(normalMap - 0.5) > 0.01) {
        normalMap = normalMap * 2.0 - 1.0;
        enhancedNormal = normalize(normal + normalMap * 0.5);
    }
    
    float brightness = dot(color, vec3(0.299, 0.587, 0.114));
    float metallic = brightness > 0.5 ? 0.7 : 0.0;
    
    vec3 specColor = mix(vec3(0.04), color, metallic);
    vec3 totalSpecular = vec3(0.0);
    
    // FIXED: Proper Sun/Moon Direction Calculation
    if (skyLight > 0.3) {
        vec3 lightDirection;
        
        // Use provided sun/moon positions if available, otherwise calculate
        if (worldTime >= 6000 && worldTime <= 18000) {
            // Daytime - use sun
            if (length(sunPosition) > 0.1) {
                lightDirection = normalize(sunPosition);
            } else {
                // Fallback calculation
                float dayProgress = (float(worldTime) - 6000.0) / 12000.0; // 0 to 1
                float sunAngle = dayProgress * PI; // 0 to PI
                lightDirection = normalize(vec3(
                    cos(sunAngle) * 0.8,  // East-West movement
                    sin(sunAngle),        // Height
                    -0.3                  // Slight south bias
                ));
            }
        } else {
            // Nighttime - use moon
            if (length(moonPosition) > 0.1) {
                lightDirection = normalize(moonPosition);
            } else {
                // Fallback calculation
                float nightProgress = worldTime > 18000 ? 
                    (float(worldTime) - 18000.0) / 6000.0 :  // 18000-24000
                    (float(worldTime) + 6000.0) / 6000.0;    // 0-6000
                float moonAngle = nightProgress * PI;
                lightDirection = normalize(vec3(
                    cos(moonAngle) * 0.6,
                    sin(moonAngle) * 0.5 + 0.2, // Lower than sun
                    0.4
                ));
            }
        }
        
        // FIXED: Proper reflection calculation
        float NdotL = max(dot(enhancedNormal, lightDirection), 0.0);
        
        // Calculate reflection properly in world space
        vec3 reflectionDir = reflect(-lightDirection, enhancedNormal);
        float VdotR = max(dot(viewDir, reflectionDir), 0.0);
        float specPower = worldTime >= 6000 && worldTime <= 18000 ? 64.0 : 32.0;
        float specular = pow(VdotR, specPower);
        
        vec3 lightColor = getSmoothDayNightColor();
        float lightIntensity = worldTime >= 6000 && worldTime <= 18000 ? 1.0 : 0.4;
        
        totalSpecular += specColor * specular * skyLight * lightColor * NdotL * lightIntensity;
    }
    
    // FIXED: Torch lighting with proper omni-directional calculation
    if (blockLight > 0.2) {
        vec3 torchColor = vec3(0.8, 0.4, 0.2);
        
        // For torch light, we assume it's coming from above the surface
        // but we'll make it more omnidirectional by using the normal more directly
        vec3 torchLightDir = normalize(vec3(0.2, 0.8, 0.2)); // Slightly angled from above
        
        float torchNdotL = max(dot(enhancedNormal, torchLightDir), 0.2); // Min 0.2 for ambient
        
        // For torch specular, use a wider, softer reflection
        vec3 torchReflectDir = reflect(-torchLightDir, enhancedNormal);
        float torchVdotR = max(dot(viewDir, torchReflectDir), 0.0);
        float torchSpecular = pow(torchVdotR, 16.0); // Lower specular power for softer highlights
        
        // Add some rim lighting effect for torches
        float rimFactor = 1.0 - max(dot(viewDir, enhancedNormal), 0.0);
        float rimLight = pow(rimFactor, 3.0) * 0.3;
        
        float torchIntensity = (blockLight - 0.2) * 1.25;
        totalSpecular += (specColor * torchSpecular + rimLight) * torchIntensity * torchColor * torchNdotL;
    }
    
    return color + totalSpecular;
}

void main() {
    vec2 coord = texcoord;
    vec4 color = texture2D(texture, coord) * glcolor;
    vec3 lightmapColor = texture2D(lightmap, lmcoord).rgb;
    
    if (color.a < 0.1) {
        discard;
    }
    
    float skyLight = lmcoord.y;
    float blockLight = lmcoord.x;
    
    // Day/night tint
    vec3 dayNightTint = vec3(1.0);
    if (skyLight > 0.2) {
        dayNightTint = getSmoothDayNightColor();
    }
    
    // FIXED: Proper view direction calculation in world space
    vec3 viewDir = normalize(cameraPosition - worldPos);
    float lightStrength = max(max(skyLight, blockLight), 0.1);
    
    // Apply effects
    color.rgb = depthColorGrading(color.rgb, viewPos, skyLight);
    
    // PBR only for well-lit surfaces
    if (lightStrength > 0.2) {
        vec3 pbrResult = dynamicPBR(color.rgb, normal, worldPos, viewDir, skyLight, blockLight);
        float pbrStrength = smoothstep(0.3, 0.8, lightStrength);
        color.rgb = mix(color.rgb, pbrResult, pbrStrength * 2.0);
    }
    
    // Apply day/night tint
    color.rgb *= dayNightTint;
    
    // Enhanced lightmap
    vec3 enhancedLightmap = pow(lightmapColor, vec3(0.9)) * 1.1;
    color.rgb *= enhancedLightmap;
    
    // Torch warmth
    float torchLight = max(blockLight - 0.4, 0.0) * 2.5;
    color.rgb += vec3(1.0, 0.6, 0.3) * torchLight * 0.2;
    
    // Rain effects
    if (rainStrength > 0.4) {
        color.rgb = saturation(color.rgb, 1.0 - rainStrength * 0.15);
    }
    
    // Underwater effects
    if (isEyeInWater == 1) {
        color.rgb *= vec3(0.6, 0.75, 1.8);
        float distance = length(viewPos);
        float depthFade = exp(-distance * 0.03);
        color.rgb = mix(vec3(0.2, 0.5, 1.3), color.rgb, depthFade);
    }
    
    // Final adjustments
    color.rgb = contrast(color.rgb, 1.35);
    color.rgb = saturation(color.rgb, 1.3);
    
    gl_FragColor = color;
}