#version 120

uniform vec3 sunPosition;
uniform vec3 moonPosition;
uniform int worldTime;
uniform float rainStrength;
uniform float frameTimeCounter;

varying vec3 viewDir;
varying vec4 glcolor;

const float PI = 3.14159265359;

float hash(vec2 p) {
    return fract(sin(dot(p, vec2(12.9898, 78.233))) * 43758.5453);
}

float noise(vec2 p) {
    vec2 i = floor(p);
    vec2 f = fract(p);
    
    float a = hash(i);
    float b = hash(i + vec2(1.0, 0.0));
    float c = hash(i + vec2(0.0, 1.0));
    float d = hash(i + vec2(1.0, 1.0));
    
    vec2 u = f * f * (3.0 - 2.0 * f);
    
    return mix(a, b, u.x) + (c - a) * u.y * (1.0 - u.x) + (d - b) * u.x * u.y;
}

float fbm(vec2 p) {
    float value = 0.0;
    float amplitude = 0.5;
    float frequency = 1.0;
    
    for (int i = 0; i < 5; i++) {
        value += amplitude * noise(p * frequency);
        amplitude *= 0.5;
        frequency *= 2.0;
    }
    
    return value;
}

float getTimePhase(float targetTime, float width) {
    float normalizedTime = float(worldTime) / 24000.0;
    float normalizedTarget = targetTime / 24000.0;
    
    float diff = abs(normalizedTime - normalizedTarget);
    if (diff > 0.5) diff = 1.0 - diff;
    
    return smoothstep(width, 0.0, diff);
}

float getDayNightCycle() {
    float normalizedTime = float(worldTime) / 24000.0;
    float cycle = cos((normalizedTime - 0.25) * 2.0 * PI);
    return (cycle + 1.0) * 0.5;
}

vec3 getSmoothDayNightColor() {
    vec3 nightColor   = vec3(0.04, 0.07, 0.15);
    vec3 dayColor     = vec3(0.96, 0.94, 0.88);
    vec3 sunriseColor = vec3(1.00, 0.55, 0.25);
    vec3 sunsetColor  = vec3(0.95, 0.45, 0.30);
    vec3 twilightGlow = vec3(0.50, 0.58, 0.88);

    float dayIntensity = getDayNightCycle();
    dayIntensity = pow(dayIntensity, 1.2);

    float preDawnBoost = getTimePhase(0.0, 0.05) * 0.5;
    dayIntensity = clamp(dayIntensity + preDawnBoost * (1.0 - dayIntensity), 0.0, 1.0);

    float dawnPeak = getTimePhase(23000.0, 0.08) * 0.4;
    float duskPeak = getTimePhase(13000.0, 0.08) * 0.35;

    vec3 baseColor = mix(nightColor, dayColor, dayIntensity);
    vec3 warmSunrise = sunriseColor * dawnPeak;
    vec3 warmSunset = sunsetColor * duskPeak;
    vec3 glow = twilightGlow * ((dawnPeak + duskPeak) * 0.2);

    vec3 color = baseColor + warmSunrise + warmSunset + glow;
    color = pow(clamp(color, 0.0, 1.0), vec3(0.48));
    return clamp(color, 0.0, 1.0);
}

vec3 renderClouds(vec3 skyColor, vec3 dir) {
    if (dir.y < 0.02) return skyColor;
    
    float cloudHeight = smoothstep(0.05, 0.35, dir.y) * smoothstep(0.75, 0.5, dir.y);
    
    float time = frameTimeCounter * 0.012;
    vec2 cloudUV = (dir.xz / max(dir.y, 0.01)) * 0.65;
    
    vec2 cloudPos = cloudUV * 1.15 + vec2(time * 0.18, time * 0.09);
    float clouds = fbm(cloudPos);
    clouds = smoothstep(0.48, 0.82, clouds);
    clouds *= cloudHeight;
    
    if (clouds < 0.015) return skyColor;
    
    float dayFactor = getTimePhase(12000.0, 0.28);
    float sunsetFactor = getTimePhase(13000.0, 0.10);
    float sunriseFactor = getTimePhase(23000.0, 0.10);
    float nightFactor = 1.0 - max(max(dayFactor, sunsetFactor), sunriseFactor);
    
    vec3 dayCloudColor = mix(vec3(0.58, 0.58, 0.62), vec3(0.90, 0.90, 0.88), 1.0 - pow(clouds, 1.4));
    vec3 sunsetCloudColor = mix(vec3(0.35, 0.28, 0.32), vec3(0.85, 0.55, 0.45), 1.0 - pow(clouds, 1.1));
    vec3 sunriseCloudColor = mix(vec3(0.42, 0.38, 0.42), vec3(0.88, 0.72, 0.62), 1.0 - pow(clouds, 1.25));
    vec3 nightCloudColor = mix(vec3(0.08, 0.10, 0.14), vec3(0.18, 0.20, 0.24), 1.0 - pow(clouds, 0.95));
    
    vec3 cloudColor = dayCloudColor * dayFactor + 
                     sunsetCloudColor * sunsetFactor + 
                     sunriseCloudColor * sunriseFactor + 
                     nightCloudColor * nightFactor;
    
    skyColor *= (1.0 - clouds * 0.18);
    return mix(skyColor, cloudColor, clouds * 0.95);
}

vec3 calculateSkyGradient(vec3 dir) {
    vec3 baseColor = getSmoothDayNightColor();
    float altitude = max(dir.y, 0.0);
    
    float morningFactor = getTimePhase(6000.0, 0.28);
    float eveningFactor = getTimePhase(18000.0, 0.28);
    float nightFactor = 1.0 - max(morningFactor, eveningFactor);
    
    vec3 morningGradient = mix(baseColor * vec3(1.12, 1.06, 1.02), 
                              baseColor * vec3(0.45, 0.62, 0.92), 
                              pow(altitude, 0.72));
    
    float sunsetBlend = 1.0 - smoothstep(12500.0, 14000.0, float(worldTime));
    vec3 eveningGradient = mix(baseColor * mix(vec3(0.75, 0.52, 0.42), vec3(1.05, 0.72, 0.52), sunsetBlend),
                              baseColor * mix(vec3(0.35, 0.42, 0.62), vec3(0.52, 0.62, 0.82), sunsetBlend),
                              pow(altitude, 0.78));
    
    vec3 nightGradient = mix(baseColor * vec3(0.88, 0.98, 1.12),
                            baseColor * vec3(0.68, 0.78, 1.02),
                            pow(altitude, 0.88));
    
    return morningGradient * morningFactor + 
           eveningGradient * eveningFactor + 
           nightGradient * nightFactor;
}

void main() {
    vec3 dir = normalize(viewDir);
    
    vec3 skyColor = calculateSkyGradient(dir);
    skyColor = renderClouds(skyColor, dir);
    
    if (rainStrength > 0.0) {
        skyColor *= mix(1.0, 0.68, rainStrength);
        skyColor = mix(skyColor, vec3(0.38, 0.38, 0.44), rainStrength * 0.22);
    }
    
    gl_FragColor = vec4(skyColor, 1.0);
}