#version 120

uniform sampler2D texture;
uniform sampler2D lightmap;
uniform sampler2D normals;
uniform sampler2D specular;

uniform vec3 sunPosition;
uniform vec3 moonPosition;
uniform vec3 cameraPosition;
uniform float frameTimeCounter;
uniform int worldTime;
uniform float rainStrength;
uniform float wetness;
uniform int isEyeInWater;

uniform ivec2 eyeBrightnessSmooth;
uniform float near;
uniform float far;

varying vec2 texcoord;
varying vec2 lmcoord;
varying vec4 glcolor;
varying vec3 normal;
varying vec3 worldPos;
varying vec3 viewPos;
varying float blockId;

const float PI = 3.14159265359;

vec3 saturation(vec3 color, float sat) {
    float gray = dot(color, vec3(0.299, 0.587, 0.114));
    return mix(vec3(gray), color, sat);
}

vec3 contrast(vec3 color, float cont) {
    return (color - 0.5) * cont + 0.5;
}

vec3 getSmoothDayNightColor() {
    float dayTime = float(worldTime) / 24000.0;
    
    vec3 morningColor = vec3(0.82, 0.72, 0.82);
    vec3 dayColor = vec3(0.92, 0.82, 0.92);
    vec3 eveningColor = vec3(0.78, 0.58, 0.48);
    vec3 nightColor = vec3(0.58, 0.72, 0.68);
    
    if (worldTime <= 6000) {
        float t = smoothstep(500.0, 6000.0, float(worldTime));
        return mix(morningColor, dayColor, t);
    } else if (worldTime <= 13000) {
        return dayColor;
    } else if (worldTime <= 18000) {
        float t = smoothstep(13000.0, 18000.0, float(worldTime));
        return mix(dayColor, eveningColor, t);
    } else {
        float t = smoothstep(18000.0, 23000.0, float(worldTime));
        return mix(eveningColor, nightColor, t);
    }
}

vec3 depthColorGrading(vec3 color, vec3 viewPos, float skyLight) {
    float distance = length(viewPos);
    float depthFactor = clamp(distance / 18.0, 0.0, 1.0);
    
    vec3 distantColor = mix(color, vec3(0.62, 0.72, 0.92), depthFactor * 0.12);
    
    if (skyLight < 0.35) {
        vec3 warmTint = vec3(1.08, 0.96, 0.82);
        distantColor *= mix(vec3(1.0), warmTint, depthFactor * 0.18);
    }
    
    return distantColor;
}

vec3 dynamicPBR(vec3 color, vec3 normal, vec3 worldPos, vec3 viewDir, float skyLight, float blockLight) {
    vec3 normalMap = texture2D(normals, texcoord).rgb;
    vec3 enhancedNormal = normalize(normal);
    
    if (length(normalMap - 0.5) > 0.01) {
        normalMap = normalMap * 2.0 - 1.0;
        enhancedNormal = normalize(normal + normalMap * 0.45);
    }
    
    float brightness = dot(color, vec3(0.299, 0.587, 0.114));
    float metallic = brightness > 0.52 ? 0.72 : 0.0;
    
    vec3 specColor = mix(vec3(0.04), color, metallic);
    vec3 totalSpecular = vec3(0.0);
    
    if (skyLight > 0.32) {
        vec3 lightDirection;
        
        if (worldTime >= 6000 && worldTime <= 18000) {
            if (length(sunPosition) > 0.1) {
                lightDirection = normalize(sunPosition);
            } else {
                float dayProgress = (float(worldTime) - 6000.0) / 12000.0;
                float sunAngle = dayProgress * PI;
                lightDirection = normalize(vec3(
                    cos(sunAngle) * 0.82,
                    sin(sunAngle),
                    -0.28
                ));
            }
        } else {
            if (length(moonPosition) > 0.1) {
                lightDirection = normalize(moonPosition);
            } else {
                float nightProgress = worldTime > 18000 ? 
                    (float(worldTime) - 18000.0) / 6000.0 :
                    (float(worldTime) + 6000.0) / 6000.0;
                float moonAngle = nightProgress * PI;
                lightDirection = normalize(vec3(
                    cos(moonAngle) * 0.62,
                    sin(moonAngle) * 0.52 + 0.22,
                    0.38
                ));
            }
        }
        
        float NdotL = max(dot(enhancedNormal, lightDirection), 0.0);
        
        vec3 reflectionDir = reflect(-lightDirection, enhancedNormal);
        float VdotR = max(dot(viewDir, reflectionDir), 0.0);
        float specPower = worldTime >= 6000 && worldTime <= 18000 ? 72.0 : 36.0;
        float specular = pow(VdotR, specPower);
        
        vec3 lightColor = getSmoothDayNightColor();
        float lightIntensity = worldTime >= 6000 && worldTime <= 18000 ? 1.05 : 0.42;
        
        totalSpecular += specColor * specular * skyLight * lightColor * NdotL * lightIntensity;
    }
    
    if (blockLight > 0.22) {
        vec3 torchColor = vec3(0.82, 0.42, 0.22);
        vec3 torchLightDir = normalize(vec3(0.18, 0.82, 0.18));
        
        float torchNdotL = max(dot(enhancedNormal, torchLightDir), 0.22);
        
        vec3 torchReflectDir = reflect(-torchLightDir, enhancedNormal);
        float torchVdotR = max(dot(viewDir, torchReflectDir), 0.0);
        float torchSpecular = pow(torchVdotR, 18.0);
        
        float rimFactor = 1.0 - max(dot(viewDir, enhancedNormal), 0.0);
        float rimLight = pow(rimFactor, 3.2) * 0.28;
        
        float torchIntensity = (blockLight - 0.22) * 1.3;
        totalSpecular += (specColor * torchSpecular + rimLight) * torchIntensity * torchColor * torchNdotL;
    }
    
    return color + totalSpecular;
}

void main() {
    vec2 coord = texcoord;
    vec4 color = texture2D(texture, coord) * glcolor;
    vec3 lightmapColor = texture2D(lightmap, lmcoord).rgb;
    
    if (color.a < 0.1) {
        discard;
    }
    
    float skyLight = lmcoord.y;
    float blockLight = lmcoord.x;
    
    vec3 dayNightTint = vec3(1.0);
    if (skyLight > 0.22) {
        dayNightTint = getSmoothDayNightColor();
    }
    
    vec3 viewDir = normalize(cameraPosition - worldPos);
    float lightStrength = max(max(skyLight, blockLight), 0.12);
    
    color.rgb = depthColorGrading(color.rgb, viewPos, skyLight);
    
    if (lightStrength > 0.25) {
        vec3 pbrResult = dynamicPBR(color.rgb, normal, worldPos, viewDir, skyLight, blockLight);
        float pbrStrength = smoothstep(0.32, 0.82, lightStrength);
        color.rgb = mix(color.rgb, pbrResult, pbrStrength * 2.1);
    }
    
    color.rgb *= dayNightTint;
    
    vec3 enhancedLightmap = pow(lightmapColor, vec3(0.88)) * 1.12;
    color.rgb *= enhancedLightmap;
    
    float torchLight = max(blockLight - 0.38, 0.0) * 2.6;
    color.rgb += vec3(1.0, 0.58, 0.32) * torchLight * 0.22;
    
    if (rainStrength > 0.4) {
        color.rgb = saturation(color.rgb, 1.0 - rainStrength * 0.16);
    }
    
    if (isEyeInWater == 1) {
        color.rgb *= vec3(0.58, 0.72, 1.82);
        float distance = length(viewPos);
        float depthFade = exp(-distance * 0.032);
        color.rgb = mix(vec3(0.18, 0.48, 1.32), color.rgb, depthFade);
    }
    
    color.rgb = contrast(color.rgb, 1.2);
    color.rgb = saturation(color.rgb, 1.32);
    
    gl_FragColor = color;
}