#version 120

attribute vec4 mc_Entity;
attribute vec4 mc_midTexCoord;
attribute vec4 at_tangent;

uniform mat4 gbufferModelView;
uniform mat4 gbufferProjection;
uniform mat4 gbufferModelViewInverse;
uniform vec3 cameraPosition;
uniform float frameTimeCounter;
uniform int worldTime;

varying vec2 texcoord;
varying vec2 lmcoord;
varying vec4 glcolor;
varying vec3 normal;
varying vec3 worldPos;
varying vec3 viewPos;
varying float blockId;
varying float waveEffect;

const float PI = 3.14159265359;

// ============================================================================
// BLOCK ID DEFINITIONS
// ============================================================================

const float TALLGRASS = 31.0;
const float LEAVES = 18.0;
const float LEAVES2 = 161.0;
const float WHEAT = 59.0;
const float FLOWERS = 38.0;
const float ROSE = 37.0;
const float LILYPAD = 111.0;
const float VINES = 106.0;
const float SAPLINGS = 6.0;
const float DEADBUSH = 32.0;
const float SUGARCANE = 83.0;
const float DANDELION = 37.0;

bool shouldWave(float id) {
    return id == TALLGRASS || id == LEAVES || id == LEAVES2 || 
           id == WHEAT || id == FLOWERS || id == ROSE ||
           id == LILYPAD || id == VINES || id == SAPLINGS ||
           id == DEADBUSH || id == SUGARCANE || id == DANDELION;
}

// ============================================================================
// WIND ANIMATION
// ============================================================================

vec3 calculateWindAnimation(vec3 position, vec3 worldPosition, float blockID) {
    if (!shouldWave(blockID)) {
        return position;
    }
    
    float time = frameTimeCounter;
    
    // Different wind patterns for different block types
    float windSpeed = 1.0;
    float windStrength = 0.0;
    
    if (blockID == LEAVES || blockID == LEAVES2) {
        // Leaves - gentle swaying
        windSpeed = 0.8;
        windStrength = 0.015;
    } else if (blockID == TALLGRASS || blockID == WHEAT || blockID == FLOWERS || 
               blockID == ROSE || blockID == DANDELION) {
        // Grass and flowers - more pronounced movement
        windSpeed = 1.2;
        windStrength = 0.035;
    } else if (blockID == VINES) {
        // Vines - hanging movement
        windSpeed = 0.6;
        windStrength = 0.025;
    } else if (blockID == LILYPAD) {
        // Lilypads - subtle floating
        windSpeed = 0.5;
        windStrength = 0.008;
    } else {
        // Default
        windSpeed = 1.0;
        windStrength = 0.02;
    }
    
    // Multi-frequency wind waves
    float windX = sin(time * windSpeed + worldPosition.x * 0.3 + worldPosition.z * 0.15) * 
                  cos(time * windSpeed * 0.7 + worldPosition.z * 0.2);
    float windZ = cos(time * windSpeed * 0.9 + worldPosition.z * 0.3 + worldPosition.x * 0.12) * 
                  sin(time * windSpeed * 0.6 + worldPosition.x * 0.18);
    
    // Add some vertical bobbing
    float windY = sin(time * windSpeed * 1.5 + worldPosition.x * 0.25 + worldPosition.z * 0.25) * 0.3;
    
    // Vertex height influence (top vertices move more)
    float heightFactor = position.y - floor(position.y);
    heightFactor = pow(heightFactor, 1.5);
    
    // Apply wind displacement
    vec3 windOffset = vec3(windX, windY, windZ) * windStrength * heightFactor;
    
    return position + windOffset;
}

// ============================================================================
// MAIN VERTEX SHADER
// ============================================================================

void main() {
    // Get block ID
    blockId = mc_Entity.x;
    
    // Calculate wave effect for fragment shader
    waveEffect = shouldWave(blockId) ? 1.0 : 0.0;
    
    // Pass through texture coordinates
    texcoord = (gl_TextureMatrix[0] * gl_MultiTexCoord0).xy;
    lmcoord = (gl_TextureMatrix[1] * gl_MultiTexCoord1).xy;
    glcolor = gl_Color;
    
    // Calculate normals
    normal = normalize(gl_NormalMatrix * gl_Normal);
    
    // Get position
    vec4 position = gl_Vertex;
    
    // Calculate world position before animation
    vec4 viewPosition = gbufferModelView * position;
    worldPos = (gbufferModelViewInverse * viewPosition).xyz + cameraPosition;
    
    // Apply wind animation
    if (waveEffect > 0.5) {
        position.xyz = calculateWindAnimation(position.xyz, worldPos, blockId);
    }
    
    // Recalculate view position after animation
    viewPos = (gbufferModelView * position).xyz;
    
    // Final position
    gl_Position = gl_ProjectionMatrix * vec4(viewPos, 1.0);
}