package me.juancarloscp52.bedrockify.common.block;

import com.mojang.serialization.MapCodec;
import me.juancarloscp52.bedrockify.Bedrockify;
import me.juancarloscp52.bedrockify.common.block.cauldron.BedrockCauldronBehavior;
import me.juancarloscp52.bedrockify.common.features.cauldron.BedrockCauldronBlocks;
import me.juancarloscp52.bedrockify.common.features.cauldron.BedrockCauldronProperties;
import net.minecraft.block.*;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2275;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2398;
import net.minecraft.class_2680;
import net.minecraft.class_2689;
import net.minecraft.class_2758;
import net.minecraft.class_3532;
import net.minecraft.class_5556;
import net.minecraft.class_5712;
import net.minecraft.class_5819;
import net.minecraft.class_9381;

/**
 * Allows to keep the potion fluid.
 */
public class PotionCauldronBlock extends AbstractBECauldronBlock {

    public static final MapCodec<PotionCauldronBlock> field_46280 = PotionCauldronBlock.method_54094(PotionCauldronBlock::new);
    public static final class_2758 LEVEL = BedrockCauldronProperties.LEVEL_8;
    public static final int MAX_LEVEL = BedrockCauldronProperties.MAX_LEVEL_8;
    /**
     * The level to increase/decrease using a Glass Bottle.
     */
    public static final int BOTTLE_LEVEL = 3;
    /**
     * The level to decrease when creating tipped arrows.
     */
    public static final int ARROW_TIP_LEVEL_PER_STEP = 2;

    /**
     * How many Max Item Stacks are to be divided.
     */
    private static final int ARROW_TIP_STEP = 4;

    public PotionCauldronBlock(class_2251 settings) {
        super(settings, BedrockCauldronBehavior.POTION_CAULDRON_BEHAVIOR);
        this.method_9590(this.method_9595().method_11664().method_11657(LEVEL, 2));
    }

    @Override
    protected void method_9515(class_2689.class_2690<class_2248, class_2680> builder) {
        builder.method_11667(LEVEL);
    }

    @Override
    public boolean method_32766(class_2680 state) {
        return state.method_11654(LEVEL) == MAX_LEVEL;
    }

    @Override
    public int method_9572(class_2680 state, class_1937 world, class_2338 pos, class_2350 direction) {
        return (int) Math.ceil((float) state.method_11654(LEVEL) / MAX_LEVEL * class_5556.field_31108);
    }

    @Override
    protected MapCodec<? extends class_2275> method_53969() {
        return field_46280;
    }

    @Override
    public double method_31615(class_2680 state) {
        return class_3532.method_16436((float) state.method_11654(LEVEL) / MAX_LEVEL, 0.375, 0.9375);
    }

    @Override
    public boolean method_9542(class_2680 state) {
        return true;
    }

    @Override
    public void method_9496(class_2680 state, class_1937 world, class_2338 pos, class_5819 random) {
        if (world == null || pos == null || state == null) {
            return;
        }
        if (random.method_43054() % 7 != 0) {
            return;
        }

        world.method_35230(pos, BedrockCauldronBlocks.WATER_CAULDRON_ENTITY).ifPresent(blockEntity -> {
            final int effectColor = blockEntity.getTintColor();
            final float red = ((effectColor >> 16) & 0xff) / 255.f;
            final float green = ((effectColor >> 8) & 0xff) / 255.f;
            final float blue = (effectColor & 0xff) / 255.f;
            final double offsetY;
            if (state.method_26204() instanceof PotionCauldronBlock potionCauldronBlock) {
                offsetY = potionCauldronBlock.method_31615(state);
            } else {
                offsetY = 0.5;
            }
            final double x = pos.method_10263() + 0.45 + random.method_43058() * 0.2;
            final double y = pos.method_10264() + offsetY;
            final double z = pos.method_10260() + 0.45 + random.method_43058() * 0.2;
            world.method_8406(class_9381.method_58255(class_2398.field_11226, red, green, blue), x, y, z, red, green, blue);
        });
    }

    /**
     * Tries to take out the fluid. If succeeded, the {@link class_2680} will be changed.
     *
     * @param state The BlockState that contains the state of {@link PotionCauldronBlock}.
     * @param world The instance of {@link class_1937}.
     * @param pos   Target block position.
     * @return <code>true</code> if it can be taken out.
     */
    public static boolean tryPickFluid(class_2680 state, class_1937 world, class_2338 pos) {
        if (world.method_8608()) {
            return false;
        }

        if (!state.method_28498(LEVEL)) {
            Bedrockify.LOGGER.error(
                    "[{}] cannot retrieve fluid level", Bedrockify.class.getSimpleName(),
                    new IllegalStateException("BlockState of %s does not have state: %s".formatted(state.method_26204(), LEVEL)));
            return false;
        }

        final int currentLevel = state.method_11654(LEVEL);
        if (currentLevel < MAX_LEVEL - BOTTLE_LEVEL * 2) {
            return false;
        }

        final int nextLevel = currentLevel - BOTTLE_LEVEL;
        final class_2680 blockState = (nextLevel <= 0) ? class_2246.field_10593.method_9564() : state.method_11657(LEVEL, nextLevel);
        world.method_8501(pos, blockState);
        world.method_43276(class_5712.field_28733, pos, class_5712.class_7397.method_43287(blockState));

        return true;
    }

    /**
     * Gets the maximum number of available stack count for creating Tipped Arrow.<br>
     * This calculation is based on {@link #ARROW_TIP_STEP} and {@link #ARROW_TIP_LEVEL_PER_STEP}.
     * The result is as follows:<br>
     * <ul>
     *     <li>fluid: 2 - Potion 1x<br>
     *     <pre>16 * floor( 2 / 2) => 16 * 1 => returns 16</pre></li>
     *     <li>fluid: 5 - Potion 2x<br>
     *     <pre>16 * floor( 5 / 2) => 16 * 2 => returns 32</pre></li>
     *     <li>fluid: 8: MAX - Potion 3x<br>
     *     <pre>16 * floor( 8 / 2) => 16 * 4 => returns 64</pre></li>
     *     <li>fluid: 6 - when level is full and then 16 arrows tipped<br>
     *     <pre>16 * floor( 6 / 2) => 16 * 3 => returns 48</pre></li>
     * </ul>
     *
     * @param itemStack Target {@link class_1799} that for creating Tipped Arrow.
     * @param state     The {@link class_2680} that contains the state of {@link PotionCauldronBlock}.
     * @return The stack count.
     * @see PotionCauldronBlock#getArrowTipStepCount
     */
    public static int getMaxTippedArrowCount(class_1799 itemStack, class_2680 state) {
        if (!state.method_28498(LEVEL)) {
            Bedrockify.LOGGER.error(
                    "[{}] cannot retrieve fluid level", Bedrockify.class.getSimpleName(),
                    new IllegalStateException("BlockState of %s does not have the state: LEVEL".formatted(state.method_26204())));
            return 0;
        }

        final int mul = (int) Math.floor((float) state.method_11654(LEVEL) / ARROW_TIP_LEVEL_PER_STEP);
        return Math.min(getArrowTipStepCount(itemStack) * mul, itemStack.method_7947());
    }

    /**
     * Gets the level to decrease the fluid.<br>
     * This calculation is based on {@link #ARROW_TIP_STEP} and {@link #ARROW_TIP_LEVEL_PER_STEP}.
     * The result is as follows:<br>
     * <ul>
     *     <li>count: 16<br>
     *     <pre>(ceil(16 / 16) + 0) * 2 - 0 => 1 * 2 - 0 => returns 2</pre></li>
     *     <li>count: 32<br>
     *     <pre>(ceil(32 / 16) + 1) * 2 - 1 => 3 * 2 - 1 => returns 5</pre></li>
     *     <li>count: 48<br>
     *     <pre>(ceil(48 / 16) + 1) * 2 - 2 => 4 * 2 - 2 => returns 6</pre></li>
     *     <li>count: 64<br>
     *     <pre>(ceil(64 / 16) + 2) * 2 - 3 => 6 * 2 - 3 => returns 9</pre></li>
     * </ul>
     *
     * @param itemStack Target {@link class_1799} that for creating Tipped Arrow.
     * @param count     The item count in stack.
     * @return The fluid level to decrease.
     * @see PotionCauldronBlock#getArrowTipStepCount
     */
    public static int getDecLevelByStack(class_1799 itemStack, int count) {
        if (count <= 0) {
            return 0;
        }

        final int step = (int) Math.ceil((float) count / getArrowTipStepCount(itemStack));
        return (step + (step >> 1)) * ARROW_TIP_LEVEL_PER_STEP - (step - 1);
    }

    /**
     * Returns the step count.<br>
     * The count will be divided by {@link #ARROW_TIP_STEP} for the maximum number that can retrieve by {@link class_1799#method_7914}.<br>
     * Default return value is <code>16</code>.
     *
     * @param itemStack Target item stack.
     * @return Divided by {@link #ARROW_TIP_STEP}.
     */
    private static int getArrowTipStepCount(class_1799 itemStack) {
        return (int) Math.ceil((float) itemStack.method_7914() / ARROW_TIP_STEP);
    }
}
