package me.juancarloscp52.bedrockify.common.block.cauldron;

import me.juancarloscp52.bedrockify.Bedrockify;
import me.juancarloscp52.bedrockify.common.block.ColoredWaterCauldronBlock;
import me.juancarloscp52.bedrockify.common.block.PotionCauldronBlock;
import me.juancarloscp52.bedrockify.common.block.entity.WaterCauldronBlockEntity;
import me.juancarloscp52.bedrockify.common.features.cauldron.BedrockCauldronBlocks;
import me.juancarloscp52.bedrockify.common.features.cauldron.ColorBlenderHelper;
import me.juancarloscp52.bedrockify.common.payloads.CauldronParticlePayload;
import net.fabricmc.fabric.api.networking.v1.PlayerLookup;
import net.fabricmc.fabric.api.networking.v1.ServerPlayNetworking;
import net.minecraft.class_1268;
import net.minecraft.class_1269;
import net.minecraft.class_1657;
import net.minecraft.class_1769;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1812;
import net.minecraft.class_1842;
import net.minecraft.class_1844;
import net.minecraft.class_1847;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2398;
import net.minecraft.class_243;
import net.minecraft.class_2680;
import net.minecraft.class_2960;
import net.minecraft.class_3218;
import net.minecraft.class_3417;
import net.minecraft.class_3419;
import net.minecraft.class_3468;
import net.minecraft.class_3489;
import net.minecraft.class_5328;
import net.minecraft.class_5556;
import net.minecraft.class_5620;
import net.minecraft.class_5712;
import net.minecraft.class_6880;
import net.minecraft.class_7923;
import net.minecraft.class_9334;
import net.minecraft.item.*;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

/**
 * Defines the behavior of Bedrock's Cauldron.
 */
public interface BedrockCauldronBehavior {
    class_5620.class_8821 POTION_CAULDRON_BEHAVIOR = class_5620.method_32206("potion");
    class_5620.class_8821 COLORED_WATER_CAULDRON_BEHAVIOR = class_5620.method_32206("dye");

    class_5620 DYE_ITEM_BY_COLORED_WATER = (state, world, pos, player, hand, stack) -> {
        if (state == null || world == null || pos == null || !Bedrockify.getInstance().settings.bedrockCauldron) {
            return class_1269.field_52423;
        }

        if (!stack.method_31573(class_3489.field_48803)) {
            return class_1269.field_52423;
        }

        Optional<WaterCauldronBlockEntity> entity = retrieveCauldronEntity(world, pos);
        if (entity.isEmpty()) {
            return emptyCauldronFromWrongState(state, world, pos, player, hand, stack);
        }

        final WaterCauldronBlockEntity blockEntity = entity.get();
        if (!world.method_8608()) {
            ColoredWaterCauldronBlock.decrementWhenDye(state, world, pos);
            player.method_7281(class_3468.field_15373);
            player.method_6122(hand, ColorBlenderHelper.blendColors(stack, blockEntity.getTintColor()));
            world.method_8396(null, pos, class_3417.field_14737, class_3419.field_15245, 0.15f, 1.25f);
            world.method_33596(null, class_5712.field_28167, pos);
        }

        return class_1269.field_5812;
    };

    class_5620 DYE_WATER = (state, world, pos, player, hand, stack) -> {
        if (state == null || world == null || pos == null || !Bedrockify.getInstance().settings.bedrockCauldron) {
            return class_1269.field_52423;
        }

        class_1792 item = stack.method_7909();
        if (!(item instanceof class_1769 dyeItem)) {
            return class_1269.field_52423;
        }

        Optional<WaterCauldronBlockEntity> entity = world.method_35230(pos, BedrockCauldronBlocks.WATER_CAULDRON_ENTITY);
        final int level;
        final int nextColor;
        if (entity.isPresent()) {
            // The Cauldron already has color.
            final int currentColor = entity.get().getTintColor();
            nextColor = ColorBlenderHelper.blendColors(currentColor, ColorBlenderHelper.fromDyeItem(dyeItem));
            if (nextColor == currentColor) {
                return class_1269.field_5812;
            }
            level = state.method_11654(ColoredWaterCauldronBlock.LEVEL);
        } else {
            // Otherwise it may be Water Cauldron.
            nextColor = ColorBlenderHelper.fromDyeItem(dyeItem);
            level = ColoredWaterCauldronBlock.getLevelFromWaterCauldronState(state);
        }

        if (!world.method_8608()) {
            class_2680 newState = BedrockCauldronBlocks.COLORED_WATER_CAULDRON.method_9564()
                    .method_11657(ColoredWaterCauldronBlock.LEVEL, level);
            world.method_8501(pos, newState);
            if (!player.method_31549().field_7477) {
                stack.method_7934(1);
            }
            player.method_7259(class_3468.field_15372.method_14956(item));
            world.method_45447(null, pos, class_3417.field_28391, class_3419.field_15245);
            world.method_33596(null, class_5712.field_28733, pos);
        }
        world.method_35230(pos, BedrockCauldronBlocks.WATER_CAULDRON_ENTITY).ifPresent(blockEntity -> {
            blockEntity.setDyeColor(nextColor);
        });

        return class_1269.field_5812;
    };

    /**
     * Increases the water level and replaces the block with the vanilla Water Cauldron.
     */
    class_5620 PLACE_WATER_BY_POTION = (state, world, pos, player, hand, stack) -> {
        if (state == null || world == null || pos == null || !Bedrockify.getInstance().settings.bedrockCauldron) {
            return class_1269.field_52423;
        }
        var component = stack.method_58694(class_9334.field_49651);
        if (component != null && !component.method_57401(class_1847.field_8991)) {
            return class_1269.field_5812;
        }

        final int nextLevel;
        if (state.method_27852(class_2246.field_27097)) {
            if (state.method_11654(class_5556.field_27206) >= class_5556.field_31108) {
                return class_1269.field_5812;
            }
            nextLevel = state.method_28493(class_5556.field_27206).method_11654(class_5556.field_27206);
        } else if (state.method_27852(BedrockCauldronBlocks.COLORED_WATER_CAULDRON)) {
            if (state.method_11654(ColoredWaterCauldronBlock.LEVEL) >= ColoredWaterCauldronBlock.MAX_LEVEL) {
                return class_1269.field_5812;
            }
            nextLevel = Math.min(ColoredWaterCauldronBlock.convertToWaterCauldronLevel(state) + 1, class_5556.field_31108);
        } else {
            nextLevel = class_2246.field_27097.method_9564().method_11654(class_5556.field_27206);
        }

        if (!world.method_8608()) {
            // Replace with Water Cauldron.
            player.method_7281(class_3468.field_15373);
            player.method_7259(class_3468.field_15372.method_14956(stack.method_7909()));
            player.method_6122(hand, class_5328.method_30012(stack, player, new class_1799(class_1802.field_8469)));
            world.method_8501(pos, class_2246.field_27097.method_9564().method_11657(class_5556.field_27206, nextLevel));
            world.method_45447(null, pos, class_3417.field_14826, class_3419.field_15245);
            world.method_33596(null, class_5712.field_28166, pos);
        }

        return class_1269.field_5812;
    };

    /**
     * Decreases the fluid level and takes out from the Potion Cauldron.
     */
    class_5620 PICK_POTION_FLUID = (state, world, pos, player, hand, stack) -> {
        if (state == null || world == null || pos == null || !Bedrockify.getInstance().settings.bedrockCauldron) {
            return class_1269.field_52423;
        }

        Optional<WaterCauldronBlockEntity> entity = retrieveCauldronEntity(world, pos);
        if (entity.isEmpty()) {
            return emptyCauldronFromWrongState(state, world, pos, player, hand, stack);
        }

        final WaterCauldronBlockEntity blockEntity = entity.get();
        final class_1792 item = stack.method_7909();
        Optional<class_1842> optionalPotion = retrievePotion(blockEntity);
        if (optionalPotion.isEmpty()) {
            return emptyCauldronFromWrongState(state, world, pos, player, hand, stack);
        }

        final class_1842 potion = optionalPotion.get();
        class_6880<class_1842> potionEntry = class_7923.field_41179.method_47983(potion);
        final class_1792 potionType = blockEntity.getPotionType();
        if (!world.method_8608()) {
            if (!PotionCauldronBlock.tryPickFluid(state, world, pos)) {
                return class_1269.field_52422;
            }
            final class_1799 potionStack = class_1844.method_57400(potionType, potionEntry);
            if (!potionStack.method_57826(class_9334.field_49651)) {
                return class_1269.field_52422;
            }
            final int potionColor = Objects.requireNonNull(potionStack.method_58694(class_9334.field_49651)).method_8064();
            player.method_7281(class_3468.field_15373);
            player.method_7259(class_3468.field_15372.method_14956(item));
            player.method_6122(hand, class_5328.method_30012(stack, player, potionStack));
            world.method_45447(null, pos, class_3417.field_14779, class_3419.field_15245);
            world.method_33596(null, class_5712.field_28167, pos);
            addPotionParticle(world.method_8320(pos), world, pos, potionColor);
        }

        return class_1269.field_5812;
    };

    /**
     * Replaces Empty Cauldron with Potion Cauldron, or increases its fluid.
     */
    class_5620 PLACE_POTION_FLUID = (state, world, pos, player, hand, stack) -> {
        if (state == null || world == null || pos == null || !Bedrockify.getInstance().settings.bedrockCauldron) {
            return class_1269.field_52423;
        }

        if (!(stack.method_7909() instanceof class_1812)) {
            return class_1269.field_52423;
        }

        var component = stack.method_58694(class_9334.field_49651);
        if (component == null) {
            return class_1269.field_52423;
        }
        var optionalPotion = component.comp_2378();
        if (optionalPotion.isEmpty()) {
            return class_1269.field_52423;
        }
        var potionEntry = optionalPotion.get();
        if (potionEntry.comp_349().method_8049().isEmpty() && !component.method_57401(class_1847.field_8991)) {
            // Prevent to place the fluid of Awkward Potion, Mundane Potion, etc.
            return class_1269.field_5812;
        }

        final class_2680 newState;
        final class_2960 inStackPotionId = class_7923.field_41179.method_10221(potionEntry.comp_349());
        if (state.method_28498(PotionCauldronBlock.LEVEL)) {
            // The cauldron already has fluid.
            Optional<WaterCauldronBlockEntity> entity = retrieveCauldronEntity(world, pos);
            if (entity.isEmpty()) {
                return emptyCauldronFromWrongState(state, world, pos, player, hand, stack);
            }

            final class_2960 currentPotionId = entity.get().getFluidId();
            // Not the same potion.
            if (!Objects.equals(currentPotionId, inStackPotionId)) {
                return evaporateCauldron(state, world, pos, player, hand, stack, new class_1799(class_1802.field_8469));
            }

            // The cauldron is full.
            final int currentLevel = state.method_11654(PotionCauldronBlock.LEVEL);
            if (currentLevel >= PotionCauldronBlock.MAX_LEVEL) {
                return class_1269.field_52422;
            }

            final int nextLevel = Math.min(currentLevel + PotionCauldronBlock.BOTTLE_LEVEL, PotionCauldronBlock.MAX_LEVEL);
            newState = BedrockCauldronBlocks.POTION_CAULDRON.method_9564().method_11657(PotionCauldronBlock.LEVEL, nextLevel);
        } else {
            // The cauldron is empty.
            newState = BedrockCauldronBlocks.POTION_CAULDRON.method_9564();
        }

        if (component.method_57401(class_1847.field_8991)) {
            // Redirect to the behavior of Water Cauldron.
            return PLACE_WATER_BY_POTION.interact(state, world, pos, player, hand, stack);
        }

        final class_1799 processing = stack.method_7972();
        if (!world.method_8608()) {
            player.method_7281(class_3468.field_15373);
            player.method_7259(class_3468.field_15372.method_14956(stack.method_7909()));
            player.method_6122(hand, class_5328.method_30012(stack, player, new class_1799(class_1802.field_8469)));
            world.method_8501(pos, newState);
            world.method_45447(null, pos, class_3417.field_14826, class_3419.field_15245);
            world.method_33596(null, class_5712.field_28166, pos);
            addPotionParticle(newState, world, pos, component.method_8064());
        }

        // BlockEntity spawns after replacing the block with BedrockCauldronBlocks#POTION_CAULDRON.
        world.method_35230(pos, BedrockCauldronBlocks.WATER_CAULDRON_ENTITY).ifPresent(blockEntity -> {
            blockEntity.setPotion(processing);
        });

        return class_1269.field_5812;
    };

    /**
     * Creates tipped arrows.
     */
    class_5620 TIPPED_ARROW_WITH_POTION = (state, world, pos, player, hand, stack) -> {
        if (state == null || world == null || pos == null || !Bedrockify.getInstance().settings.bedrockCauldron) {
            return class_1269.field_52423;
        }

        if (!stack.method_31574(class_1802.field_8107)) {
            return class_1269.field_52423;
        }

        Optional<WaterCauldronBlockEntity> entity = retrieveCauldronEntity(world, pos);
        if (entity.isEmpty()) {
            return emptyCauldronFromWrongState(state, world, pos, player, hand, stack);
        }

        final WaterCauldronBlockEntity blockEntity = entity.get();
        Optional<class_1842> optionalPotion = retrievePotion(blockEntity);
        if (optionalPotion.isEmpty()) {
            return emptyCauldronFromWrongState(state, world, pos, player, hand, stack);
        }

        final class_1842 potion = optionalPotion.get();
        class_6880<class_1842> potionEntry = class_7923.field_41179.method_47983(potion);
        if (!world.method_8608()) {
            // Determine arrow count and fluid level to decrease.
            final int tippedArrowCount = PotionCauldronBlock.getMaxTippedArrowCount(stack, state);
            if (tippedArrowCount <= 0) {
                return class_1269.field_52422;
            }
            final int consumedFluidLevel = PotionCauldronBlock.getDecLevelByStack(stack, tippedArrowCount);
            final int afterFluidLevel = state.method_11654(PotionCauldronBlock.LEVEL) - consumedFluidLevel;

            // Create tipped arrows.
            final class_1799 result = new class_1799(class_1802.field_8087, tippedArrowCount);
            final class_1799 lingeringEffect = class_1844.method_57400(class_1802.field_8150, potionEntry);
            result.method_57379(class_9334.field_49651, lingeringEffect.method_58694(class_9334.field_49651));

            if (player.method_68878()) {
                if (!player.method_31548().method_7379(result)) {
                    player.method_31548().method_7394(result);
                }

                player.method_6122(hand, stack);
            } else {
                // Exchange an item.
                stack.method_7934(tippedArrowCount);

                if (stack.method_7960()) {
                    player.method_6122(hand, result);
                } else {
                    if (!player.method_31548().method_7394(result)) {
                        player.method_7328(result, false);
                    }
                    player.method_6122(hand, stack);
                }
            }

            // Consume the fluid.
            if (afterFluidLevel <= 1) {
                world.method_8501(pos, class_2246.field_10593.method_9564());
            } else {
                world.method_8501(pos, BedrockCauldronBlocks.POTION_CAULDRON.method_9564().method_11657(PotionCauldronBlock.LEVEL, afterFluidLevel));
            }
            player.method_7281(class_3468.field_15373);
            player.method_7342(class_3468.field_15372.method_14956(class_1802.field_8107), tippedArrowCount);
            world.method_8396(null, pos, class_3417.field_14737, class_3419.field_15245, 0.15f, 1.25f);
            world.method_33596(null, class_5712.field_28167, pos);
        }

        return class_1269.field_5812;
    };

    class_5620 PICK_COLORED_WATER = (state, world, pos, player, hand, stack) -> {
        if (!world.method_8608()) {
            if (!ColoredWaterCauldronBlock.tryPickFluid(state, world, pos)) {
                return class_1269.field_52422;
            }
            class_1792 item = stack.method_7909();
            player.method_6122(hand, class_5328.method_30012(stack, player, class_1844.method_57400(class_1802.field_8574, class_1847.field_8991)));
            player.method_7281(class_3468.field_15373);
            player.method_7259(class_3468.field_15372.method_14956(item));
            world.method_45447(null, pos, class_3417.field_14779, class_3419.field_15245);
            world.method_33596(null, class_5712.field_28167, pos);
        }

        return class_1269.field_5812;
    };

    class_5620 FILL_BUCKET_WITH_COLORED_WATER = (state, world, pos, player, hand, stack) -> {
        return class_5620.method_32210(state, world, pos, player, hand, stack, new class_1799(class_1802.field_8705), (statex) -> {
            return statex.method_11654(ColoredWaterCauldronBlock.LEVEL) == ColoredWaterCauldronBlock.MAX_LEVEL;
        }, class_3417.field_15126);
    };

    /**
     * Helper method that checks and retrieves the {@link WaterCauldronBlockEntity}.<br>
     * There are no Entity, the error log will appear.
     *
     * @return {@link Optional} of WaterCauldronBlockEntity.
     */
    static Optional<WaterCauldronBlockEntity> retrieveCauldronEntity(class_1937 world, class_2338 pos) {
        Optional<WaterCauldronBlockEntity> entity = world.method_35230(pos, BedrockCauldronBlocks.WATER_CAULDRON_ENTITY);
        if (entity.isEmpty()) {
            // Increasing the fluid requires to retrieve the potion.
            Bedrockify.LOGGER.error(
                    "[{}] something went wrong to get the fluid in cauldron",
                    Bedrockify.class.getSimpleName(),
                    new NullPointerException("RegistryWorldView#getBlockEntity is not present at " + pos));
            return Optional.empty();
        }
        return entity;
    }

    /**
     * Helper method that checks exact potion fluid and returns {@link class_1842}.
     *
     * @return {@link Optional} of Potion.
     */
    static Optional<class_1842> retrievePotion(WaterCauldronBlockEntity blockEntity) {
        final class_2960 potionId = blockEntity.getFluidId();
        final class_1842 potion = class_7923.field_41179.method_63535(potionId);
        if (!class_7923.field_41179.method_10250(potionId) || potion == null) {
            Bedrockify.LOGGER.error(
                    "[{}] something went wrong to get the potion from Registries",
                    Bedrockify.class.getSimpleName(),
                    new IllegalStateException("potion has disappeared, maybe the mod is gone?"));
            return Optional.empty();
        }
        return Optional.of(potion);
    }

    static class_1269 emptyCauldronFromWrongState(class_2680 state, class_1937 world, class_2338 pos, class_1657 player, class_1268 hand, class_1799 stack) {
        return evaporateCauldron(state, world, pos, player, hand, stack, stack.method_7972());
    }

    static void registerEvaporateBucketBehavior(Map<class_1792, class_5620> map) {
        map.put(class_1802.field_8705, (state, world, pos, player, hand, stack) -> {
            return evaporateCauldron(state, world, pos, player, hand, stack, new class_1799(class_1802.field_8550));
        });
        map.put(class_1802.field_8187, (state, world, pos, player, hand, stack) -> {
            return evaporateCauldron(state, world, pos, player, hand, stack, new class_1799(class_1802.field_8550));
        });
        map.put(class_1802.field_27876, (state, world, pos, player, hand, stack) -> {
            return evaporateCauldron(state, world, pos, player, hand, stack, new class_1799(class_1802.field_8550));
        });
    }

    /**
     * Empties the cauldron by spawning particles and sound.
     */
    static class_1269 evaporateCauldron(class_2680 state, class_1937 world, class_2338 pos, class_1657 player, class_1268 hand, class_1799 current, class_1799 after) {
        if (!world.method_8608()) {
            final class_2960 particleId = class_7923.field_41180.method_10221(class_2398.field_11203);
            for (int i = 0; i < 10; ++i) {
                CauldronParticlePayload particlePayload = new CauldronParticlePayload();
                particlePayload.setParticleType(particleId);
                particlePayload.setPosition(new class_243(pos.method_10263() + 0.25 + Math.random() * 0.5, pos.method_10264() + 0.35, pos.method_10260() + 0.25 + Math.random() * 0.5));
                particlePayload.setVelocity(new class_243(Math.random() * 0.075, 0, Math.random() * 0.075));

                PlayerLookup.world((class_3218) world).forEach(serverPlayerEntity -> {
                    ServerPlayNetworking.send(serverPlayerEntity, particlePayload);
                });
            }
        }
        return class_5620.method_32210(state, world, pos, player, hand, current, after, statex -> true, class_3417.field_15102);
    }

    /**
     * Spawns particles after interacting with a potion.
     */
    static void addPotionParticle(class_2680 state, class_1937 world, class_2338 pos, int color) {
        if (world.method_8608()) {
            return;
        }

        final double offsetY;
        if (state.method_26204() instanceof PotionCauldronBlock potionCauldronBlock) {
            offsetY = 0.05 + potionCauldronBlock.method_31615(state);
        } else {
            offsetY = 0.5;
        }
        final double red = ((color >> 16) & 0xff) / 255.;
        final double green = ((color >> 8) & 0xff) / 255.;
        final double blue = (color & 0xff) / 255.;
        final class_2960 particleId = class_7923.field_41180.method_10221(class_2398.field_11226);
        for (int i = 0; i < 7; ++i) {
            CauldronParticlePayload particlePayload = new CauldronParticlePayload();
            particlePayload.setParticleType(particleId);
            particlePayload.setPosition(new class_243(pos.method_10263() + 0.15 + Math.random() * 0.7, pos.method_10264() + offsetY, pos.method_10260() + 0.15 + Math.random() * 0.7));
            particlePayload.setVelocity(new class_243(red, green, blue));

            PlayerLookup.world((class_3218) world).forEach(serverPlayerEntity -> {
                ServerPlayNetworking.send(serverPlayerEntity, particlePayload);
            });
        }
    }

    /**
     * Registers the behavior of Bedrock's cauldron from all items and potions.<br>
     * This method needs to be executed after all the registries are ready.
     */
    static void registerBehavior() {
        final var dyeableBehaviorMap = COLORED_WATER_CAULDRON_BEHAVIOR.comp_1982();
        final var potionBehaviorMap = POTION_CAULDRON_BEHAVIOR.comp_1982();
        final var vanillaWaterBehaviorMap = class_5620.field_27776.comp_1982();
        final var vanillaEmptyBehaviorMap = class_5620.field_27775.comp_1982();

        // Clear all modded maps and get ready to enter the world.
        class_7923.field_41178.method_10220().filter(item -> item instanceof class_1769).forEach(vanillaWaterBehaviorMap::remove);
        class_7923.field_41178.method_10220().filter(item -> item instanceof class_1812).forEach(potionItem -> {
            vanillaEmptyBehaviorMap.remove(potionItem);
            vanillaWaterBehaviorMap.remove(potionItem);
        });
        dyeableBehaviorMap.clear();
        potionBehaviorMap.clear();

        // Behavior of the dye item for water cauldron.
        class_7923.field_41178.method_10220().filter(item -> item instanceof class_1769).forEach(dyeItem -> {
            vanillaWaterBehaviorMap.putIfAbsent(dyeItem, DYE_WATER);
            dyeableBehaviorMap.putIfAbsent(dyeItem, DYE_WATER);
        });

        // Behavior of the colored cauldron.
        class_7923.field_41178.method_10220().filter(item -> item.method_7854().method_31573(class_3489.field_48803)).forEach(item -> {
            dyeableBehaviorMap.putIfAbsent(item, DYE_ITEM_BY_COLORED_WATER);
        });
        dyeableBehaviorMap.putIfAbsent(class_1802.field_8550, FILL_BUCKET_WITH_COLORED_WATER);
        dyeableBehaviorMap.putIfAbsent(class_1802.field_8469, PICK_COLORED_WATER);
        class_5620.method_34850(dyeableBehaviorMap);

        // Behavior of the potion.
        class_7923.field_41178.method_10220().filter(item -> item instanceof class_1812).forEach(potionItem -> {
            vanillaEmptyBehaviorMap.putIfAbsent(potionItem, PLACE_POTION_FLUID);
            potionBehaviorMap.putIfAbsent(potionItem, PLACE_POTION_FLUID);
            // Allows to accept any potion type for the Water Cauldron.
            vanillaWaterBehaviorMap.putIfAbsent(potionItem, PLACE_WATER_BY_POTION);
            dyeableBehaviorMap.putIfAbsent(potionItem, PLACE_WATER_BY_POTION);
        });
        potionBehaviorMap.putIfAbsent(class_1802.field_8469, PICK_POTION_FLUID);
        potionBehaviorMap.putIfAbsent(class_1802.field_8107, TIPPED_ARROW_WITH_POTION);
        registerEvaporateBucketBehavior(potionBehaviorMap);
    }
}
