/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.api;

import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientConfig.api.ui.screen.ConfigScreen;
import io.github.axolotlclient.api.chat.ChannelInvitesScreen;
import io.github.axolotlclient.api.chat.ChatListScreen;
import io.github.axolotlclient.api.requests.AccountDataRequest;
import io.github.axolotlclient.api.requests.AccountSettingsRequest;
import io.github.axolotlclient.util.ThreadExecuter;
import io.github.axolotlclient.util.options.GenericOption;
import lombok.Getter;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.unmapped.C_1441789;
import net.minecraft.unmapped.C_3020744;
import net.minecraft.unmapped.C_3390001;
import net.minecraft.unmapped.C_7778778;
import net.minecraft.unmapped.C_8105098;
import net.ornithemc.osl.keybinds.api.KeyBindingEvents;
import net.ornithemc.osl.lifecycle.api.client.MinecraftClientEvents;
import org.lwjgl.input.Keyboard;
import org.lwjgl.system.MemoryStack;
import org.lwjgl.util.tinyfd.TinyFileDialogs;

public class APIOptions extends Options {

	@Getter
	private static final Options Instance = new APIOptions();

	@Override
	public void init() {
		super.init();
		C_8105098 client = C_8105098.m_0408063();

		openPrivacyNoteScreen = () -> {
			var fut = new CompletableFuture<Boolean>();
			client.m_2167033(() -> client.m_6408915(new PrivacyNoticeScreen(client.f_0723335, fut)));
			return fut;
		};
		C_7778778 openSidebar = new C_7778778("api.chats.sidebar.open", Keyboard.KEY_O, "category.axolotlclient");
		KeyBindingEvents.REGISTER_KEYBINDS.register(registry -> registry.register(openSidebar));
		MinecraftClientEvents.TICK_END.register(minecraft -> {
			if (openSidebar.m_4823804() && API.getInstance().isAuthenticated()) {
				minecraft.m_6408915(new ChatsSidebar(client.f_0723335));
			}
		});
		category.add(new GenericOption("viewFriends", "clickToOpen",
			() -> client.m_6408915(new FriendsScreen(client.f_0723335))));
		category.add(new GenericOption("viewChats", "clickToOpen",
			() -> client.m_6408915(new ChatListScreen(client.f_0723335))));
		category.add(new GenericOption("api.channels.invites.view", "clickToOpen",
			() -> client.m_6408915(new ChannelInvitesScreen(client.f_0723335))));
		account.add(new GenericOption("api.account.usernames", "clickToOpen",
			() -> client.m_6408915(new UsernameManagementScreen(client.f_0723335))));
		account.add(new GenericOption("api.account.export", "api.account.export_data", () -> ThreadExecuter.scheduleTask(() -> {
			if (!API.getInstance().isAuthenticated()) {
				API.getInstance().getNotificationProvider().addStatus("api.account.export.failure.title", "api.error.unauthenticated");
				return;
			}
			try (MemoryStack stack = MemoryStack.stackPush()) {
				var pointers = stack.pointers(stack.UTF8("*.json"));
				var result = TinyFileDialogs.tinyfd_saveFileDialog("Choose export destination", FabricLoader.getInstance().getGameDir().toString(), pointers, null);
				if (result != null) {
					AccountDataRequest.get(Path.of(result));
				}
			}
		})));
		account.add(new GenericOption("api.account.delete", "api.account.delete_account", () -> {
			C_3020744 previous = client.f_0723335;
			client.m_6408915(new C_1441789((b, i) -> {
				if (b) {
					if (!API.getInstance().isAuthenticated()) {
						API.getInstance().getNotificationProvider().addStatus("api.account.deletion.failure", "api.error.unauthenticated");
					} else {
						AccountSettingsRequest.deleteAccount().thenAccept(r -> {
							if (r) {
								API.getInstance().getNotificationProvider().addStatus("api.account.deletion.success", "api.account.deletion.success.desc");
							} else {
								API.getInstance().getNotificationProvider().addStatus("api.account.deletion.failure", "api.account.deletion.failure.desc");
							}
							enabled.set(false);
						});
					}
				}
				client.m_6408915(previous);
			}, C_3390001.m_2053009("api.account.confirm_deletion"),
				C_3390001.m_2053009("api.account.confirm_deletion.desc"), 0));
		}));
		Consumer<Boolean> consumer = settingUpdated;
		settingUpdated = b -> {
			if (client.f_0723335 instanceof ConfigScreen) {
				consumer.accept(b);
			}
		};
		if (Constants.ENABLED) {
			AxolotlClient.config().addCategory(category);
		}
	}
}
