/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.api;

import java.util.List;
import net.minecraft.unmapped.C_3020744;
import net.minecraft.unmapped.C_3390001;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_4976084;
import net.minecraft.unmapped.C_5819357;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.ClickableWidget;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.vanilla.widgets.VanillaButtonWidget;
import io.github.axolotlclient.api.requests.GlobalDataRequest;
import io.github.axolotlclient.modules.hud.util.DrawUtil;

public class NewsScreen extends io.github.axolotlclient.AxolotlClientConfig.impl.ui.Screen {


	private final C_3020744 parent;

	public NewsScreen(C_3020744 parent) {
		super(C_3390001.m_2053009("api.notes.title"));

		this.parent = parent;
	}

	@Override
	public void m_7261014(int mouseX, int mouseY, float delta) {
		super.m_7261014(mouseX, mouseY, delta);

		m_2717572(f_2020658, title, f_5465691 / 2, 20, -1);
	}

	@Override
	public void m_3593494() {
		GlobalDataRequest.get().thenAccept(data -> addDrawableChild(new NewsWidget(25, 35, f_5465691 - 50, f_3080061 - 100, data.notes().trim().replaceAll("([^\n])\n([^\n])", "$1 $2"))));

		addDrawableChild(new VanillaButtonWidget(f_5465691 / 2 - 100, f_3080061 - 45, 200, 20, C_3390001.m_2053009("gui.back"), buttonWidget -> f_7153641.m_6408915(parent)));
	}

	private class NewsWidget extends AbstractTextAreaWidget {

		private final List<String> lines;
		private final int contentHeight;

		public NewsWidget(int x, int y, int width, int height, String component) {
			super(x, y, width, height, component);
			lines = f_2020658.m_0466830(getMessage(), getWidth() - 4);
			contentHeight = lines.size() * f_2020658.f_6725889;
		}

		@Override
		protected int getInnerHeight() {
			return contentHeight;
		}

		@Override
		protected void renderContents(int mouseX, int mouseY, float partialTick) {
			int y = getY() + 2;
			for (String chsq : lines) {
				m_6649515(f_2020658, chsq, getX() + 2, y, -1);
				y += f_2020658.f_6725889;
			}
		}

		@Override
		protected double scrollRate() {
			return f_2020658.f_6725889;
		}

		@Override
		protected void renderBackground() {
		}
	}

	public abstract static class AbstractTextAreaWidget extends AbstractScrollArea {
		private static final int INNER_PADDING = 4;

		public AbstractTextAreaWidget(int x, int y, int width, int height, String component) {
			super(x, y, width, height, component);
		}

		@Override
		public boolean mouseClicked(double mouseX, double mouseY, int button) {
			boolean bl = this.updateScrolling(mouseX, mouseY, button);
			return super.mouseClicked(mouseX, mouseY, button) || bl;
		}

		@Override
		public boolean keyPressed(int keyCode, int scanCode, int modifiers) {
			boolean bl = keyCode == 265;
			boolean bl2 = keyCode == 264;
			if (bl || bl2) {
				double d = this.scrollAmount();
				this.setScrollAmount(this.scrollAmount() + (double) (bl ? -1 : 1) * this.scrollRate());
				if (d != this.scrollAmount()) {
					return true;
				}
			}

			return super.keyPressed(keyCode, scanCode, modifiers);
		}

		@Override
		public void drawWidget(int mouseX, int mouseY, float partialTick) {
			if (this.visible) {
				this.renderBackground();
				DrawUtil.enableScissor(this.getX() + 1, this.getY() + 1, this.getX() + this.getWidth() - 1, this.getY() + this.getHeight() - 1);
				C_3754158.m_8373640();
				C_3754158.m_5322104(0.0, -this.scrollAmount(), 0.0);
				this.renderContents(mouseX, mouseY, partialTick);
				C_3754158.m_2041265();
				DrawUtil.disableScissor();
				renderDecorations();
			}
		}

		protected void renderDecorations() {
			this.renderScrollbar();
		}

		protected int innerPadding() {
			return INNER_PADDING;
		}

		protected int totalInnerPadding() {
			return this.innerPadding() * 2;
		}

		@Override
		public boolean isMouseOver(double mouseX, double mouseY) {
			return this.active
				&& this.visible
				&& mouseX >= (double) this.getX()
				&& mouseY >= (double) this.getY()
				&& mouseX < (double) (this.getXEnd() + 6)
				&& mouseY < (double) this.getYEnd();
		}

		@Override
		protected int scrollBarX() {
			return this.getXEnd();
		}

		@Override
		protected int contentHeight() {
			return this.getInnerHeight() + this.totalInnerPadding();
		}

		protected void renderBackground() {
			this.renderBorder(this.getX(), this.getY(), this.getWidth(), this.getHeight());
		}

		protected void renderBorder(int x, int y, int width, int height) {
			int i = this.isFocused() ? -1 : -6250336;
			m_7865719(this.getX(), this.getY(), this.getXEnd(), this.getYEnd(), i);
			m_7865719(this.getX() + 1, this.getY() + 1, this.getXEnd() - 1, this.getYEnd() - 1, -16777216);
		}

		protected boolean withinContentAreaTopBottom(int top, int bottom) {
			return (double) bottom - this.scrollAmount() >= (double) this.getY() && (double) top - this.scrollAmount() <= (double) (this.getY() + this.getHeight());
		}

		protected abstract int getInnerHeight();

		protected abstract void renderContents(int mouseX, int mouseY, float partialTick);

		protected int getInnerLeft() {
			return this.getX() + this.innerPadding();
		}

		protected int getInnerTop() {
			return this.getY() + this.innerPadding();
		}

		@Override
		public void playDownSound(C_5819357 handler) {
		}
	}

	public abstract static class AbstractScrollArea extends ClickableWidget {
		public static final int SCROLLBAR_WIDTH = 6;
		private double scrollAmount;
		private boolean scrolling;

		public AbstractScrollArea(int x, int y, int width, int height, String component) {
			super(x, y, width, height, component);
		}

		@Override
		public boolean mouseScrolled(double mouseX, double mouseY, double amountX, double scrollY) {
			if (!this.visible) {
				return false;
			} else {
				this.setScrollAmount(this.scrollAmount() - scrollY * this.scrollRate());
				return true;
			}
		}

		@Override
		public boolean mouseDragged(double mouseX, double mouseY, int button, double dragX, double dragY) {
			if (this.scrolling) {
				if (mouseY < (double) this.getY()) {
					this.setScrollAmount(0.0);
				} else if (mouseY > (double) this.getYEnd()) {
					this.setScrollAmount(this.maxScrollAmount());
				} else {
					double d = Math.max(1, this.maxScrollAmount());
					int i = this.scrollerHeight();
					double e = Math.max(1.0, d / (double) (this.getHeight() - i));
					this.setScrollAmount(this.scrollAmount() + dragY * e);
				}

				return true;
			} else {
				return super.mouseDragged(mouseX, mouseY, button, dragX, dragY);
			}
		}

		@Override
		public void onRelease(double mouseX, double mouseY) {
			this.scrolling = false;
		}

		public double scrollAmount() {
			return this.scrollAmount;
		}

		public void setScrollAmount(double scrollAmount) {
			this.scrollAmount = C_4976084.m_0987703(scrollAmount, 0.0, this.maxScrollAmount());
		}

		public boolean updateScrolling(double mouseX, double mouseY, int button) {
			this.scrolling = this.scrollbarVisible()
				&& this.isValidClickButton(button)
				&& mouseX >= (double) this.scrollBarX()
				&& mouseX <= (double) (this.scrollBarX() + 6)
				&& mouseY >= (double) this.getY()
				&& mouseY < (double) this.getYEnd();
			return this.scrolling;
		}

		public void refreshScrollAmount() {
			this.setScrollAmount(this.scrollAmount);
		}

		public int maxScrollAmount() {
			return Math.max(0, this.contentHeight() - this.getHeight());
		}

		protected boolean scrollbarVisible() {
			return this.maxScrollAmount() > 0;
		}

		protected int scrollerHeight() {
			return C_4976084.m_1109374((int) ((float) (this.getHeight() * this.getHeight()) / (float) this.contentHeight()), 32, this.getHeight() - 8);
		}

		protected int scrollBarX() {
			return this.getXEnd() - 6;
		}

		protected int scrollBarY() {
			return Math.max(this.getY(), (int) this.scrollAmount * (this.getHeight() - this.scrollerHeight()) / this.maxScrollAmount() + this.getY());
		}

		protected void renderScrollbar() {
			if (this.scrollbarVisible()) {
				int i = this.scrollBarX();
				int j = this.scrollerHeight();
				int k = this.scrollBarY();
				m_7865719(i, getY(), i + SCROLLBAR_WIDTH, getYEnd(), -16777216);
				m_7865719(i, k, i + SCROLLBAR_WIDTH, k + j, -8355712);
				m_7865719(i, k, i + SCROLLBAR_WIDTH - 1, k + j - 1, -4144960);
			}
		}

		protected int getYEnd() {
			return getY() + getHeight();
		}

		protected abstract int contentHeight();

		protected abstract double scrollRate();

		protected int getXEnd() {
			return getX() + getWidth();
		}
	}
}
