/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.api.chat;

import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.api.ChatsSidebar;
import io.github.axolotlclient.api.ContextMenu;
import io.github.axolotlclient.api.ContextMenuScreen;
import io.github.axolotlclient.api.handlers.ChatHandler;
import io.github.axolotlclient.api.requests.ChannelRequest;
import io.github.axolotlclient.api.types.Channel;
import io.github.axolotlclient.api.types.ChatMessage;
import io.github.axolotlclient.modules.auth.Auth;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import io.github.axolotlclient.util.ClientColors;
import lombok.Getter;
import net.minecraft.unmapped.C_1441789;
import net.minecraft.unmapped.C_1630332;
import net.minecraft.unmapped.C_1716360;
import net.minecraft.unmapped.C_3020744;
import net.minecraft.unmapped.C_3390001;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_6237110;
import net.minecraft.unmapped.C_8105098;

public class ChatWidget extends C_1630332 {

	private final List<ChatMessage> messages = new ArrayList<>();
	private final List<ChatLine> entries = new ArrayList<>();
	private final Channel channel;
	private final C_8105098 client;
	private final ContextMenuScreen screen;
	private int selectedEntry = -1;

	public ChatWidget(Channel channel, int x, int y, int width, int height, ContextMenuScreen screen) {
		super(C_8105098.m_0408063(), width, height, y, y + height, 13);
		this.channel = channel;
		this.client = C_8105098.m_0408063();
		super.m_3867153(x + 5);

		m_9693623(false, 0);
		this.screen = screen;
		channel.getMessages().forEach(this::addMessage);

		ChatHandler.getInstance().setMessagesConsumer(chatMessages -> chatMessages.forEach(this::addMessage));
		ChatHandler.getInstance().setMessageConsumer(this::addMessage);
		ChatHandler.getInstance().setEnableNotifications(message -> !message.channelId().equals(channel.getId()));

		m_9909453(m_6867946());
		f_3108331 = false;
	}

	@Override
	protected int m_3021017() {
		return f_8129586 + f_2695432 - 6;
	}

	@Override
	public int m_6519286() {
		return f_2695432 - 60;
	}

	public int getX() {
		return f_8129586;
	}

	protected boolean m_5993229(int i) {
		return i == selectedEntry;
	}

	private void addMessage(ChatMessage message) {
		List<String> list = client.f_0426313.m_0466830(message.content(), m_6519286());

		boolean scrollToBottom = m_0467656() == m_6867946();

		if (!messages.isEmpty()) {
			ChatMessage prev = messages.get(messages.size() - 1);
			if (!(prev.sender().equals(message.sender()) && prev.senderDisplayName().equals(message.senderDisplayName()))) {
				entries.add(new NameChatLine(message));
			} else {
				if (message.timestamp().getEpochSecond() - prev.timestamp().getEpochSecond() > 150) {
					entries.add(new NameChatLine(message));
				}
			}
		} else {
			entries.add(new NameChatLine(message));
		}

		list.forEach(t -> entries.add(new ChatLine(t, message)));
		messages.add(message);

		entries.sort(Comparator.comparingLong(c -> c.getOrigin().timestamp().getEpochSecond()));

		if (scrollToBottom) {
			m_9909453(m_6867946());
		}
		messages.sort(Comparator.comparingLong(value -> value.timestamp().getEpochSecond()));
	}

	private void loadMessages() {
		long before;
		if (!messages.isEmpty()) {
			before = messages.get(0).timestamp().getEpochSecond();
		} else {
			before = Instant.now().getEpochSecond();
		}
		ChatHandler.getInstance().getMessagesBefore(channel, before);
	}

	@Override
	public void m_9909453(int i) {
		this.f_1867063 += (float) i;
		if (f_1867063 < 0) {
			loadMessages();
		}
		this.m_7189774();
		this.f_9208105 = -2;
	}

	public void remove() {
		ChatHandler.getInstance().setMessagesConsumer(ChatHandler.DEFAULT_MESSAGES_CONSUMER);
		ChatHandler.getInstance().setMessageConsumer(ChatHandler.DEFAULT_MESSAGE_CONSUMER);
		ChatHandler.getInstance().setEnableNotifications(ChatHandler.DEFAULT);
	}

	@Override
	protected void m_2972483(int i, int j, int k, int l) {
		DrawUtil.enableScissor(f_8129586, f_9975269, f_7901480, f_9145833);
		super.m_2972483(i, j, k, l);
		DrawUtil.disableScissor();
	}

	@Override
	public void m_9734698(int mouseX, int mouseY, float delta) {
		if (screen.getSelf() instanceof ChatsSidebar) {
			if (this.f_1810131) {
				this.f_9602731 = mouseX;
				this.f_1139530 = mouseY;
				this.m_7189774();
				int m = this.f_8129586 + this.f_2695432 / 2 - this.m_6519286() / 2 + 2;
				int n = this.f_9975269 + 4 - (int) this.f_1867063;

				this.m_2972483(m, n, mouseX, mouseY);
			}
		} else {
			super.m_9734698(mouseX, mouseY, delta);
		}
	}

	@Override
	public C_8277969 m_7118949(int i) {
		return entries.get(i);
	}

	@Override
	protected int m_3791031() {
		return entries.size();
	}

	public class ChatLine extends DrawUtil implements C_8277969 {
		protected final C_8105098 client = C_8105098.m_0408063();
		@Getter
		private final String content;
		@Getter
		private final ChatMessage origin;

		public ChatLine(String content, ChatMessage origin) {
			super();
			this.content = content;
			this.origin = origin;
		}

		@Override
		public void m_1433237(int i, int j, int k) {

		}

		@Override
		public boolean m_3739705(int index, int mouseX, int mouseY, int button, int x, int y) {
			if (button == 0) {
				ChatWidget.this.selectedEntry = index;
				return true;
			}
			if (button == 1) {
				ContextMenu.Builder builder = ContextMenu.builder()
					.title(origin.sender().getName())
					.spacer();
				if (!origin.sender().equals(API.getInstance().getSelf())) {
					builder.entry("api.friends.chat", buttonWidget -> {
							ChannelRequest.getOrCreateDM(origin.sender())
								.whenCompleteAsync((channel, throwable) -> client.m_2167033(() -> client.m_6408915(new ChatScreen(screen.getParent(), channel))));
						})
						.spacer();
				}
				builder.entry("api.chat.report.message", buttonWidget -> {
						C_3020744 previous = client.f_0723335;
						client.m_6408915(new C_1441789((b, i) -> {
							if (b) {
								ChatHandler.getInstance().reportMessage(origin);
							}
							client.m_6408915(previous);
						}, C_3390001.m_2053009("api.channels.confirm_report"),
							C_3390001.m_2053009("api.channels.confirm_report.desc", origin.content()), 0));
					})
					.spacer()
					.entry("action.copy", buttonWidget -> {
						C_3020744.m_5360526(origin.content());
					});
				screen.setContextMenu(builder.build());
				return true;
			}
			return false;
		}

		@Override
		public void m_4703785(int i, int j, int k, int l, int m, int n) {

		}

		protected void renderExtras(int x, int y, int mouseX, int mouseY) {
		}

		@Override
		public void m_2013749(int index, int x, int y, int entryWidth, int entryHeight, int mouseX, int mouseY, boolean hovered) {
			for (int i = 0; i < entries.size(); i++) {
				ChatLine l = entries.get(i);
				if (l.getOrigin().equals(origin)) {
					if (m_3816410(mouseX, mouseY) == i) {
						hovered = true;
						break;
					}
				}
			}
			if (hovered && !screen.hasContextMenu()) {
				m_7865719(x - 2 - 22, y - 2, x + entryWidth + 20, y + entryHeight - 1, 0x33FFFFFF);
				if (index < entries.size() - 1 && entries.get(index + 1).getOrigin().equals(origin)) {
					m_7865719(x - 2 - 22, y + entryHeight - 1, x + entryWidth + 20, y + entryHeight + 2, 0x33FFFFFF);
				}
				if ((index < entries.size() - 1 && !entries.get(index + 1).getOrigin().equals(origin)) || index == entries.size() - 1) {
					m_7865719(x - 2 - 22, y + entryHeight - 1, x + entryWidth + 20, y + entryHeight, 0x33FFFFFF);
				}
			}
			renderExtras(x, y, mouseX, mouseY);
			client.f_0426313.m_4683674(content, x, y, -1);
		}
	}

	public class NameChatLine extends ChatLine {

		private final String formattedTime;

		public NameChatLine(ChatMessage message) {
			super(new C_1716360(message.senderDisplayName())
				.m_7551367(new C_6237110().m_5934684(true)).m_2117196(), message);

			DateTimeFormatter DATE_FORMAT = DateTimeFormatter.ofPattern("d/M/yyyy H:mm");
			formattedTime = DATE_FORMAT.format(message.timestamp().atZone(ZoneId.systemDefault()));
		}

		@Override
		protected void renderExtras(int x, int y, int mouseX, int mouseY) {
			C_3754158.m_5313301();
			C_3754158.m_7547086();
			client.m_1218956().m_5325521(Auth.getInstance().getSkinTexture(getOrigin().sender().getUuid(),
				getOrigin().sender().getName()));
			m_4673442(x - 22, y, 8, 8, 8, 8, 18, 18, 64, 64);
			m_4673442(x - 22, y, 40, 8, 8, 8, 18, 18, 64, 64);
			C_3754158.m_9671730();
			client.f_0426313.m_4683674(formattedTime, client.f_0426313.m_0040387(getContent()) + x + 5, y, ClientColors.GRAY.toInt());
		}
	}
}
