/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.api.multiplayer;

import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.stream.Collectors;

import com.google.common.hash.Hashing;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Color;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.vanilla.EntryListWidget;
import io.github.axolotlclient.api.types.PkSystem;
import io.github.axolotlclient.api.types.Status;
import io.github.axolotlclient.api.types.User;
import io.github.axolotlclient.modules.auth.Auth;
import io.github.axolotlclient.util.Util;
import io.netty.buffer.ByteBufInputStream;
import io.netty.buffer.Unpooled;
import io.netty.handler.codec.base64.Base64;
import lombok.Getter;
import lombok.Setter;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.unmapped.C_0561170;
import net.minecraft.unmapped.C_0979881;
import net.minecraft.unmapped.C_1533205;
import net.minecraft.unmapped.C_1945050;
import net.minecraft.unmapped.C_2691939;
import net.minecraft.unmapped.C_3390001;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_3779718;
import net.minecraft.unmapped.C_8105098;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.Nullable;

public class FriendsMultiplayerSelectionList extends EntryListWidget<FriendsMultiplayerSelectionList.Entry> {
	private static final int PROTOCOL_VERSION = 47;
	private static final C_0561170 UNKNOWN_SERVER_TEXTURE = new C_0561170("textures/misc/unknown_server.png");
	static final C_0561170 SERVER_SELECTION_TEXTURE = new C_0561170("textures/gui/server_selection.png");
	static final C_0561170 ICONS = new C_0561170("textures/gui/icons.png");
	static final Logger LOGGER = LogManager.getLogger();
	static final ThreadPoolExecutor THREAD_POOL = new ScheduledThreadPoolExecutor(
		5,
		new ThreadFactoryBuilder()
			.setNameFormat("Friends Server Pinger #%d")
			.setDaemon(true)
			.setUncaughtExceptionHandler((t, e) -> LOGGER.error("Caught previously unhandled exception :", e)).build()
	);
	static final String CANT_RESOLVE_TEXT = Util.getFormatCode(new Color(-65536)) + C_3390001.m_2053009("multiplayer.status.cannot_resolve");
	static final String CANT_CONNECT_TEXT = Util.getFormatCode(new Color(-65536)) + C_3390001.m_2053009("multiplayer.status.cannot_connect");
	static final String INCOMPATIBLE_STATUS = C_3390001.m_2053009("multiplayer.status.incompatible");
	static final String NO_CONNECTION_STATUS = C_3390001.m_2053009("multiplayer.status.no_connection");
	static final String PINGING_STATUS = C_3390001.m_2053009("multiplayer.status.pinging");
	static final String NOT_PUBLISHED_STATUS = C_1945050.f_9213701 + C_3390001.m_2053009("api.worldhost.joinability.not_published");
	private final FriendsMultiplayerScreen screen;
	private final List<io.github.axolotlclient.api.multiplayer.FriendsMultiplayerSelectionList.Entry> friendEntries = new ArrayList<>();
	private final LoadingHeader loadingHeader = new LoadingHeader();

	public FriendsMultiplayerSelectionList(FriendsMultiplayerScreen screen, C_8105098 minecraft, int width, int height, int y, int itemHeight) {
		super(minecraft, screen.f_5465691, screen.f_3080061, y, y + height, itemHeight);
		this.screen = screen;
		addEntry(loadingHeader);
	}

	private void refreshEntries() {
		this.clearEntries();
		this.friendEntries.forEach(this::addEntry);
	}

	public void setSelected(@Nullable FriendsMultiplayerSelectionList.Entry entry) {
		super.setSelected(entry);
		this.screen.onSelectedChange();
	}

	@Override
	public boolean keyPressed(int keyCode, int scanCode, int modifiers) {
		io.github.axolotlclient.api.multiplayer.FriendsMultiplayerSelectionList.Entry entry = this.getSelectedOrNull();
		return entry != null && entry.keyPressed(keyCode, scanCode, modifiers) || super.keyPressed(keyCode, scanCode, modifiers);
	}

	public void updateList(List<User> friends) {
		this.friendEntries.clear();

		for (User friend : friends) {
			if (friend.getStatus().isOnline()) {
				this.friendEntries.add(createEntry(friend));
			}
		}

		this.refreshEntries();
	}

	private io.github.axolotlclient.api.multiplayer.FriendsMultiplayerSelectionList.Entry createEntry(User friend) {
		if (friend.getStatus().getActivity() != null && friend.getStatus().getActivity().hasMetadata()) {
			if (friend.getStatus().getActivity().hasMetadata(Status.Activity.ExternalServerMetadata.ID)) {
				return externalServerEntry(this.screen, friend);
			} else {
				return e4mcServerFriendEntry(this.screen, friend);
			}
		}
		return new StatusFriendEntry(screen, friend);
	}

	public void updateEntry(User user) {
		this.friendEntries.stream().filter(e1 -> {
			if (e1 instanceof StatusFriendEntry statusFriendEntry) {
				return statusFriendEntry.getUser().equals(user);
			} else if (e1 instanceof ServerEntry serverEntry) {
				return serverEntry.getUser().equals(user);
			}
			return false;
		}).findFirst().ifPresent(e -> {
			this.friendEntries.set(friendEntries.indexOf(e), createEntry(user));
			refreshEntries();
		});
	}

	@Override
	public int getRowWidth() {
		return 305;
	}

	@Environment(EnvType.CLIENT)
	public abstract static class Entry extends EntryListWidget.Entry<io.github.axolotlclient.api.multiplayer.FriendsMultiplayerSelectionList.Entry> implements AutoCloseable {
		private boolean focused;

		public void close() {
		}

		public boolean canJoin() {
			return false;
		}

		public C_1533205 getServerData() {
			return null;
		}

		@Override
		public boolean isFocused() {
			return focused;
		}

		@Override
		public void setFocused(boolean b) {
			this.focused = b;
		}
	}

	@Getter
	public class StatusFriendEntry extends io.github.axolotlclient.api.multiplayer.FriendsMultiplayerSelectionList.Entry {

		private final User user;

		protected StatusFriendEntry(final FriendsMultiplayerScreen screen, final User friend) {
			this.user = friend;
		}

		@Override
		public void render(int index, int top, int left, int width, int height, int mouseX, int mouseY, boolean hovering, float partialTick) {
			if (user.isSystem()) {
				String fronters =
					user.getSystem().getFronters().stream().map(PkSystem.Member::getDisplayName)
						.collect(Collectors.joining("/"));
				String tag = C_1945050.f_8012292 + C_1945050.f_7540461.toString() + "(" + user.getSystem().getName() + "/" + user.getName() + ")" + C_1945050.f_1149207;
				client.f_0426313.m_4683674(fronters + " " + tag, left + 3, top + 1, -1);
			} else {
				client.f_0426313.m_4683674(user.getName(), left + 3 + 32, top + 1, -1);
			}

			if (user.getStatus().isOnline() && user.getStatus().getActivity() != null) {
				client.f_0426313.m_4683674(user.getStatus().getTitle(), left + 3 + 32, top + 12, 8421504);
				client.f_0426313.m_4683674(user.getStatus().getDescription(), left + 3 + 40, top + 23, 8421504);
			} else if (user.getStatus().getLastOnline() != null) {
				client.f_0426313.m_4683674(user.getStatus().getLastOnline(), left + 3 + 32, top + 12, 8421504);
			}

			client.m_1218956().m_5325521(Auth.getInstance().getSkinTexture(user));
			C_3754158.m_9671730();
			C_3754158.m_3323122(1, 1, 1, 1);
			m_4673442(left - 1, top - 1, 8, 8, 8, 8, 32, 32, 64, 64);
			m_4673442(left - 1, top - 1, 40, 8, 8, 8, 32, 32, 64, 64);
			C_3754158.m_5313301();
		}
	}

	private static final int STATUS_ICON_HEIGHT = 8;
	private static final int STATUS_ICON_WIDTH = 10;

	protected class ServerEntry extends io.github.axolotlclient.api.multiplayer.FriendsMultiplayerSelectionList.Entry {
		private static final int ICON_WIDTH = 32;
		private static final int ICON_HEIGHT = 32;
		private static final int SPACING = 5;
		private static final int STATUS_ICON_WIDTH = 10;
		private static final int STATUS_ICON_HEIGHT = 8;
		private final FriendsMultiplayerScreen screen;
		private final C_8105098 minecraft;
		protected final ServerInfoEx serverData;
		private final C_0561170 iconId;
		private C_3779718 icon;
		private String lastIconBytes;
		private long lastClickTime;
		@Nullable
		private List<String> onlinePlayersTooltip;
		private Sprite statusIcon;
		@Nullable
		private String statusIconTooltip;
		@Getter
		protected final User user;

		@SuppressWarnings("UnstableApiUsage")
		protected ServerEntry(FriendsMultiplayerScreen screen, C_1533205 serverData, User user) {
			this.screen = screen;
			this.minecraft = C_8105098.m_0408063();
			this.serverData = new ServerInfoEx(serverData);
			this.iconId = new C_0561170("servers/" + Hashing.sha1().hashUnencodedChars(serverData.f_7619058 != null ? serverData.f_7619058 : user.getUuid() + "_" + serverData.f_4296074) + "/icon");
			this.user = user;
		}

		@Override
		public C_1533205 getServerData() {
			return serverData.serverInfo();
		}

		protected void refreshStatus() {
			this.onlinePlayersTooltip = null;
			if (!isPublished()) {
				this.serverData.setPingResult(PingResult.UNREACHABLE);
			}
			switch (this.serverData.pingResult()) {
				case INITIAL:
				case PINGING:
					this.statusIcon = Sprite.PING_1_SPRITE;
					this.statusIconTooltip = FriendsMultiplayerSelectionList.PINGING_STATUS;
					break;
				case INCOMPATIBLE:
					this.statusIcon = Sprite.INCOMPATIBLE_SPRITE;
					var tooltipString = this.serverData.serverInfo.f_2418160;
					if (tooltipString != null) {
						this.onlinePlayersTooltip = List.of(tooltipString.split("\n"));
					} else {
						this.onlinePlayersTooltip = null;
					}
					this.statusIconTooltip = FriendsMultiplayerSelectionList.INCOMPATIBLE_STATUS;
					break;
				case UNREACHABLE:
					this.statusIcon = Sprite.UNREACHABLE_SPRITE;
					if (!isPublished()) {
						break;
					}
					this.statusIconTooltip = FriendsMultiplayerSelectionList.NO_CONNECTION_STATUS;
					break;
				case SUCCESSFUL:
					if (this.serverData.serverInfo.f_0578448 < 150L) {
						this.statusIcon = Sprite.PING_5_SPRITE;
					} else if (this.serverData.serverInfo.f_0578448 < 300L) {
						this.statusIcon = Sprite.PING_4_SPRITE;
					} else if (this.serverData.serverInfo.f_0578448 < 600L) {
						this.statusIcon = Sprite.PING_3_SPRITE;
					} else if (this.serverData.serverInfo.f_0578448 < 1000L) {
						this.statusIcon = Sprite.PING_2_SPRITE;
					} else {
						this.statusIcon = Sprite.PING_1_SPRITE;
					}

					this.statusIconTooltip = C_3390001.m_2053009("multiplayer.status.ping", this.serverData.serverInfo.f_0578448);
					var tooltipStr = this.serverData.serverInfo.f_2418160;
					if (tooltipStr != null) {
						this.onlinePlayersTooltip = List.of(tooltipStr.split("\n"));
					} else {
						this.onlinePlayersTooltip = null;
					}
			}
		}

		@Override
		public void render(int index, int top, int left, int width, int height, int mouseX, int mouseY, boolean hovering, float partialTick) {
			if (this.serverData.pingResult() == PingResult.INITIAL) {
				this.serverData.setPingResult(PingResult.PINGING);
				this.serverData.serverInfo.f_2449978 = "";
				this.serverData.serverInfo.f_8782169 = "";
				FriendsMultiplayerSelectionList.THREAD_POOL
					.submit(
						() -> {
							try {
								this.screen
									.getPinger()
									.add(
										this.serverData.serverInfo
									);
							} catch (UnknownHostException var2) {
								this.serverData.setPingResult(PingResult.UNREACHABLE);
								this.serverData.serverInfo.f_2449978 = FriendsMultiplayerSelectionList.CANT_RESOLVE_TEXT;
								this.minecraft.m_2167033(this::refreshStatus);
							} catch (Exception var3) {
								this.serverData.setPingResult(PingResult.UNREACHABLE);
								this.serverData.serverInfo.f_2449978 = FriendsMultiplayerSelectionList.CANT_CONNECT_TEXT;
								this.minecraft.m_2167033(this::refreshStatus);
							}
						}
					);
			}

			if (serverData.pingResult == PingResult.PINGING && serverData.serverInfo.f_0578448 != -2) {
				this.serverData.setPingResult(
					this.serverData.serverInfo.f_9567457 == PROTOCOL_VERSION ? PingResult.SUCCESSFUL : PingResult.INCOMPATIBLE
				);
			}
			refreshStatus();

			minecraft.f_0426313.m_1950885(this.serverData.serverInfo.f_4296074, left + ICON_WIDTH + 3, top + 1, -1);
			List<String> list = this.minecraft.f_0426313.m_0466830(this.serverData.serverInfo.f_2449978, width - ICON_WIDTH - 2);

			for (int i = 0; i < Math.min(list.size(), 2); i++) {
				minecraft.f_0426313.m_1950885(list.get(i), left + ICON_WIDTH + 3, top + 12 + 9 * i, -8355712);
			}

			C_3754158.m_3323122(1, 1, 1, 1);
			client.m_1218956().m_5325521(this.icon != null ? this.iconId : UNKNOWN_SERVER_TEXTURE);
			m_5935491(left, top, 0.0F, 0.0F, 32, 32, 32, 32);
			client.m_1218956().m_5325521(Auth.getInstance().getSkinTexture(user));
			C_3754158.m_9671730();
			m_4673442(left + ICON_WIDTH - 10, top + ICON_HEIGHT - 10, 8, 8, 8, 8, 10, 10, 64, 64);
			m_4673442(left + ICON_WIDTH - 10, top + ICON_HEIGHT - 10, 40, 8, 8, 8, 10, 10, 64, 64);
			C_3754158.m_5313301();
			if (this.serverData.pingResult() == PingResult.PINGING) {
				int i = (int) (C_8105098.m_6224581() / 100L + index * 2 & 7L);
				if (i > 4) {
					i = 8 - i;
				}
				this.statusIcon = switch (i) {
					case 1 -> Sprite.PINGING_2_SPRITE;
					case 2 -> Sprite.PINGING_3_SPRITE;
					case 3 -> Sprite.PINGING_4_SPRITE;
					case 4 -> Sprite.PINGING_5_SPRITE;
					default -> Sprite.PINGING_1_SPRITE;
				};
			}

			int i = left + width - STATUS_ICON_WIDTH - SPACING;
			if (this.statusIcon != null) {
				statusIcon.draw(i, top);
			}

			String bs = this.serverData.serverInfo.m_6218080();
			if (!Objects.equals(bs, this.lastIconBytes)) {
				if (this.uploadIcon(bs)) {
					this.lastIconBytes = bs;
				} else {
					this.serverData.serverInfo.m_1622917(null);
				}
			}

			String component;
			if (!isPublished()) {
				component = NOT_PUBLISHED_STATUS;
			} else {
				if (this.serverData.pingResult() == PingResult.INCOMPATIBLE) {
					component = C_1945050.f_9213701 + this.serverData.serverInfo.f_3146804 + C_1945050.f_1149207;
				} else {
					component = this.serverData.serverInfo.f_8782169;
				}
			}
			int j = this.minecraft.f_0426313.m_0040387(component);
			int k = i - j - SPACING;
			minecraft.f_0426313.m_1950885(component, k, top + 1, -8355712);
			if (this.statusIconTooltip != null && mouseX >= i && mouseX <= i + STATUS_ICON_WIDTH && mouseY >= top && mouseY <= top + STATUS_ICON_HEIGHT) {
				this.screen.setDeferredTooltip(this.statusIconTooltip);
			} else if (this.onlinePlayersTooltip != null && mouseX >= k && mouseX <= k + j && mouseY >= top && mouseY <= top - 1 + 9) {
				this.screen.setDeferredTooltip(this.onlinePlayersTooltip);
			}

			if (this.minecraft.f_9967940.f_6270799 || hovering) {
				int l = mouseX - left;
				int m = mouseY - top;
				if (this.canJoin()) {
					C_2691939.m_7865719(left, top, left + ICON_WIDTH, top + ICON_HEIGHT, -1601138544);
					if (l < ICON_WIDTH && l > ICON_WIDTH / 2) {
						Sprite.JOIN_HIGHLIGHTED_SPRITE.draw(left, top);
					} else {
						Sprite.JOIN_SPRITE.draw(left, top);
					}
				}
			}
		}

		protected boolean isPublished() {
			return true;
		}

		@Override
		public boolean canJoin() {
			return serverData.pingResult() == PingResult.SUCCESSFUL && isPublished();
		}

		private boolean uploadIcon(String iconBytes) {
			if (iconBytes == null) {
				this.minecraft.m_1218956().m_3775266(this.iconId);
				this.icon = null;
			} else {
				var buf = Unpooled.copiedBuffer(iconBytes, StandardCharsets.UTF_8);
				var decoded = Base64.decode(buf);
				try {

					var image = C_0979881.m_2641486(new ByteBufInputStream(decoded));
					if (image.getWidth() != 64 || image.getHeight() != 64) {
						throw new IllegalArgumentException("Icon must be 64x64, but was " + image.getWidth() + "x" + image.getHeight());
					}
					if (this.icon == null) {
						this.icon = new C_3779718(image);
						this.minecraft.m_1218956().m_0381205(this.iconId, this.icon);
					}

					image.getRGB(0, 0, image.getWidth(), image.getHeight(), this.icon.m_1823577(), 0, image.getWidth());
					this.icon.m_4420833();

				} catch (Throwable var3) {
					FriendsMultiplayerSelectionList.LOGGER.error("Invalid icon for server {} ({})", this.serverData.serverInfo.f_4296074, this.serverData.serverInfo.f_7619058, var3);
					return false;
				} finally {
					buf.release();
					decoded.release();
				}
			}

			return true;
		}

		@Override
		public boolean mouseClicked(double mouseX, double mouseY, int button) {
			double d = mouseX - FriendsMultiplayerSelectionList.this.getRowLeft();
			double e = mouseY - FriendsMultiplayerSelectionList.this.getRowTop(FriendsMultiplayerSelectionList.this.children().indexOf(this));
			if (d <= 32.0) {
				if (d < 32.0 && d > 16.0 && this.canJoin()) {
					this.screen.setSelected(this);
					this.screen.joinSelectedServer();
					return true;
				}
			}

			this.screen.setSelected(this);
			if (C_8105098.m_6224581() - this.lastClickTime < 250L && canJoin()) {
				this.screen.joinSelectedServer();
			}

			this.lastClickTime = C_8105098.m_6224581();
			return super.mouseClicked(mouseX, mouseY, button);
		}

		@Override
		public void close() {
			this.minecraft.m_1218956().m_3775266(this.iconId);
		}
	}

	private ExternalServerFriendEntry externalServerEntry(FriendsMultiplayerScreen screen, User friend) {
		Status.Activity.ExternalServerMetadata metadata = (Status.Activity.ExternalServerMetadata) friend.getStatus().getActivity().metadata().attributes();
		return new ExternalServerFriendEntry(screen, metadata, new C_1533205(metadata.serverName(), metadata.address(), false), friend);
	}

	public class ExternalServerFriendEntry extends ServerEntry {
		private final Status.Activity.ExternalServerMetadata statusDescription;

		private ExternalServerFriendEntry(FriendsMultiplayerScreen screen, Status.Activity.ExternalServerMetadata statusDescription, C_1533205 serverData, User friend) {
			super(screen, serverData, friend);
			this.statusDescription = statusDescription;
			refreshStatus();
		}

		@Override
		public boolean canJoin() {
			return statusDescription.address() != null;
		}

	}

	private E4mcServerFriendEntry e4mcServerFriendEntry(FriendsMultiplayerScreen screen, User friend) {
		var activity = friend.getStatus().getActivity();
		Status.Activity.E4mcMetadata metadata;
		if (activity.hasMetadata(Status.Activity.WorldHostMetadata.ID)) {
			metadata = ((Status.Activity.WorldHostMetadata) activity.metadata().attributes()).asE4mcMetadata();
		} else {
			metadata = (Status.Activity.E4mcMetadata) activity.metadata().attributes();
		}
		return new E4mcServerFriendEntry(screen, metadata, ServerInfoUtil.getServerData(friend.getName(), metadata), friend);
	}

	public class E4mcServerFriendEntry extends ServerEntry {

		private final Status.Activity.E4mcMetadata statusDescription;

		protected E4mcServerFriendEntry(FriendsMultiplayerScreen screen, Status.Activity.E4mcMetadata statusDescription, C_1533205 serverData, User friend) {
			super(screen, serverData, friend);
			this.statusDescription = statusDescription;
			refreshStatus();
		}

		@Override
		protected boolean isPublished() {
			return statusDescription.domain() != null;
		}

		@Override
		protected void refreshStatus() {
			super.refreshStatus();
			serverData.serverInfo.f_2449978 = statusDescription.serverInfo().levelName();
		}
	}

	@Environment(EnvType.CLIENT)
	public static class LoadingHeader extends io.github.axolotlclient.api.multiplayer.FriendsMultiplayerSelectionList.Entry {
		private final C_8105098 minecraft = C_8105098.m_0408063();

		@Override
		public void render(int index, int top, int left, int width, int height, int mouseX, int mouseY, boolean hovering, float partialTick) {
			int i = top + height / 2 - 9 / 2;

			String string = switch ((int) (C_8105098.m_6224581() / 300L % 4L)) {
				case 1, 3 -> "o O o";
				case 2 -> "o o O";
				default -> "O o o";
			};
			minecraft.f_0426313.m_1950885(string, this.minecraft.f_0723335.f_5465691 / 2f - this.minecraft.f_0426313.m_0040387(string) / 2f, i, -8355712);
		}
	}

	protected static final class ServerInfoEx {
		private final C_1533205 serverInfo;
		@Setter
		private PingResult pingResult;

		private ServerInfoEx(C_1533205 serverInfo) {
			this.serverInfo = serverInfo;
			this.pingResult = PingResult.INITIAL;
		}

		public C_1533205 serverInfo() {
			return serverInfo;
		}

		public PingResult pingResult() {
			return pingResult;
		}

	}

	protected enum PingResult {
		INITIAL,
		PINGING,
		UNREACHABLE,
		INCOMPATIBLE,
		SUCCESSFUL
	}

	private enum Sprite {
		UNREACHABLE_SPRITE(0, 5),
		INCOMPATIBLE_SPRITE(0, 5),
		PING_1_SPRITE(0, 4),
		PING_2_SPRITE(0, 3),
		PING_3_SPRITE(0, 2),
		PING_4_SPRITE(0, 3),
		PING_5_SPRITE(0, 0),
		PINGING_1_SPRITE(10, 0),
		PINGING_2_SPRITE(10, 1),
		PINGING_3_SPRITE(10, 2),
		PINGING_4_SPRITE(10, 3),
		PINGING_5_SPRITE(10, 4),
		JOIN_HIGHLIGHTED_SPRITE(0, 32, 32, 32, SERVER_SELECTION_TEXTURE),
		JOIN_SPRITE(0, 0, 32, 32, SERVER_SELECTION_TEXTURE),

		;
		private final int u, v, texWidth, texHeight;
		private final C_0561170 atlas;

		Sprite(int u, int v, int texWidth, int texHeight, C_0561170 texture) {
			this.u = u;
			this.v = v;
			this.texWidth = texWidth;
			this.texHeight = texHeight;
			this.atlas = texture;
		}

		Sprite(int u, int v) {
			this(u, 176 + v * 8, STATUS_ICON_WIDTH, STATUS_ICON_HEIGHT, ICONS);
		}

		public void draw(int x, int y) {
			C_3754158.m_3323122(1, 1, 1, 1);
			C_8105098.m_0408063().m_1218956().m_5325521(atlas);
			C_2691939.m_5935491(x, y, u, v, texWidth, texHeight, 256, 256);
		}

	}
}
