/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.bridge.impl.commands;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.Preconditions;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.exceptions.CommandExceptionType;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import io.github.axolotlclient.bridge.commands.AxoClientCmdSrcStack;
import io.github.axolotlclient.bridge.commands.Commands;
import io.github.axolotlclient.bridge.util.AxoText;
import lombok.Getter;
import net.minecraft.unmapped.C_1160932;
import net.minecraft.unmapped.C_1716360;
import net.minecraft.unmapped.C_1730529;
import net.minecraft.unmapped.C_8105098;
import net.minecraft.unmapped.C_9550253;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class CommandsImpl implements Commands {
	public record SourceStack(C_1730529 origin) implements AxoClientCmdSrcStack {
		@Override
		public void br$sendError(AxoText text) {
			origin.m_3031403((C_9550253) text);
		}

		@Override
		public void br$sendFeedback(AxoText text) {
			origin.m_3031403((C_9550253) text);
		}
	}

	@Getter
	private static final CommandsImpl instance = new CommandsImpl();
	@Getter
	private final CommandDispatcher<AxoClientCmdSrcStack> dispatcher = new CommandDispatcher<>();
	private final Logger logger = LogManager.getLogger("ClientCommandHandler");

	private static boolean isIgnoredException(CommandExceptionType type) {
		return type == CommandSyntaxException.BUILT_IN_EXCEPTIONS.dispatcherUnknownCommand() ||
			type == CommandSyntaxException.BUILT_IN_EXCEPTIONS.dispatcherParseException();
	}

	private static C_9550253 getErrorMessage(CommandSyntaxException e) {
		String context = e.getContext();
		return context != null ?
			new C_1160932("lcu.command.parse_error", e.getMessage(), e.getCursor(), context) : new C_1716360(e.getMessage());
	}

	private static SourceStack buildClientSource(C_8105098 client) {
		Preconditions.checkState(client.f_7663840 != null);
		return new SourceStack(client.f_7663840);
	}

	public boolean dispatchClient(String command) {
		if (!command.startsWith("/")) {
			return false;
		}

		C_8105098 client = C_8105098.m_0408063();
		final var source = buildClientSource(client);
		// cancel if present
		command = command.trim().substring(1);

		try {
			dispatcher.execute(command, source);
			return true;
		} catch (CommandSyntaxException e) {
			if (isIgnoredException(e.getType())) {
				return false;
			}

			logger.warn("Syntax exception for command '{}'", command, e);
			source.origin().m_3031403(getErrorMessage(e));
			return true;
		} catch (Exception e) {
			logger.warn("Error while executing command '{}'", command, e);
			source.origin().m_3031403(new C_1716360(e.getMessage() == null ? "" : e.getMessage()));
			return true;
		}
	}

	public CompletableFuture<List<String>> getCompletionsClient(String command) {
		C_8105098 client = C_8105098.m_0408063();

		String command0 = command.startsWith("/") ? command.substring(1) : command;
		return dispatcher.getCompletionSuggestions(dispatcher.parse(command0, buildClientSource(client)))
			.thenApply(suggestions -> suggestions.getList()
				.stream()
				.map(x -> command0.contains(" ") ? x.getText() : "/" + x.getText())
				.toList()
			);
	}
}
