/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.config;

import java.util.ArrayList;
import java.util.List;

import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientCommon;
import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.api.options.OptionCategory;
import io.github.axolotlclient.AxolotlClientConfig.api.ui.ConfigUI;
import io.github.axolotlclient.AxolotlClientConfig.api.ui.screen.ConfigScreen;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Color;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.*;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.RecreatableScreen;
import io.github.axolotlclient.AxolotlClientConfig.impl.util.ConfigStyles;
import io.github.axolotlclient.AxolotlClientConfigCommon;
import io.github.axolotlclient.config.screen.CreditsScreen;
import io.github.axolotlclient.config.screen.ProfilesScreen;
import io.github.axolotlclient.modules.Module;
import io.github.axolotlclient.util.WindowAccess;
import io.github.axolotlclient.util.options.ForceableBooleanOption;
import io.github.axolotlclient.util.options.GenericOption;
import lombok.Getter;
import net.minecraft.unmapped.C_3020744;
import net.minecraft.unmapped.C_7778778;
import net.minecraft.unmapped.C_8105098;
import net.ornithemc.osl.keybinds.api.KeyBindingEvents;
import net.ornithemc.osl.lifecycle.api.client.MinecraftClientEvents;

public class AxolotlClientConfig extends AxolotlClientConfigCommon {

	public final BooleanOption showOwnNametag = new BooleanOption("showOwnNametag", false);
	public final BooleanOption useShadows = new BooleanOption("useShadows", false);
	public final BooleanOption nametagBackground = new BooleanOption("nametagBackground", true);

	public final BooleanOption showBadges = new BooleanOption("showBadges", true);
	public final BooleanOption customBadge = new BooleanOption("customBadge", false);
	public final StringOption badgeText = new StringOption("badgeText", "");

	public final ForceableBooleanOption timeChangerEnabled = new ForceableBooleanOption("enabled", false);
	public final IntegerOption customTime = new IntegerOption("time", 0, 0, 24000);
	public final BooleanOption customSky = new BooleanOption("customSky", true);
	public final IntegerOption cloudHeight = new IntegerOption("cloudHeight", 128, 100, 512);
	public final BooleanOption dynamicFOV = new BooleanOption("dynamicFov", true);
	public final ForceableBooleanOption fullBright = new ForceableBooleanOption("fullBright", false);
	public final BooleanOption removeVignette = new BooleanOption("removeVignette", false);
	public final ForceableBooleanOption lowFire = new ForceableBooleanOption("lowFire", false);
	public final ColorOption hitColor = new ColorOption("hitColor", new Color(255, 0, 0, 77));
	public final BooleanOption minimalViewBob = new BooleanOption("minimalViewBob", false);
	public final BooleanOption noHurtCam = new BooleanOption("noHurtCam", false);
	public final BooleanOption flatItems = new BooleanOption("flatItems", false);
	public final BooleanOption inventoryPotionEffectOffset = new BooleanOption("inventory.potion_effect_offset", true);

	public final ColorOption loadingScreenColor = new ColorOption("loadingBgColor", new Color(-1));
	public final BooleanOption nightMode = new BooleanOption("nightMode", false);
	public final BooleanOption rawMouseInput = new BooleanOption("rawMouseInput", false, v ->
		WindowAccess.getInstance().setRawMouseMotion(v));

	public final BooleanOption enableCustomOutlines = new BooleanOption("enabled", false);
	public final ColorOption outlineColor = new ColorOption("color", Color.parse("#DD000000"));
	public final IntegerOption outlineWidth = new IntegerOption("outlineWidth", 1, 1, 10);

	public final BooleanOption noRain = new BooleanOption("noRain", false);

	public final BooleanOption debugLogOutput = new BooleanOption("debugLogOutput", false);
	public final GenericOption openCredits = new GenericOption("Credits", "Open Credits",
		() -> C_8105098.m_0408063()
			.m_6408915(new CreditsScreen(C_8105098.m_0408063().f_0723335)));
	public final BooleanOption creditsBGM = new BooleanOption("creditsBGM", true);
	public final BooleanOption customWindowTitle = new BooleanOption("customWindowTitle", true);

	public final BooleanOption scaleTitles = new BooleanOption("titles.scaling", false);
	public final IntegerOption titlePadding = new IntegerOption("titles.padding", 4, 1, 10);

	public final OptionCategory general = OptionCategory.create("general");
	public final OptionCategory nametagOptions = OptionCategory.create("nametagOptions");
	public final OptionCategory rendering = OptionCategory.create("rendering");
	public final OptionCategory outlines = OptionCategory.create("blockOutlines");
	public final OptionCategory timeChanger = OptionCategory.create("timeChanger");
	public final OptionCategory titles = OptionCategory.create("titles");

	@Getter
	private final List<Option<?>> options = new ArrayList<>();

	public AxolotlClientConfig() {
		config.add(general);
		config.add(nametagOptions);
		config.add(rendering);
		config.add(hidden);

		nametagOptions.add(showOwnNametag);
		nametagOptions.add(useShadows);
		nametagOptions.add(nametagBackground);

		nametagOptions.add(showBadges);
		nametagOptions.add(customBadge);
		nametagOptions.add(badgeText);

		general.add(loadingScreenColor);
		general.add(nightMode);
		general.add(customWindowTitle);
		general.add(rawMouseInput);
		general.add(openCredits);
		general.add(debugLogOutput);

		general.add(datetimeFormat);
		general.add(titleScreenOptionButtonMode);
		general.add(gameMenuScreenOptionButtonMode);

		ConfigUI.getInstance().runWhenLoaded(() -> {
			general.getOptions().removeIf(o -> "configStyle".equals(o.getName()));
			String[] themes = ConfigUI.getInstance().getStyleNames().stream().map(s -> "configStyle." + s)
				.filter(s -> AxolotlClientCommon.NVG_SUPPORTED || !s.startsWith("rounded"))
				.toArray(String[]::new);
			if (themes.length > 1) {
				StringArrayOption configStyle;
				general.add(configStyle = new StringArrayOption("configStyle", themes,
					"configStyle." + ConfigUI.getInstance().getCurrentStyle().getName(), s -> {
					ConfigUI.getInstance().setStyle(s.split("\\.")[1]);

					C_3020744 newScreen = RecreatableScreen.tryRecreate(C_8105098.m_0408063().f_0723335);
					C_8105098.m_0408063().m_6408915(newScreen);
				}));
				AxolotlClient.getInstance().getConfigManager().load();
				ConfigUI.getInstance().setStyle(configStyle.get().split("\\.")[1]);
			} else {
				AxolotlClient.getInstance().getConfigManager().load();
			}
		});

		rendering.add(customSky,
			cloudHeight,
			dynamicFOV,
			fullBright,
			removeVignette,
			lowFire,
			hitColor,
			minimalViewBob,
			flatItems,
			noHurtCam,
			inventoryPotionEffectOffset);

		timeChanger.add(timeChangerEnabled);
		timeChanger.add(customTime);
		rendering.add(timeChanger);

		outlines.add(enableCustomOutlines);
		outlines.add(outlineColor);
		outlines.add(outlineWidth);
		rendering.add(outlines);

		rendering.add(noRain);
		titles.add(scaleTitles, titlePadding);
		rendering.add(titles);

		hidden.add(creditsBGM, someNiceBackground);

		AxolotlClient.getInstance().modules.add(new Module() {
			@Override
			public void lateInit() {
				if (WindowAccess.getInstance().rawMouseMotionAvailable()) {

					if (System.getProperty("org.lwjgl.input.Mouse.disableRawInput") == null) {
						System.setProperty("org.lwjgl.input.Mouse.disableRawInput", "true");
					}
					WindowAccess.getInstance().setRawMouseMotion(rawMouseInput.get());
				} else {
					AxolotlClient.getInstance().getConfigManager().suppressName(rawMouseInput.getName());
				}
			}
		});

		general.add(new GenericOption("profiles.title", "profiles.configure", () ->
			C_8105098.m_0408063().m_6408915(new ProfilesScreen(C_8105098.m_0408063().f_0723335))), false);

		var toggleFullbright = new C_7778778("toggle_fullbright", 0, "category.axolotlclient");
		KeyBindingEvents.REGISTER_KEYBINDS.register(reg -> reg.register(toggleFullbright));
		MinecraftClientEvents.TICK_END.register(minecraft -> {
			if (toggleFullbright.m_4823804()) {
				fullBright.toggle();
			}
		});
	}
}
