/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.config.screen;

import java.util.List;

import io.github.axolotlclient.AxolotlClientConfig.impl.ui.ButtonWidget;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.Element;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.TextFieldWidget;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.vanilla.ElementListWidget;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.vanilla.widgets.VanillaButtonWidget;
import io.github.axolotlclient.config.profiles.Profiles;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.unmapped.C_3020744;
import net.minecraft.unmapped.C_3390001;
import net.minecraft.unmapped.C_8105098;

public class ProfilesScreen extends io.github.axolotlclient.AxolotlClientConfig.impl.ui.Screen {

	private final C_3020744 parent;

	public ProfilesScreen(C_3020744 parent) {
		super(C_3390001.m_2053009("profiles.configure.list"));
		this.parent = parent;
	}

	@Override
	public void m_7261014(int mouseX, int mouseY, float delta) {
		super.m_7261014(mouseX, mouseY, delta);
		m_2717572(f_2020658, getTitle(), f_5465691 / 2, 33 / 2 - f_2020658.f_6725889 / 2, -1);
	}

	@Override
	public void m_3593494() {
		addDrawableChild(new ProfilesList(f_7153641, f_5465691, f_3080061, 33, f_3080061 - 33, 25));

		addDrawableChild(new VanillaButtonWidget(f_5465691 / 2 - 75, f_3080061 - 33 / 2 - 10, 150, 20, C_3390001.m_2053009("gui.back"), btn -> closeScreen()));
	}


	public void closeScreen() {
		Profiles.getInstance().saveProfiles();
		f_7153641.m_6408915(parent);
	}

	public class ProfilesList extends ElementListWidget<ProfilesList.Entry> {
		private static final io.github.axolotlclient.config.screen.ProfilesScreen.ProfilesList.Entry SPACER = new SpacerEntry();
		private final io.github.axolotlclient.config.screen.ProfilesScreen.ProfilesList.Entry ADD = new NewEntry();

		public ProfilesList(C_8105098 minecraft, int width, int height, int top, int bottom, int itemHeight) {
			super(minecraft, width, height, top, bottom, itemHeight);
			reload();
		}

		public void reload() {
			clearEntries();
			Profiles.getInstance().iterateAvailable(p -> addEntry(new ProfileEntry(p)));
			addEntry(SPACER);
			addEntry(ADD);
		}

		@Override
		protected int getScrollbarPositionX() {
			return getRowLeft() + getRowWidth() + 10;
		}

		@Override
		public int getRowWidth() {
			return 340;
		}

		@Override
		public boolean mouseClicked(double mouseX, double mouseY, int button) {
			children().stream().filter(e -> e instanceof ProfileEntry)
				.map(e -> (ProfileEntry) e).map(e -> e.profileName).forEach(e -> e.setFocused(false));
			return super.mouseClicked(mouseX, mouseY, button);
		}

		@Environment(EnvType.CLIENT)
		public abstract static class Entry extends ElementListWidget.Entry<io.github.axolotlclient.config.screen.ProfilesScreen.ProfilesList.Entry> {

		}

		public static class SpacerEntry extends io.github.axolotlclient.config.screen.ProfilesScreen.ProfilesList.Entry {
			@Override
			public void render(int index, int top, int left, int width, int height, int mouseX, int mouseY, boolean hovering, float partialTick) {

			}

			@Override
			public List<? extends Element> children() {
				return List.of();
			}
		}

		@Environment(EnvType.CLIENT)
		public class ProfileEntry extends io.github.axolotlclient.config.screen.ProfilesScreen.ProfilesList.Entry {
			private static final String EXPORT_BUTTON_TITLE = C_3390001.m_2053009("profiles.profile.export");
			private static final String CURRENT_TEXT = C_3390001.m_2053009("profiles.profile.current");
			private static final String LOAD_BUTTON_TITLE = C_3390001.m_2053009("profiles.profile.load");
			private static final String DUPLICATE_BUTTON_TITLE = C_3390001.m_2053009("profiles.profile.duplicate");
			private static final String REMOVE_BUTTON_TITLE = C_3390001.m_2053009("profiles.profile.remove");
			private final TextFieldWidget profileName;
			private final ButtonWidget exportButton, loadButton, duplicateButton, removeButton;
			private final Profiles.Profile profile;

			ProfileEntry(Profiles.Profile profile) {
				this.profile = profile;
				profileName = new TextFieldWidget(f_2020658, 0, 0, 150, 20, "");
				profileName.setText(profile.name());
				profileName.setChangedListener(profile::setName);
				exportButton = new VanillaButtonWidget(0, 0, 50, 20, EXPORT_BUTTON_TITLE, btn ->
					Profiles.getInstance().exportProfile(profile));
				loadButton = new VanillaButtonWidget(0, 0, 50, 20, LOAD_BUTTON_TITLE, btn ->
					Profiles.getInstance().switchTo(profile));
				duplicateButton = new VanillaButtonWidget(0, 0, 50, 20, DUPLICATE_BUTTON_TITLE, b -> {
					var dup = Profiles.getInstance().duplicate(profile);
					double d = (double) ProfilesList.this.getMaxScroll() - ProfilesList.this.getScrollAmount();
					ProfilesList.this.children().add(ProfilesList.this.children().indexOf(ProfileEntry.this) + 1, new ProfileEntry(dup));
					ProfilesList.this.setScrollAmount(ProfilesList.this.getMaxScroll() - d);
				});

				this.removeButton = new VanillaButtonWidget(0, 0, 50, 20, REMOVE_BUTTON_TITLE, b -> {
					removeEntry(this);
					Profiles.getInstance().remove(profile);
					setScrollAmount(getScrollAmount());
				});
			}

			@Override
			public void render(int index, int top, int left, int width, int height, int mouseX, int mouseY, boolean hovering, float partialTick) {
				int i = getScrollbarPositionX() - removeButton.getWidth() - 4;
				int j = top - 2;
				this.removeButton.setPosition(i, j);
				this.removeButton.render(mouseX, mouseY, partialTick);

				i -= duplicateButton.getWidth();
				duplicateButton.setPosition(i, j);
				duplicateButton.render(mouseX, mouseY, partialTick);

				boolean current = Profiles.getInstance().getCurrent() == profile;
				loadButton.setMessage(current ? CURRENT_TEXT : LOAD_BUTTON_TITLE);
				loadButton.active = removeButton.active = !current;
				i -= loadButton.getWidth();
				this.loadButton.setPosition(i, j);
				this.loadButton.render(mouseX, mouseY, partialTick);
				i -= exportButton.getWidth();
				exportButton.setPosition(i, j);
				exportButton.render(mouseX, mouseY, partialTick);
				profileName.setWidth(i - left - 4);
				profileName.setPosition(left, j);
				profileName.render(mouseX, mouseY, partialTick);
			}

			@Override
			public List<? extends Element> children() {
				return List.of(profileName, this.loadButton, duplicateButton, removeButton);
			}
		}

		public class NewEntry extends io.github.axolotlclient.config.screen.ProfilesScreen.ProfilesList.Entry {

			private final ButtonWidget addButton, importButton;

			public NewEntry() {
				this.addButton = new VanillaButtonWidget(0, 0, 150, 20, C_3390001.m_2053009("profiles.profile.add"), button -> {
					int i = ProfilesList.this.children().indexOf(this);
					ProfilesList.this.children().add(Math.max(i - 1, 0), new ProfileEntry(Profiles.getInstance().newProfile(C_3390001.m_2053009("profiles.profile.default_new_name"))));
					Profiles.getInstance().saveProfiles();
					setScrollAmount(getMaxScroll());
				});
				this.importButton = new VanillaButtonWidget(0, 0, 150, 20, C_3390001.m_2053009("profiles.profile.import"), btn ->
					Profiles.getInstance().importProfiles().thenRun(ProfilesList.this::reload));
			}

			@Override
			public void render(int index, int top, int left, int width, int height, int mouseX, int mouseY, boolean hovering, float partialTick) {
				int i = getScrollbarPositionX() - width / 2 - 10 - addButton.getWidth() + 2;
				int j = top - 2;
				this.addButton.setPosition(i, j);
				this.addButton.render(mouseX, mouseY, partialTick);
				this.importButton.setPosition(addButton.getX() + addButton.getWidth() + 2, j);
				this.importButton.render(mouseX, mouseY, partialTick);
			}

			@Override
			public List<? extends Element> children() {
				return List.of(addButton, importButton);
			}
		}
	}
}
