/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.mixin;

import com.llamalad7.mixinextras.injector.wrapoperation.Operation;
import com.llamalad7.mixinextras.injector.wrapoperation.WrapOperation;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.bridge.events.types.WorldLoadEvent;
import io.github.axolotlclient.modules.auth.Auth;
import io.github.axolotlclient.modules.blur.MenuBlur;
import io.github.axolotlclient.modules.hud.HudManager;
import io.github.axolotlclient.modules.hud.HudManagerCommon;
import io.github.axolotlclient.modules.rpc.DiscordRPC;
import io.github.axolotlclient.modules.zoom.Zoom;
import io.github.axolotlclient.util.Util;
import io.github.axolotlclient.util.events.Events;
import io.github.axolotlclient.util.events.impl.MouseInputEvent;
import net.minecraft.unmapped.C_0488071;
import net.minecraft.unmapped.C_0876962;
import net.minecraft.unmapped.C_1023567;
import net.minecraft.unmapped.C_1331819;
import net.minecraft.unmapped.C_2670477;
import net.minecraft.unmapped.C_2712203;
import net.minecraft.unmapped.C_3020744;
import net.minecraft.unmapped.C_7778778;
import net.minecraft.unmapped.C_7873567;
import net.minecraft.unmapped.C_8105098;
import net.minecraft.unmapped.C_8373595;
import org.apache.logging.log4j.Logger;
import org.lwjgl.input.Keyboard;
import org.lwjgl.input.Mouse;
import org.lwjgl.opengl.Display;
import org.spongepowered.asm.mixin.Final;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Shadow;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.Redirect;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;

@Mixin(C_8105098.class)
public abstract class MinecraftClientMixin {

	@Shadow
	public C_1331819 options;
	@Shadow
	public C_1023567 player;
	@Shadow
	@Final
	private String gameVersion;
	@Shadow
	private C_0876962 textureManager;

	protected MinecraftClientMixin(C_0876962 textureManager) {
		this.textureManager = textureManager;
	}

	@SuppressWarnings("EmptyMethod")
	@Redirect(method = "<init>", at = @At(value = "INVOKE", target = "Lorg/apache/logging/log4j/Logger;info(Ljava/lang/String;)V", ordinal = 1), remap = false)
	public void axolotlclient$noSessionIDLeak(Logger instance, String s) {
	}

	/**
	 * @author TheKodeToad & Sk1erLLC (initially created this fix).
	 * @reason unnecessary garbage collection
	 */
	@SuppressWarnings("EmptyMethod")
	@Redirect(method = "startGame", at = @At(value = "INVOKE", target = "Ljava/lang/System;gc()V"))
	public void axolotlclient$noWorldGC() {
	}

	@Inject(method = "setWorld(Lnet/minecraft/client/world/ClientWorld;Ljava/lang/String;)V", at = @At("HEAD"))
	private void axolotlclient$onWorldLoad(C_7873567 clientWorld, String string, CallbackInfo ci) {
		io.github.axolotlclient.bridge.events.Events.WORLD_LOAD_EVENT.invoker().accept(new WorldLoadEvent(clientWorld));
	}

	/**
	 * @author moehreag
	 * @reason Customize Window title for use in AxolotlClient
	 */
	@Inject(method = "initDisplay", at = @At("TAIL"))
	public void axolotlclient$setWindowTitle(CallbackInfo ci) {
		if (AxolotlClient.config().customWindowTitle.get()) {
			Display.setTitle("AxolotlClient " + this.gameVersion);
		}
	}

	@Redirect(method = "handleGuiKeyBindings", at = @At(value = "INVOKE", target = "Lnet/minecraft/client/options/KeyBinding;getKeyCode()I", ordinal = 5))
	// Fix taking a screenshot when pressing '<' (Because it has the same keyCode as F2)
	public int axolotlclient$iTryToFixTheScreenshotKey(C_7778778 instance) {
		if (Keyboard.getEventCharacter() != '<') {
			return instance.m_6463487();
		}

		return -999; // There is no key with this Code, but this is what we want here.
	}

	// Don't ask me why we need both here, but otherwise it looks ugly
	@Redirect(method = "renderMojangLogo", at = @At(value = "INVOKE", target = "Lcom/mojang/blaze3d/vertex/BufferBuilder;color(IIII)Lcom/mojang/blaze3d/vertex/BufferBuilder;"))
	public C_8373595 axolotlclient$loadingScreenColor(C_8373595 instance, int red, int green, int blue, int alpha) {
		return instance.m_9724942(AxolotlClient.config().loadingScreenColor.get().getRed(),
			AxolotlClient.config().loadingScreenColor.get().getGreen(),
			AxolotlClient.config().loadingScreenColor.get().getBlue(),
			AxolotlClient.config().loadingScreenColor.get().getAlpha());
	}

	@Redirect(method = "renderLoadingScreen", at = @At(value = "INVOKE", target = "Lcom/mojang/blaze3d/vertex/BufferBuilder;color(IIII)Lcom/mojang/blaze3d/vertex/BufferBuilder;"))
	public C_8373595 axolotlclient$loadingScreenBg(C_8373595 instance, int red, int green, int blue, int alpha) {
		return instance.m_9724942(AxolotlClient.config().loadingScreenColor.get().getRed(),
			AxolotlClient.config().loadingScreenColor.get().getGreen(),
			AxolotlClient.config().loadingScreenColor.get().getBlue(),
			AxolotlClient.config().loadingScreenColor.get().getAlpha());
	}

	@Inject(method = "init", at = @At(value = "INVOKE", target = "Lnet/minecraft/client/render/texture/TextureManager;close(Lnet/minecraft/resource/Identifier;)V"))
	private void axolotlclient$onLaunch(CallbackInfo ci) {
		HudManager.getInstance().refreshAllBounds();
		HudManagerCommon.getInstance().refreshAllBounds();
		if (!API.getInstance().isSocketConnected() && !Auth.getInstance().getCurrent().isOffline()) {
			API.getInstance().startup(Auth.getInstance().getCurrent());
		}
	}

	@Redirect(method = "<init>", at = @At(value = "FIELD", target = "Lnet/minecraft/client/main/RunArgs$Game;version:Ljava/lang/String;"))
	private String axolotlclient$redirectVersion(C_0488071.C_5779055 game) {
		return "1.8.9";
	}

	@Inject(method = "startGame", at = @At("HEAD"))
	public void axolotlclient$startup(String worldFileName, String worldName, C_2670477 levelInfo, CallbackInfo ci) {
		DiscordRPC.getInstance().setWorld(worldFileName);
	}

	@Inject(method = "stop", at = @At("HEAD"))
	public void axolotlclient$stop(CallbackInfo ci) {
		DiscordRPC.getInstance().shutdown();
	}

	@Redirect(method = "tick", at = @At(value = "INVOKE", target = "Lorg/lwjgl/input/Mouse;getEventDWheel()I"), remap = false)
	public int axolotlclient$onScroll() {
		int amount = Mouse.getEventDWheel();
		if (amount != 0 && Zoom.scroll(amount)) {
			return 0;
		}
		return amount;
	}

	@Inject(method = "tick", at = @At(value = "INVOKE", target = "Lnet/minecraft/client/Minecraft;getTime()J", ordinal = 0))
	public void axolotlclient$onMouseButton(CallbackInfo ci) {
		if (Mouse.getEventButtonState()) {
			Events.MOUSE_INPUT.invoker().invoke(new MouseInputEvent(Mouse.getEventButton()));
		}
	}

	@Inject(method = "updateWindow", at = @At(value = "INVOKE", target = "Lnet/minecraft/client/Minecraft;onResolutionChanged(II)V"))
	public void axolotlclient$onResize(CallbackInfo ci) {
		Util.window = null;
		HudManager.getInstance().refreshAllBounds();
		HudManagerCommon.getInstance().refreshAllBounds();
	}

	@Inject(method = "openScreen", at = @At("HEAD"))
	private void axolotlclient$onScreenOpen(C_3020744 screen, CallbackInfo ci) {
		if (C_8105098.m_0408063().f_0723335 == null) {
			MenuBlur.getInstance().onScreenOpen();
		}
	}

	@WrapOperation(method = "openScreen", at = @At(value = "INVOKE", target = "Lnet/minecraft/client/gui/chat/ChatGui;clear()V"))
	private void keepChatMessagesOnDisconnect(C_2712203 instance, Operation<Void> original) {
		io.github.axolotlclient.modules.hud.gui.hud.ChatHud hud = (io.github.axolotlclient.modules.hud.gui.hud.ChatHud) HudManager
			.getInstance().get(io.github.axolotlclient.modules.hud.gui.hud.ChatHud.ID);
		if (hud != null && !hud.keepMessagesOnDisconnect.get()) {
			original.call(instance);
		}
	}

	@Inject(method = "init", at = @At(value = "INVOKE", target = "Lnet/minecraft/client/gui/screen/TitleScreen;<init>()V"))
	private void onGameLoad(CallbackInfo ci) {
		Events.GAME_LOAD_EVENT.invoker().invoke((C_8105098) (Object) this);
	}
}
