/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.auth;

import java.util.*;
import java.util.concurrent.CompletableFuture;

import com.mojang.authlib.GameProfile;
import com.mojang.authlib.minecraft.MinecraftProfileTexture;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.api.types.User;
import io.github.axolotlclient.api.util.UUIDHelper;
import io.github.axolotlclient.mixin.MinecraftClientAccessor;
import io.github.axolotlclient.modules.Module;
import io.github.axolotlclient.modules.auth.skin.SkinManager;
import io.github.axolotlclient.util.ThreadExecuter;
import io.github.axolotlclient.util.notifications.Notifications;
import io.github.axolotlclient.util.options.GenericOption;
import lombok.Getter;
import net.minecraft.unmapped.C_0561170;
import net.minecraft.unmapped.C_1441789;
import net.minecraft.unmapped.C_3020744;
import net.minecraft.unmapped.C_3390001;
import net.minecraft.unmapped.C_5946149;
import net.minecraft.unmapped.C_8105098;
import net.minecraft.unmapped.C_8730536;

public class Auth extends Accounts implements Module {

	@Getter
	private static final Auth Instance = new Auth();
	public final BooleanOption showButton = new BooleanOption("auth.showButton", false);
	public final BooleanOption skinManagerAnimations = new BooleanOption("skins.manage.animations", true);
	private final C_8105098 client = C_8105098.m_0408063();
	private final GenericOption viewAccounts = new GenericOption("viewAccounts", "clickToOpen", () -> client.m_6408915(new AccountsScreen(client.f_0723335)));
	private final Map<String, C_0561170> textures = new HashMap<>();
	private final Set<String> loadingTexture = new HashSet<>();
	private final Map<String, GameProfile> profileCache = new WeakHashMap<>();
	@Getter
	private final SkinManager skinManager = new SkinManager();

	@Override
	public void init() {
		load();
		this.msApi = new MSApi(this, () -> client.f_9967940.f_4977087);
		if (isContained(client.m_3271200().m_2698451())) {
			current = getAccounts().stream().filter(account -> account.getUuid().equals(client.m_3271200().m_2698451())).toList().get(0);
			current.setAuthToken(client.m_3271200().m_1063667());
			current.setName(client.m_3271200().m_4631150());
			/*if (current.needsRefresh()) {
				current.refresh(auth).thenRun(this::save);
			}*/
		} else {
			current = new Account(client.m_3271200().m_4631150(), client.m_3271200().m_2698451(), client.m_3271200().m_1063667());
		}

		category.add(showButton, viewAccounts);
		AxolotlClient.config().general.add(category);
	}

	@Override
	protected void login(Account account) {
		if (client.f_4601986 != null) {
			return;
		}

		if (account.needsRefresh() && !account.isOffline()) {
			if (account.isExpired()) {
				Notifications.getInstance().addStatus("auth.notif.title", "auth.notif.refreshing", account.getName());
			}
			account.refresh(msApi).thenAccept(a -> {
				if (!a.isExpired()) {
					login(a);
				}
			}).thenRun(this::save);
		} else {
			try {
				API.getInstance().shutdown();
				((MinecraftClientAccessor) client).setSession(new C_8730536(account.getName(), account.getUuid(), account.getAuthToken(), C_8730536.C_5889190.f_7683414.name()));
				client.m_4708991().clear();
				client.m_4708991();
				save();
				current = account;
				Notifications.getInstance().addStatus("auth.notif.title", "auth.notif.login.successful", current.getName());
				AxolotlClient.LOGGER.info("Successfully logged in as " + current.getName());
				API.getInstance().startup(account);
			} catch (Exception e) {
				AxolotlClient.LOGGER.error("Failed to log in! ", e);
				Notifications.getInstance().addStatus("auth.notif.title", "auth.notif.login.failed");
			}
		}
	}

	public void loadTextures(String uuid, String name) {
		if (!textures.containsKey(uuid) && !loadingTexture.contains(uuid)) {
			ThreadExecuter.scheduleTask(() -> {
				loadingTexture.add(uuid);
				GameProfile gameProfile;
				if (profileCache.containsKey(uuid)) {
					gameProfile = profileCache.get(uuid);
				} else {
					try {
						UUID uUID = UUIDHelper.fromUndashed(uuid);
						gameProfile = new GameProfile(uUID, name);
						client.m_7883741().fillProfileProperties(gameProfile, false);
					} catch (IllegalArgumentException var2) {
						gameProfile = new GameProfile(null, name);
					}
					profileCache.put(uuid, gameProfile);
				}
				client.m_4177774().m_8307311(gameProfile, (type, identifier, minecraftProfileTexture) -> {
					if (type == MinecraftProfileTexture.Type.SKIN) {
						textures.put(uuid, identifier);
						loadingTexture.remove(uuid);
					}
				}, false);
			});
		}
	}

	@Override
	CompletableFuture<Account> showAccountsExpiredScreen(Account account) {
		C_3020744 current = client.f_0723335;
		var fut = new CompletableFuture<Account>();
		client.m_2167033(() -> client.m_6408915(new C_1441789((bl, i) -> {
			if (bl) {
				msApi.startDeviceAuth().thenRun(() -> fut.complete(account));
			} else {
				fut.cancel(true);
			}
			client.m_6408915(current);
		}, C_3390001.m_2053009("auth"), C_3390001.m_2053009("auth.accountExpiredNotice", account.getName()), 1)));
		return fut;
	}

	@Override
	void displayDeviceCode(DeviceFlowData data) {
		client.m_2167033(() -> client.m_6408915(new DeviceCodeDisplayScreen(client.f_0723335, data)));
	}

	public C_0561170 getSkinTexture(Account account) {
		return getSkinTexture(account.getUuid(), account.getName());
	}

	public C_0561170 getSkinTexture(User user) {
		return getSkinTexture(user.getUuid(), user.getName());
	}

	public C_0561170 getSkinTexture(String uuid, String name) {
		loadTextures(uuid, name);
		C_0561170 id;
		if ((id = textures.get(uuid)) != null) {
			return id;
		}
		try {
			UUID uUID = UUIDHelper.fromUndashed(uuid);
			return C_5946149.m_2817663(uUID);
		} catch (IllegalArgumentException ignored) {
			return C_5946149.m_8990473();
		}
	}
}
