/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.hud;

import java.util.List;
import java.util.Optional;
import net.minecraft.unmapped.C_2348249;
import net.minecraft.unmapped.C_3020744;
import net.minecraft.unmapped.C_3390001;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_8105098;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientConfig.api.AxolotlClientConfig;
import io.github.axolotlclient.AxolotlClientConfig.api.options.OptionCategory;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.util.ConfigStyles;
import io.github.axolotlclient.bridge.impl.AxoRenderContextImpl;
import io.github.axolotlclient.modules.hud.gui.component.HudEntry;
import io.github.axolotlclient.modules.hud.snapping.SnappingHelper;
import io.github.axolotlclient.modules.hud.util.DrawPosition;
import io.github.axolotlclient.modules.hud.util.Rectangle;
import io.github.axolotlclient.util.WindowAccess;

/**
 * This implementation of Hud modules is based on KronHUD.
 * <a href="https://github.com/DarkKronicle/KronHUD">Github Link.</a>
 *
 * <p>License: GPL-3.0</p>
 */

public class HudEditScreen extends C_3020744 {

	private static final BooleanOption snapping = new BooleanOption("snapping", true);
	private static final OptionCategory hudEditScreenCategory = OptionCategory.create("hudEditScreen");
	private static final int GRAB_TOLERANCE = 5;
	private final long MOVE_CURSOR = WindowAccess.getInstance().createCursor(WindowAccess.Cursor.RESIZE_ALL);
	private final long DEFAULT_CURSOR = WindowAccess.getInstance().createCursor(WindowAccess.Cursor.ARROW);
	private final long NWSE_RESIZE_CURSOR = WindowAccess.getInstance().createCursor(WindowAccess.Cursor.RESIZE_NWSE),
		NESW_RESIZE_CURSOR = WindowAccess.getInstance().createCursor(WindowAccess.Cursor.RESIZE_NESW);

	public static boolean isSnappingEnabled() {
		return snapping.get();
	}

	public static void toggleSnapping() {
		snapping.toggle();
	}

	static {
		hudEditScreenCategory.add(snapping);
		AxolotlClient.config().hidden.add(hudEditScreenCategory);
	}

	private final C_3020744 parent;

	private HudEntry current;
	private DrawPosition offset = null;
	private boolean mouseDown;
	private SnappingHelper snap;
	private long currentCursor;
	private ModificationMode mode = ModificationMode.NONE;

	public HudEditScreen() {
		this(null);
	}

	public HudEditScreen(C_3020744 parent) {
		super();
		updateSnapState();
		mouseDown = false;
		this.parent = parent;
	}

	private void updateSnapState() {
		if (snapping.get() && current != null) {
			List<Rectangle> bounds = HudManager.getInstance().getAllBounds();
			bounds.remove(current.getTrueBounds());
			snap = new SnappingHelper(bounds, current.getTrueBounds());
		} else if (snap != null) {
			snap = null;
		}
	}

	private void setCursor(long cursor) {
		if (cursor > 0 && cursor != currentCursor) {
			currentCursor = cursor;
			WindowAccess.getInstance().setCursor(cursor);
		}
	}

	@Override
	public void m_7261014(int mouseX, int mouseY, float delta) {
		if (C_8105098.m_0408063().f_4601986 != null)
			m_9889399(0, 0, f_5465691, f_3080061, 0xB0100E0E,
				0x46212020);
		else {
			m_4184721(0);
		}

		super.m_7261014(mouseX, mouseY, delta);
		C_3754158.m_7547086();

		Optional<HudEntry> entry;
		if (current != null && mode != ModificationMode.NONE) {
			current.setHovered(true);
			entry = Optional.of(current);
		} else {
			entry = HudManager.getInstance().getEntryXY(mouseX, mouseY);
			entry.ifPresent(abstractHudEntry -> abstractHudEntry.setHovered(true));
		}
		HudManager.getInstance().renderPlaceholder(AxoRenderContextImpl.getInstance(), delta);
		if (entry.isPresent()) {
			var bounds = entry.get().getTrueBounds();
			if (mode == ModificationMode.NONE && bounds.isMouseOver(mouseX, mouseY)) {
				var supportsScaling = entry.get().supportsScaling();
				var xBound = Math.max(0, mouseX - bounds.x());
				var yBound = Math.max(0, mouseY - bounds.y());
				var tolerance = GRAB_TOLERANCE;
				if (supportsScaling && xBound < tolerance && yBound < tolerance) {
					// top-left
					setCursor(NWSE_RESIZE_CURSOR);
				} else if (supportsScaling && Math.abs(xBound - bounds.width()) < tolerance && Math.abs(yBound - bounds.height()) < tolerance) {
					// bottom-right
					setCursor(NWSE_RESIZE_CURSOR);
				} else if (supportsScaling && xBound < tolerance && Math.abs(yBound - bounds.height()) < tolerance) {
					// bottom-left
					setCursor(NESW_RESIZE_CURSOR);
				} else if (supportsScaling && yBound < tolerance && Math.abs(xBound - bounds.width()) < tolerance) {
					// top-right
					setCursor(NESW_RESIZE_CURSOR);
				} else {
					setCursor(MOVE_CURSOR);
				}
			}
		} else if (current == null) {
			setCursor(DEFAULT_CURSOR);
			mode = ModificationMode.NONE;
		}
		if (mouseDown && snap != null) {
			snap.renderSnaps();
		}
	}

	@Override
	public void m_7362766(int mouseX, int mouseY, int button) {
		super.m_7362766(mouseX, mouseY, button);
		Optional<HudEntry> entry = HudManager.getInstance().getEntryXY(mouseX, mouseY);
		if (button == 0) {
			mouseDown = true;
			if (entry.isPresent()) {
				current = entry.get();
				offset = new DrawPosition(mouseX - current.getTruePos().x(),
					mouseY - current.getTruePos().y());
				var bounds = entry.get().getTrueBounds();
				var xBound = Math.max(0, mouseX - bounds.x());
				var yBound = Math.max(0, mouseY - bounds.y());
				if (currentCursor == NWSE_RESIZE_CURSOR) {
					if (xBound < bounds.width() / 2 && yBound < bounds.height() / 2) {
						// top-left corner
						mode = ModificationMode.TOP_LEFT;
					} else if (xBound - bounds.width() / 2 > 0 && yBound - bounds.height() / 2 > 0) {
						// bottom-right corner
						mode = ModificationMode.BOTTOM_RIGHT;
					}
				} else if (currentCursor == NESW_RESIZE_CURSOR) {
					if (xBound < bounds.width() / 2 && yBound - bounds.height() / 2 > 0) {
						// bottom-left corner
						mode = ModificationMode.BOTTOM_LEFT;
					} else if (xBound - bounds.width() / 2 > 0 && yBound < bounds.height() / 2) {
						// top-right corner
						mode = ModificationMode.TOP_RIGHT;
					}
				} else if (currentCursor == MOVE_CURSOR) {
					updateSnapState();
					mode = ModificationMode.MOVE;
				}
			} else {
				mode = ModificationMode.NONE;
				current = null;
			}
		} else if (button == 1) {
			entry.ifPresent(hudEntry -> {
				C_3020744 screen = ConfigStyles.createScreen(this, hudEntry.getCategory());
				mode = ModificationMode.NONE;
				setCursor(DEFAULT_CURSOR);
				C_8105098.m_0408063().m_6408915(screen);
			});
		}
	}

	@Override
	public void m_5308748(int mouseX, int mouseY, int button) {
		if (current != null) {
			AxolotlClientConfig.getInstance().getConfigManager(current.getCategory()).save();
		}

		current = null;
		snap = null;
		mouseDown = false;
		mode = ModificationMode.NONE;
		setCursor(DEFAULT_CURSOR);
		super.m_5308748(mouseX, mouseY, button);
	}

	@Override
	protected void m_5535405(int mouseX, int mouseY, int button, long mouseLastClicked) {
		if (current != null) {
			if (mode == ModificationMode.MOVE) {
				current.setX((mouseX - offset.x()) + current.offsetTrueWidth());
				current.setY(mouseY - offset.y() + current.offsetTrueHeight());
				if (snap != null) {
					Integer snapX, snapY;
					snap.setCurrent(current.getTrueBounds());
					if ((snapX = snap.getCurrentXSnap()) != null) {
						current.setX(snapX + current.offsetTrueWidth());
					}
					if ((snapY = snap.getCurrentYSnap()) != null) {
						current.setY(snapY + current.offsetTrueHeight());
					}
				}
			} else {
				var bounds = current.getTrueBounds();
				int newWidth, newHeight;
				if (mode == ModificationMode.TOP_LEFT) {
					// top-left corner
					newWidth = mouseX - bounds.xEnd();
					newHeight = mouseY - bounds.yEnd();
				} else if (mode == ModificationMode.BOTTOM_LEFT) {
					// bottom-left corner
					newWidth = mouseX - bounds.xEnd();
					newHeight = mouseY - bounds.y();
				} else if (mode == ModificationMode.TOP_RIGHT) {
					// top-right corner
					newWidth = mouseX - bounds.x();
					newHeight = mouseY - bounds.yEnd();
				} else if (mode == ModificationMode.BOTTOM_RIGHT) {
					// bottom-right corner
					newWidth = mouseX - bounds.x();
					newHeight = mouseY - bounds.y();
				} else {
					newWidth = bounds.width();
					newHeight = bounds.height();
				}
				float newScale = current.getScale() * Math.max((float) Math.abs(newWidth) / bounds.width(), (float) Math.abs(newHeight) / bounds.height());
				current.setScale(Math.max(0.1f, newScale));
				if (mode == ModificationMode.TOP_LEFT) {
					// top-left corner
					current.setX(bounds.xEnd() - current.getTrueWidth());
					current.setY(bounds.yEnd() - current.getTrueHeight());
				} else if (mode == ModificationMode.BOTTOM_LEFT) {
					// bottom-left corner
					current.setX(bounds.xEnd() - current.getTrueWidth());
				} else if (mode == ModificationMode.TOP_RIGHT) {
					// top-right corner
					current.setY(bounds.yEnd() - current.getTrueHeight());
				}
			}
			if (current.tickable()) {
				current.tick();
			}
		}
	}

	@Override
	public void m_8984281() {
		super.m_8984281();
		setCursor(DEFAULT_CURSOR);
		mode = ModificationMode.NONE;
		WindowAccess.getInstance().destroyStandardCursor(DEFAULT_CURSOR, NWSE_RESIZE_CURSOR, NESW_RESIZE_CURSOR, MOVE_CURSOR);
	}

	@Override
	protected void m_7971793(C_2348249 button) {
		switch (button.f_5920996) {
			case 3:
				snapping.toggle();
				button.f_4865617 = C_3390001.m_2053009("hud.snapping") + ": "
					+ C_3390001.m_2053009(snapping.get() ? "options.on" : "options.off");
				AxolotlClient.getInstance().getConfigManager().save();
				break;
			case 1:
				C_3020744 screen = ConfigStyles.createScreen(this, AxolotlClient.getInstance().getConfigManager().getRoot());
				C_8105098.m_0408063().m_6408915(screen);
				break;
			case 0:
				C_8105098.m_0408063().m_6408915(parent);
				break;
			case 2:
				C_8105098.m_0408063().m_6408915(null);
				break;
		}
	}

	@Override
	public void m_3593494() {
		mode = ModificationMode.NONE;
		this.f_2213969.add(new C_2348249(3, f_5465691 / 2 - 50, f_3080061 / 2 + 12, 100, 20,
			C_3390001.m_2053009("hud.snapping") + ": " + C_3390001.m_2053009(snapping.get() ? "options.on" : "options.off")));

		this.f_2213969.add(
			new C_2348249(1, f_5465691 / 2 - 75, f_3080061 / 2 - 10, 150, 20, C_3390001.m_2053009("hud.clientOptions")));

		if (parent != null)
			f_2213969.add(new C_2348249(0, f_5465691 / 2 - 75, f_3080061 - 50 + 22, 150, 20, C_3390001.m_2053009("back")));
		else
			f_2213969.add(new C_2348249(2, f_5465691 / 2 - 75, f_3080061 - 50 + 22, 150, 20, C_3390001.m_2053009("close")));
	}

	private enum ModificationMode {
		NONE,
		MOVE,
		TOP_LEFT,
		TOP_RIGHT,
		BOTTOM_LEFT,
		BOTTOM_RIGHT
	}
}
