/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.hud.gui.hud;

import java.util.List;
import net.minecraft.unmapped.C_0561170;
import net.minecraft.unmapped.C_1716360;
import net.minecraft.unmapped.C_2848808;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_4976084;
import net.minecraft.unmapped.C_7225666;
import net.minecraft.unmapped.C_7669754;
import net.minecraft.unmapped.C_8105098;
import net.minecraft.unmapped.C_9550253;
import net.minecraft.unmapped.C_9590849;
import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Color;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.ColorOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.IntegerOption;
import io.github.axolotlclient.bridge.render.AxoRenderContext;
import io.github.axolotlclient.mixin.ChatHudAccessor;
import io.github.axolotlclient.modules.hud.gui.entry.TextHudEntry;
import io.github.axolotlclient.modules.hud.util.DrawPosition;

public class ChatHud extends TextHudEntry {

	public static final C_0561170 ID = new C_0561170("axolotlclient", "chathud");
	// tooltip: "chathud"
	public final BooleanOption background = new BooleanOption("background", true);
	public final ColorOption bgColor = new ColorOption("bgcolor", Color.parse("#80000000"));

	public final IntegerOption chatHistory = new IntegerOption("chatHistoryLength", 100, 10, 5000);
	public final ColorOption scrollbarColor = new ColorOption("scrollbarColor", Color.parse("#70CCCCCC"));
	public final IntegerOption lineSpacing = new IntegerOption("lineSpacing", 0, 0, 10);
	public final BooleanOption keepMessagesOnDisconnect = new BooleanOption("keep_messages_on_disconnect", false);
	public final BooleanOption animateChat = new BooleanOption("animate_chat", false);

	public int ticks;
	private int lastHeight;

	private float percentComplete;
	public int newLines;
	private long prevMillis = System.currentTimeMillis();
	public float animationPercent;

	private final C_8105098 client = (C_8105098) super.client;

	private void updatePercentage(long diff) {
		if (percentComplete < 1)
			percentComplete += 0.004f * diff;
		percentComplete = C_4976084.m_7164829(percentComplete, 0, 1);
	}

	public ChatHud() {
		super(320, 80, false);
	}

	public static int getHeight(float chatHeight) {
		int i = 180;
		int j = 20;
		return C_4976084.m_7192202(chatHeight * (float) (i - j) + (float) j);
	}

	public void resetAnimation() {
		percentComplete = 0;
	}

	@Override
	public void render(AxoRenderContext context, float delta) {
		long current = System.currentTimeMillis();
		long diff = current - prevMillis;
		prevMillis = current;
		updatePercentage(diff);
		float t = percentComplete - 1;
		animationPercent = C_4976084.m_7164829(1 - (float) Math.pow(t, 4), 0, 1);
		int scrolledLines = ((ChatHudAccessor) client.f_3501374.m_8449802()).getScrolledLines();
		List<C_7225666> visibleMessages = ((ChatHudAccessor) client.f_3501374.m_8449802()).getVisibleMessages();

		if (this.client.f_9967940.f_6177536 != C_9590849.C_4285597.f_7311895) {
			C_3754158.m_8373640();
			scale(context);
			DrawPosition pos = getPos();

			int i = getVisibleLineCount();
			int j = 0;
			int k = visibleMessages.size();
			if (k > 0) {
				float g = getScale();
				int l = C_4976084.m_4705638((float) getWidth() / g);
				C_3754158.m_8373640();
				if (animateChat.get() && !((ChatHudAccessor) this.client.f_3501374.m_8449802()).hasUnreadNewMessages()) {
					C_3754158.m_3172490(0, (9 - 9 * animationPercent) * getScale(), 0);
				}

				for (int m = 0; m + scrolledLines < visibleMessages.size() && m < i; ++m) {
					C_7225666 chatHudLine = visibleMessages.get(m + scrolledLines);
					if (chatHudLine != null) {
						int n = ticks - chatHudLine.m_1145422();
						if (n < 200 || isChatFocused()) {
							double d = C_4976084.m_0987703((1.0 - n / 200.0) * 10.0, 0.0, 1.0);
							d *= d;
							if (animateChat.get() && m + scrolledLines < newLines) {
								d *= animationPercent;
							}

							++j;
							int alpha = Math.max(bgColor.get().getAlpha(), textColor.get().getAlpha());
							int opacity = isChatFocused() ? alpha : (int) (alpha * d);
							if (opacity > 3) {
								int y = pos.y + getHeight() - (m * (9 + lineSpacing.get()));
								if (background.get()) {
									Color bg = bgColor.get();
									if (!isChatFocused()) {
										bg = bg.withAlpha((int) (bg.getAlpha() * d));
									}
									context.br$fillRect(pos.x, y - (9 + lineSpacing.get()), pos.x + l + 4, y,
										bg.toInt());
								}
								String string = chatHudLine.m_7102466().m_2117196();
								C_3754158.m_9671730();
								Color text = textColor.get();
								if (!isChatFocused()) {
									text = text.withAlpha((int) (text.getAlpha() * d));
								}
								context.br$drawString(string, pos.x, (y - 8),
									text, shadow.get());
								C_3754158.m_6191982();
								C_3754158.m_5313301();
							}
						}
					}
				}

				if (isChatFocused()) {
					int m = getFontHeight();
					C_3754158.m_3172490(-3.0F, 0.0F, 0.0F);
					int r = k * m + k;
					int n = j * m + j;
					int y = (pos.y + getHeight()) - scrolledLines * n / k;
					if (((ChatHudAccessor) client.f_3501374.m_8449802()).getMessages()
						.size() > getVisibleLineCount()) {
						int height = n * n / r;
						context.br$fillRect(pos.x, y, 2, -height, scrollbarColor.get().toInt());
					}
				}

				C_3754158.m_2041265();
			}
			C_3754158.m_2041265();
		}
	}

	@Override
	public void renderComponent(AxoRenderContext context, float delta) {
	}

	@Override
	public void renderPlaceholderComponent(AxoRenderContext context, float delta) {
		DrawPosition pos = getPos();
		if (C_8105098.m_0408063().f_7663840 != null) {
			client.f_0426313.m_1950885("<" + C_8105098.m_0408063().f_7663840.m_0699887().m_2117196()
				+ "> OOh! There's my Chat now!", pos.x + 1, pos.y + getHeight() - 9, -1);
		} else {
			client.f_0426313.m_1950885("This is where your new and fresh looking chat will be!", pos.x + 1,
				pos.y + getHeight() - 9, -1);
		}
	}

	public int getVisibleLineCount() {
		return getHeight(
			this.isChatFocused() ? this.client.f_9967940.f_1078661 : this.client.f_9967940.f_4089725)
			/ 9;
	}

	public boolean isChatFocused() {
		return this.client.f_0723335 instanceof C_7669754;
	}

	protected int getFontHeight() {
		return C_4976084.m_7192202(C_8105098.m_0408063().f_0426313.f_6725889);
	}

	public C_9550253 getTextAt(int x, int y) {
		List<C_7225666> visibleMessages = ((ChatHudAccessor) client.f_3501374.m_8449802()).getVisibleMessages();

		int offsetOnHudX = C_4976084.m_7192202(x / getScale() - getPos().x);
		int offsetOnHudY = C_4976084.m_7192202(-(y / getScale() - (getPos().y + height)));

		int scrolledLines = ((ChatHudAccessor) client.f_3501374.m_8449802()).getScrolledLines();

		if (offsetOnHudX >= 0 && offsetOnHudY >= 0) {
			int l = Math.min(this.getVisibleLineCount(), visibleMessages.size());
			if (offsetOnHudX <= C_4976084.m_7192202((float) this.getWidth() / this.getScale())
				&& offsetOnHudY < (getFontHeight() + lineSpacing.get()) * l + l) {
				int m = offsetOnHudY / (getFontHeight() + lineSpacing.get()) + scrolledLines;
				if (m >= 0 && m < visibleMessages.size()) {
					C_7225666 chatHudLine = visibleMessages.get(m);
					int n = 0;

					for (C_9550253 text : chatHudLine.m_7102466()) {
						if (text instanceof C_1716360) {
							n += this.client.f_0426313.m_0040387(
								C_2848808.m_4320233(((C_1716360) text).m_0284473(), false));
							if (n > offsetOnHudX) {
								return text;
							}
						}
					}
				}
			}
		}
		return null;
	}

	@Override
	public boolean tickable() {
		return true;
	}

	@Override
	public boolean overridesF3() {
		return true;
	}

	@Override
	public void tick() {
		//setWidth((int) (client.options.chatWidth*320));
		if (lastHeight != getHeight(client.f_9967940.f_4089725)) {
			setHeight(getHeight(this.client.f_9967940.f_4089725));
			onBoundsUpdate();
			lastHeight = getHeight();
		}
	}

	@Override
	public double getDefaultX() {
		return 0.01;
	}

	@Override
	public double getDefaultY() {
		return 0.9;
	}

	@Override
	public C_0561170 getId() {
		return ID;
	}

	@Override
	public List<Option<?>> getConfigurationOptions() {
		List<Option<?>> options = super.getConfigurationOptions();
		options.add(background);
		options.add(bgColor);
		options.add(lineSpacing);
		options.add(scrollbarColor);
		options.add(chatHistory);
		options.add(animateChat);
		options.add(keepMessagesOnDisconnect);
		return options;
	}
}
