/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.hud.gui.hud;

import java.nio.file.Files;
import java.util.*;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientCommon;
import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Color;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.ColorOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.IntegerOption;
import io.github.axolotlclient.bridge.render.AxoRenderContext;
import io.github.axolotlclient.config.profiles.ProfileAware;
import io.github.axolotlclient.mixin.KeyBindAccessor;
import io.github.axolotlclient.modules.hud.ClickInputTracker;
import io.github.axolotlclient.modules.hud.gui.entry.TextHudEntry;
import io.github.axolotlclient.modules.hud.gui.keystrokes.KeystrokePositioningScreen;
import io.github.axolotlclient.modules.hud.gui.keystrokes.KeystrokesScreen;
import io.github.axolotlclient.modules.hud.gui.layout.Justification;
import io.github.axolotlclient.modules.hud.util.DrawPosition;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import io.github.axolotlclient.modules.hud.util.Rectangle;
import io.github.axolotlclient.util.ClientColors;
import io.github.axolotlclient.util.GsonHelper;
import io.github.axolotlclient.util.Util;
import io.github.axolotlclient.util.events.Events;
import io.github.axolotlclient.util.options.GenericOption;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.Setter;
import net.minecraft.unmapped.C_0561170;
import net.minecraft.unmapped.C_1331819;
import net.minecraft.unmapped.C_1945050;
import net.minecraft.unmapped.C_3390001;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_4976084;
import net.minecraft.unmapped.C_7778778;
import net.minecraft.unmapped.C_8105098;

import static io.github.axolotlclient.modules.hud.util.DrawUtil.drawCenteredString;
import static io.github.axolotlclient.modules.hud.util.DrawUtil.drawString;

/**
 * This implementation of Hud modules is based on KronHUD.
 * <a href="https://github.com/DarkKronicle/KronHUD">Github Link.</a>
 *
 * <p>License: GPL-3.0</p>
 */

public class KeystrokeHud extends TextHudEntry implements ProfileAware {

	private static final String KEYSTROKE_SAVE_FILE_NAME = "keystrokes.json";
	public static final C_0561170 ID = new C_0561170("kronhud", "keystrokehud");

	private final ColorOption pressedTextColor = new ColorOption("heldtextcolor", new Color(0xFF000000));
	private final ColorOption pressedBackgroundColor = new ColorOption("heldbackgroundcolor", new Color(0x64FFFFFF));
	private final ColorOption pressedOutlineColor = new ColorOption("heldoutlinecolor", ClientColors.BLACK);

	private final C_8105098 client = (C_8105098) super.client;

	private final GenericOption keystrokesOption = new GenericOption("keystrokes", "keystrokes.configure", () -> client.m_6408915(new KeystrokesScreen(KeystrokeHud.this, client.f_0723335)));
	private final GenericOption configurePositions = new GenericOption("keystrokes.positions", "keystrokes.positions.configure",
		() -> client.m_6408915(new KeystrokePositioningScreen(client.f_0723335, this)));
	private final IntegerOption animationTime = new IntegerOption("keystrokes.animation_time", 100, 0, 500);
	public ArrayList<Keystroke> keystrokes;

	public KeystrokeHud() {
		super(53, 61, true);
		Events.KEYBIND_CHANGE.register(key -> {
			if (Util.getWindow() != null) {
				C_7778778.m_7336979();
				C_7778778.m_8884824();
			}
		});
	}

	public Optional<String> getMouseKeyBindName(C_7778778 keyBinding) {
		if (keyBinding.m_4400998().equalsIgnoreCase(client.f_9967940.f_3307271.m_4400998())) {
			return Optional.of("LMB");
		} else if (keyBinding.m_4400998().equalsIgnoreCase(client.f_9967940.f_1707488.m_4400998())) {
			return Optional.of("RMB");
		} else if (keyBinding.m_4400998().equalsIgnoreCase(client.f_9967940.f_1435335.m_4400998())) {
			return Optional.of("MMB");
		}
		return Optional.empty();
	}

	public void setDefaultKeystrokes() {
		DrawPosition pos = getPos();
		// LMB
		keystrokes.add(createFromKey(new Rectangle(0, 36, 26, 17), pos, client.f_9967940.f_3307271));
		// RMB
		keystrokes.add(createFromKey(new Rectangle(27, 36, 26, 17), pos, client.f_9967940.f_1707488));
		// W
		keystrokes.add(createFromKey(new Rectangle(18, 0, 17, 17), pos, client.f_9967940.f_9911664));
		// A
		keystrokes.add(createFromKey(new Rectangle(0, 18, 17, 17), pos, client.f_9967940.f_7947370));
		// S
		keystrokes.add(createFromKey(new Rectangle(18, 18, 17, 17), pos, client.f_9967940.f_6279366));
		// D
		keystrokes.add(createFromKey(new Rectangle(36, 18, 17, 17), pos, client.f_9967940.f_2763889));

		// Space
		keystrokes.add(new CustomRenderKeystroke(SpecialKeystroke.SPACE));
	}

	public void setKeystrokes() {
		if (Util.getWindow() == null) {
			keystrokes = null;
			return;
			// Wait until render is called
		}
		keystrokes = new ArrayList<>();
		setDefaultKeystrokes();
		loadKeystrokes();
		C_7778778.m_7336979();
		C_7778778.m_8884824();
	}

	public Keystroke createFromKey(Rectangle bounds, DrawPosition offset, C_7778778 key) {
		String name = getMouseKeyBindName(key)
			.orElse(C_1331819.m_9293214(key.m_6463487()).toUpperCase());
		if (name.length() > 4) {
			name = name.substring(0, 2);
		}
		return createFromString(bounds, offset, key, name);
	}

	public Keystroke createFromString(Rectangle bounds, DrawPosition offset, C_7778778 key, String word) {
		return new LabelKeystroke(bounds, offset, key, word);
	}

	@Override
	public void render(AxoRenderContext context, float delta) {
		C_3754158.m_8373640();
		scale(context);
		renderComponent(context, delta);
		C_3754158.m_2041265();
	}

	@Override
	public void renderComponent(AxoRenderContext context, float delta) {
		if (keystrokes == null) {
			setKeystrokes();
		}
		for (Keystroke stroke : keystrokes) {
			stroke.render();
		}
	}

	@Override
	public void renderPlaceholderComponent(AxoRenderContext context, float delta) {
		renderComponent(context, delta);
	}

	@Override
	public boolean tickable() {
		return true;
	}

	@Override
	public void tick() {
		DrawPosition pos = getPos();
		if (keystrokes == null) {
			setKeystrokes();
		}
		for (Keystroke stroke : keystrokes) {
			stroke.offset = pos;
		}
	}

	@Override
	protected boolean getShadowDefault() {
		return false;
	}

	@Override
	public List<Option<?>> getConfigurationOptions() {
		// We want a specific order since this is a more complicated entry
		List<Option<?>> options = new ArrayList<>();
		options.add(enabled);
		options.add(scale);
		options.add(textColor);
		options.add(pressedTextColor);
		options.add(shadow);
		options.add(background);
		options.add(backgroundColor);
		options.add(pressedBackgroundColor);
		options.add(outline);
		options.add(outlineColor);
		options.add(pressedOutlineColor);
		options.add(animationTime);
		options.add(keystrokesOption);
		options.add(configurePositions);
		return options;
	}

	@Override
	public C_0561170 getId() {
		return ID;
	}

	@Override
	public void reloadConfig() {
		keystrokes = null;
	}

	@Override
	public void saveConfig() {
		saveKeystrokes();
	}

	public interface KeystrokeRenderer {

		void render(Keystroke stroke);
	}

	public abstract class Keystroke {

		@Getter
		@Setter
		protected C_7778778 key;
		protected KeystrokeRenderer render;
		@Getter
		protected final Rectangle bounds;
		protected DrawPosition offset;
		private long start = -1;
		private boolean wasPressed = false;

		public Keystroke(Rectangle bounds, DrawPosition offset, C_7778778 key, KeystrokeRenderer render) {
			this.bounds = bounds;
			this.offset = offset;
			this.key = key;
			this.render = render;
		}

		public void setX(int x) {
			bounds.x(x - offset.x());
		}

		public void setY(int y) {
			bounds.y(y - offset.y());
		}

		public Rectangle getRenderPosition() {
			return bounds.offset(offset);
		}

		public Color getFGColor() {
			return isKeyDown() ? ClientColors.blend(textColor.get(), pressedTextColor.get(), getPercentPressed())
				: ClientColors.blend(pressedTextColor.get(), textColor.get(), getPercentPressed());
		}

		private float getPercentPressed() {
			return start == -1 ? 1 : C_4976084.m_7164829((float) (System.currentTimeMillis() - start) / getAnimTime(), 0, 1);
		}

		public void render() {
			renderStroke();
			render.render(this);
		}

		public void renderStroke() {
			if (isKeyDown() != wasPressed) {
				start = System.currentTimeMillis();
			}
			Rectangle rect = getRenderPosition();
			if (background.get()) {
				DrawUtil.fillRect(rect, getColor());
			}
			if (outline.get()) {
				DrawUtil.outlineRect(rect, getOutlineColor());
			}
			if ((float) (System.currentTimeMillis() - start) / getAnimTime() >= 1) {
				start = -1;
			}
			wasPressed = isKeyDown();
		}

		private int getAnimTime() {
			return animationTime.get();
		}

		private boolean isKeyDown() {
			return key != null && key.m_7042641();
		}

		public Color getColor() {
			return isKeyDown()
				? ClientColors.blend(backgroundColor.get(), pressedBackgroundColor.get(), getPercentPressed())
				: ClientColors.blend(pressedBackgroundColor.get(), backgroundColor.get(), getPercentPressed());
		}

		public Color getOutlineColor() {
			return isKeyDown() ? ClientColors.blend(outlineColor.get(), pressedOutlineColor.get(), getPercentPressed())
				: ClientColors.blend(pressedOutlineColor.get(), outlineColor.get(), getPercentPressed());
		}

		public Map<String, Object> serialize() {
			Map<String, Object> map = new HashMap<>();
			map.put("key", key.m_6463487());
			map.put("bounds", Map.of("x", bounds.x(), "y", bounds.y(), "width", bounds.width(), "height", bounds.height()));
			return map;
		}

		public abstract String getLabel();

		public abstract void setLabel(String label);

		public abstract boolean isLabelEditable();
	}

	@SuppressWarnings("unchecked")
	private Keystroke deserializeKey(Map<String, Object> json) {
		if ("option".equals(json.get("type"))) {
			C_7778778 key = KeyBindAccessor.getAllKeyBinds().stream().filter(k -> k.m_4400998().equals(json.getOrDefault("key_name", json.get("option")))).findFirst().orElseThrow();
			return new CustomRenderKeystroke(SpecialKeystroke.byId.get(((String) json.get("special_name")).toLowerCase(Locale.ROOT)),
				getRectangle((Map<String, ?>) json.get("bounds")), getPos(), key);
		} else {
			var key = KeyBindAccessor.getAllKeyBinds().stream().filter(k -> k.m_4400998().equals(json.get("key_name"))).findFirst().orElseThrow();
			return new LabelKeystroke(getRectangle((Map<String, ?>) json.get("bounds")), getPos(), key, (String) json.get("label"), (boolean) json.get("synchronize_label"),
				Justification.valueOf((String) json.getOrDefault("justification", "CENTER")));
		}
	}

	private static Rectangle getRectangle(Map<String, ?> json) {
		return new Rectangle((int) (long) json.get("x"), (int) (long) json.get("y"), (int) (long) json.get("width"), (int) (long) json.get("height"));
	}

	public class CustomRenderKeystroke extends Keystroke {

		private static final Supplier<String> label = () -> C_1945050.f_8012292 + C_3390001.m_2053009("keystrokes.stroke.custom_renderer");

		private final SpecialKeystroke parent;

		public CustomRenderKeystroke(SpecialKeystroke stroke, Rectangle bounds, DrawPosition offset, C_7778778 key) {
			super(bounds, offset, key, (s) -> stroke.getRenderer().render(KeystrokeHud.this, s));
			this.parent = stroke;
		}

		public CustomRenderKeystroke(SpecialKeystroke stroke) {
			this(stroke, stroke.getRect().copy(), KeystrokeHud.this.getPos(), stroke.getKey());
		}

		@Override
		public Map<String, Object> serialize() {
			Map<String, Object> json = super.serialize();
			json.put("type", "option");
			json.put("key_name", key.m_4400998());
			json.put("special_name", parent.getId());
			return json;
		}

		@Override
		public String getLabel() {
			return label.get();
		}

		@Override
		public void setLabel(String label) {

		}

		@Override
		public boolean isLabelEditable() {
			return false;
		}
	}

	public Keystroke newSpecialStroke(SpecialKeystroke stroke) {
		return new CustomRenderKeystroke(stroke);
	}

	public LabelKeystroke newStroke() {
		return new LabelKeystroke(new Rectangle(0, 0, 17, 17), getPos(), null, "", false, Justification.CENTER);
	}

	@Setter
	public class LabelKeystroke extends Keystroke {

		private String label;
		@Getter
		private boolean synchronizeLabel;
		@Getter
		private Justification justification;

		public LabelKeystroke(Rectangle bounds, DrawPosition offset, C_7778778 key, String label) {
			this(bounds, offset, key, label, true, Justification.CENTER);
		}

		public LabelKeystroke(Rectangle bounds, DrawPosition offset, C_7778778 key, String label, boolean synchronizeLabel, Justification justification) {
			super(bounds, offset, key, (stroke) -> {
			});
			this.label = label;
			this.render = (stroke) -> {
				Rectangle strokeBounds = stroke.bounds;
				int x = strokeBounds.x() + stroke.offset.x() + 2 + this.justification.getXOffset(getLabel(), strokeBounds.width() - 3);
				float y = strokeBounds.y() + stroke.offset.y() + ((float) strokeBounds.height() / 2) - 4;

				drawString(getLabel(), x, (int) y, stroke.getFGColor().toInt(), shadow.get());
			};
			setSynchronizeLabel(synchronizeLabel);
			this.justification = justification;
		}

		@Override
		public Map<String, Object> serialize() {
			Map<String, Object> json = super.serialize();
			json.put("type", "custom");
			json.put("key_name", key.m_4400998());
			json.put("label", label);
			json.put("synchronize_label", synchronizeLabel);
			json.put("justification", justification.name());
			return json;
		}

		public void setSynchronizeLabel(boolean synchronizeLabel) {
			if (synchronizeLabel) {
				String name = getMouseKeyBindName(key)
					.orElse(C_1331819.m_9293214(key.m_6463487()).toUpperCase());
				if (name.length() > 4) {
					name = name.substring(0, 2);
				}
				this.label = name;
			}
			this.synchronizeLabel = synchronizeLabel;
		}

		@Override
		public void setKey(C_7778778 key) {
			if (synchronizeLabel) {
				String name = getMouseKeyBindName(key)
					.orElse(C_1331819.m_9293214(key.m_6463487()).toUpperCase());
				if (name.length() > 4) {
					name = name.substring(0, 2);
				}
				this.label = name;
			}
			super.setKey(key);
		}

		@Override
		public String getLabel() {
			return label;
		}

		@Override
		public boolean isLabelEditable() {
			return true;
		}
	}

	public void saveKeystrokes() {
		if (keystrokes == null) return;
		try {
			var path = AxolotlClientCommon.resolveProfileConfigFile(KEYSTROKE_SAVE_FILE_NAME);
			Files.createDirectories(path.getParent());
			Files.writeString(path, GsonHelper.GSON.toJson(keystrokes.stream().map(Keystroke::serialize).toList()));
		} catch (Exception e) {
			AxolotlClient.LOGGER.warn("Failed to save keystroke configuration!", e);
		}
	}

	@SuppressWarnings("unchecked")
	public void loadKeystrokes() {
		try {
			var path = AxolotlClientCommon.resolveProfileConfigFile(KEYSTROKE_SAVE_FILE_NAME);
			if (Files.exists(path)) {
				List<?> entries = (List<?>) GsonHelper.read(Files.readString(path));
				var loaded = entries.stream().map(e -> (Map<String, Object>) e)
					.map(KeystrokeHud.this::deserializeKey)
					.toList();
				if (keystrokes == null) {
					keystrokes = new ArrayList<>();
				} else {
					keystrokes.clear();
				}
				keystrokes.addAll(loaded);
			} else {
				saveKeystrokes();
			}
		} catch (Exception e) {
			AxolotlClient.LOGGER.warn("Failed to load keystroke configuration, using defaults!", e);
		}
	}

	@AllArgsConstructor
	@Getter
	public enum SpecialKeystroke {
		SPACE("space", new Rectangle(0, 54, 53, 7), C_8105098.m_0408063().f_9967940.f_2128824, (hud, stroke) -> {
			Rectangle bounds = stroke.bounds;
			Rectangle spaceBounds = new Rectangle(bounds.x() + stroke.offset.x() + 4,
				bounds.y() + stroke.offset.y() + bounds.height() / 2 - 1, bounds.width() - 8, 1);
			DrawUtil.fillRect(spaceBounds, stroke.getFGColor());
			if (hud.shadow.get()) {
				DrawUtil.fillRect(spaceBounds.offset(1, 1), new Color(
					(stroke.getFGColor().toInt() & 16579836) >> 2 | stroke.getFGColor().toInt() & -16777216));
			}
		}),
		LMB_CPS("lmb_cps", new Rectangle(0, 36, 26, 17), C_8105098.m_0408063().f_9967940.f_3307271, (hud, stroke) -> {
			Rectangle bounds = stroke.bounds;
			int centerX = bounds.x() + stroke.offset.x() + bounds.width() / 2;
			int y = bounds.y() + stroke.offset.y() + 3;
			int nameY = y + bounds.height() / 4 - hud.client.textRenderer.fontHeight / 2;
			drawCenteredString(hud.client.textRenderer, "LMB", centerX, nameY, stroke.getFGColor(), hud.shadow.get());
			int cpsY = y + bounds.height() * 3 / 4 - hud.client.textRenderer.fontHeight / 2;
			GlStateManager.pushMatrix();
			GlStateManager.translatef(centerX, cpsY, 0);
			GlStateManager.scalef(0.5f, 0.5f, 1);
			String cpsText = ClickInputTracker.getInstance().leftMouse.clicks() + " CPS";
			GlStateManager.translatef(-hud.client.textRenderer.getWidth(cpsText) / 2f, 0, 0);
			drawString(cpsText, 0, 0, stroke.getFGColor(), hud.shadow.get());
			GlStateManager.popMatrix();
		}),
		RMB_CPS("rmb_cps", new Rectangle(27, 36, 26, 17), C_8105098.m_0408063().f_9967940.f_1707488, (hud, stroke) -> {
			Rectangle bounds = stroke.bounds;
			int centerX = bounds.x() + stroke.offset.x() + bounds.width() / 2;
			int y = bounds.y() + stroke.offset.y() + 3;
			int nameY = y + bounds.height() / 4 - hud.client.textRenderer.fontHeight / 2;
			drawCenteredString(hud.client.textRenderer, "RMB", centerX, nameY, stroke.getFGColor(), hud.shadow.get());
			int cpsY = y + bounds.height() * 3 / 4 - hud.client.textRenderer.fontHeight / 2;
			GlStateManager.pushMatrix();
			GlStateManager.translatef(centerX, cpsY, 0);
			GlStateManager.scalef(0.5f, 0.5f, 1);
			String cpsText = ClickInputTracker.getInstance().rightMouse.clicks() + " CPS";
			GlStateManager.translatef(-hud.client.textRenderer.getWidth(cpsText) / 2f, 0, 0);
			drawString(cpsText, 0, 0, stroke.getFGColor(), hud.shadow.get());
			GlStateManager.popMatrix();
		});
		private static final Map<String, SpecialKeystroke> byId = Arrays.stream(values()).collect(Collectors.toMap(SpecialKeystroke::getId, Function.identity()));

		private final String id;
		private final Rectangle rect;
		private final C_7778778 key;
		private final SpecialKeystrokeRenderer renderer;

		public interface SpecialKeystrokeRenderer {
			void render(KeystrokeHud hud, Keystroke stroke);
		}
	}
}
