/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.hud.gui.hud;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.bridge.impl.AxoRenderContextImpl;
import io.github.axolotlclient.bridge.render.AxoRenderContext;
import io.github.axolotlclient.modules.hud.gui.entry.TextHudEntry;
import io.github.axolotlclient.modules.hud.util.DrawPosition;
import lombok.Getter;
import net.minecraft.unmapped.C_0561170;
import net.minecraft.unmapped.C_2691939;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_3779718;
import net.minecraft.unmapped.C_7532262;
import net.minecraft.unmapped.C_8105098;
import net.ornithemc.osl.resource.loader.api.ModResourcePack;

public class PackDisplayHud extends TextHudEntry {

	public static final C_0561170 ID = new C_0561170("axolotlclient", "packdisplayhud");

	private final List<PackWidget> widgets = new ArrayList<>();
	private final List<C_7532262> packs = new ArrayList<>();
	private final BooleanOption iconsOnly = new BooleanOption("iconsonly", false);
	private PackWidget placeholder;

	public PackDisplayHud() {
		super(200, 50, true);
	}

	public void setPacks(List<C_7532262> packs) {
		widgets.clear();
		this.packs.clear();
		this.packs.addAll(packs);
	}

	@Override
	public void renderComponent(AxoRenderContext context, float f) {
		DrawPosition pos = getPos();

		if (widgets.isEmpty())
			init();

		int y = pos.y + 1;
		for (int i = widgets.size() - 1; i >= 0; i--) { // Badly reverse the order (I'm sure there are better ways to do this)
			widgets.get(i).render(pos.x + 1, y);
			y += 18;
		}
		if (y - pos.y + 1 != getHeight()) {
			setHeight(y - pos.y - 1);
			onBoundsUpdate();
		}
	}

	@Override
	public void init() {
		packs.forEach(pack -> {
			try {
				if (pack instanceof ModResourcePack) {
					return;
				}
				if (pack.m_3372708() != null) {
					if (packs.size() == 1) {
						widgets.add(new PackWidget(pack));
					} else if (!pack.m_4900708().equalsIgnoreCase("Default")) {
						widgets.add(new PackWidget(pack));
					}
				}
			} catch (Exception ignored) {
			}
		});

		AtomicInteger w = new AtomicInteger(20);
		widgets.forEach(packWidget -> {
			int textW = C_8105098.m_0408063().f_0426313.m_0040387(packWidget.getName()) + 20;
			if (textW > w.get())
				w.set(textW);
		});
		setWidth(w.get());

		setHeight(widgets.size() * 18);
		onBoundsUpdate();
	}

	@Override
	public void renderPlaceholderComponent(AxoRenderContext context, float delta) {
		boolean updateBounds = false;
		if (getHeight() < 18) {
			setHeight(18);
			updateBounds = true;
		}
		if (getWidth() < 56) {
			setWidth(56);
			updateBounds = true;
		}
		if (updateBounds) {
			onBoundsUpdate();
		}
		if (placeholder == null) {
			placeholder = new PackWidget(C_8105098.m_0408063().m_7716675().f_6120754);
		}
		placeholder.render(getPos().x + 1, getPos().y + 1);
	}

	@Override
	public List<Option<?>> getConfigurationOptions() {
		List<Option<?>> options = super.getConfigurationOptions();
		options.add(iconsOnly);
		return options;
	}

	@Override
	public C_0561170 getId() {
		return ID;
	}

	private class PackWidget {

		@Getter
		private final String name;
		private int texture;

		public PackWidget(C_7532262 pack) {
			this.name = pack.m_4900708();
			try {
				this.texture = new C_3779718(pack.m_3372708()).m_6799083();
			} catch (Exception e) {
				AxolotlClient.LOGGER.warn("Pack " + pack.m_4900708()
					+ " somehow threw an error! Please investigate... Does it have an icon?");
			}
		}

		public void render(int x, int y) {
			if (!iconsOnly.get()) {
				C_3754158.m_3323122(1, 1, 1, 1F);
				C_3754158.m_2768959(texture);
				C_2691939.m_5935491(x, y, 0, 0, 16, 16, 16, 16);
			}
			AxoRenderContextImpl.getInstance().br$drawString(name, x + 18, y + 6, textColor.get().toInt(), shadow.get());
		}
	}
}
