/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.hud.gui.hud.vanilla;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;
import net.minecraft.unmapped.C_0561170;
import net.minecraft.unmapped.C_0814381;
import net.minecraft.unmapped.C_1028001;
import net.minecraft.unmapped.C_3526030;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_3945557;
import net.minecraft.unmapped.C_4143404;
import net.minecraft.unmapped.C_6476436;
import net.minecraft.unmapped.C_8105098;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Color;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Colors;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.ColorOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.EnumOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.IntegerOption;
import io.github.axolotlclient.bridge.render.AxoRenderContext;
import io.github.axolotlclient.modules.hud.gui.component.DynamicallyPositionable;
import io.github.axolotlclient.modules.hud.gui.entry.TextHudEntry;
import io.github.axolotlclient.modules.hud.gui.layout.AnchorPoint;
import io.github.axolotlclient.modules.hud.util.DefaultOptions;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import io.github.axolotlclient.modules.hud.util.Rectangle;
import io.github.axolotlclient.modules.hud.util.RenderUtil;
import io.github.axolotlclient.util.Util;

/**
 * This implementation of Hud modules is based on KronHUD.
 * <a href="https://github.com/DarkKronicle/KronHUD">Github Link.</a>
 *
 * <p>License: GPL-3.0</p>
 */

public class ScoreboardHud extends TextHudEntry implements DynamicallyPositionable {

	public static final C_0561170 ID = new C_0561170("kronhud", "scoreboardhud");
	private final C_3945557 placeholder = Util.make(() -> {
		C_0814381 placeholderScoreboard = new C_0814381();
		C_3945557 objective = placeholderScoreboard.m_8272217("Scoreboard", C_1028001.f_9119493);
		C_3526030 dark = placeholderScoreboard.m_4288528("DarkKronicle", objective);
		dark.m_9973427(8780);

		C_3526030 moeh = placeholderScoreboard.m_4288528("moehreag", objective);
		moeh.m_9973427(743);

		C_3526030 kode = placeholderScoreboard.m_4288528("TheKodeToad", objective);
		kode.m_9973427(2948);

		placeholderScoreboard.m_9771012(1, objective);

		return objective;
	});

	private final ColorOption backgroundColor = new ColorOption("backgroundcolor", new Color(0x4C000000));
	private final ColorOption topColor = new ColorOption("topbackgroundcolor", new Color(0x66000000));
	private final IntegerOption topPadding = new IntegerOption("toppadding", 0, 0, 4);
	private final BooleanOption scores = new BooleanOption("scores", true);
	private final ColorOption scoreColor = new ColorOption("scorecolor", new Color(0xFFFF5555));
	private final IntegerOption textAlpha = new IntegerOption("text_alpha", 255, 0, 255);
	private final EnumOption<AnchorPoint> anchor = DefaultOptions.getAnchorPoint(AnchorPoint.MIDDLE_RIGHT);

	private final C_8105098 client = (C_8105098) super.client;

	public ScoreboardHud() {
		super(200, 146, true);
	}

	@Override
	public void render(AxoRenderContext context, float delta) {
		C_3754158.m_8373640();
		scale(context);
		renderComponent(context, delta);
		C_3754158.m_2041265();
	}

	@Override
	public void renderComponent(AxoRenderContext context, float delta) {
		C_0814381 scoreboard = this.client.f_4601986.m_5847414();
		C_3945557 scoreboardObjective = null;
		C_4143404 team = scoreboard.m_4614828(this.client.f_7663840.m_0699887().m_6222303());
		if (team != null) {
			int t = team.m_0174211().m_4025272();
			if (t >= 0) {
				scoreboardObjective = scoreboard.m_8757270(3 + t);
			}
		}

		C_3945557 scoreboardObjective2 = scoreboardObjective != null ? scoreboardObjective
			: scoreboard.m_8757270(1);
		if (scoreboardObjective2 != null) {
			this.renderScoreboardSidebar(scoreboardObjective2, false);
		}
	}

	@Override
	public void renderPlaceholderComponent(AxoRenderContext context, float delta) {
		renderScoreboardSidebar(placeholder, true);
	}

	// Abusing this could break some stuff/could allow for unfair advantages. The goal is not to do this, so it won't
	// show any more information than it would have in vanilla.
	private void renderScoreboardSidebar(C_3945557 objective, boolean placeholder) {
		C_0814381 scoreboard = objective.m_2944653();
		Collection<C_3526030> scores = scoreboard.m_0477108(objective);
		List<C_3526030> filteredScores = scores.stream()
			.filter((testScore) -> testScore.m_2012228() != null && !testScore.m_2012228().startsWith("#"))
			.collect(Collectors.toList());

		if (filteredScores.size() > 15) {
			scores = Lists.newArrayList(Iterables.skip(filteredScores, scores.size() - 15));
		} else {
			scores = filteredScores;
		}

		List<C_6476436<C_3526030, String>> scoresWText = Lists.newArrayListWithCapacity(scores.size());
		String text = objective.m_0646168();
		int displayNameWidth = client.f_0426313.m_0040387(text);
		int maxWidth = displayNameWidth;
		int spacerWidth = client.f_0426313.m_0040387(": ");

		C_3526030 scoreboardPlayerScore;
		String formattedText;
		for (Iterator<C_3526030> scoresIterator = scores.iterator(); scoresIterator
			.hasNext(); maxWidth = Math
			.max(maxWidth,
				client.f_0426313.m_0040387(formattedText) + (this.scores.get()
					? spacerWidth + client.f_0426313
					.m_0040387(Integer.toString(scoreboardPlayerScore.m_7979602()))
					: 0))) {
			scoreboardPlayerScore = scoresIterator.next();
			C_4143404 team = scoreboard.m_2322207(scoreboardPlayerScore.m_2012228());
			formattedText = C_4143404.m_3771660(team, scoreboardPlayerScore.m_2012228());
			scoresWText.add(new C_6476436<>(scoreboardPlayerScore, formattedText));
		}
		maxWidth = maxWidth + 6;

		int scoresSize = scores.size();
		int scoreHeight = scoresSize * 9;
		int fullHeight = scoreHeight + 11 + topPadding.get() * 2;

		boolean updated = false;
		if (fullHeight + 1 != height) {
			setHeight(fullHeight + 1);
			updated = true;
		}
		if (maxWidth + 1 != width) {
			setWidth(maxWidth + 1);
			updated = true;
		}
		if (updated) {
			onBoundsUpdate();
		}

		Rectangle bounds = getBounds();

		int renderX = bounds.x() + bounds.width() - maxWidth;
		int renderY = bounds.y() + (bounds.height() / 2 - fullHeight / 2) + 1;

		int scoreX = renderX + 4;
		int scoreY = renderY + scoreHeight + 10;
		int num = 0;
		int textOffset = scoreX - 4;

		for (C_6476436<C_3526030, String> scoreboardPlayerScoreTextPair : scoresWText) {
			++num;
			C_3526030 scoreboardPlayerScore2 = scoreboardPlayerScoreTextPair.m_6146952();
			String scoreText = scoreboardPlayerScoreTextPair.m_5167526();
			String score = String.valueOf(scoreboardPlayerScore2.m_7979602());
			int relativeY = scoreY - num * 9 + topPadding.get() * 2;

			if (background.get() && backgroundColor.get().getAlpha() > 0 && !placeholder) {
				if (num == scoresSize) {
					RenderUtil.drawRectangle(textOffset, relativeY - 1, maxWidth, 10, backgroundColor.get().toInt());
				} else if (num == 1) {
					RenderUtil.drawRectangle(textOffset, relativeY, maxWidth, 10, backgroundColor.get());
				} else {
					RenderUtil.drawRectangle(textOffset, relativeY, maxWidth, 9, backgroundColor.get());
				}
			}

			if (shadow.get()) {
				client.f_0426313.m_1950885(scoreText, (float) scoreX, (float) relativeY, Colors.WHITE.withAlpha(textAlpha.get()).toInt());
			} else {
				client.f_0426313.m_4683674(scoreText, scoreX, relativeY, Colors.WHITE.withAlpha(textAlpha.get()).toInt());
			}
			if (this.scores.get()) {
				DrawUtil.drawString(score, (float) (scoreX + maxWidth - client.f_0426313.m_0040387(score) - 6),
					(float) relativeY, scoreColor.get().toInt(), shadow.get());
			}
			if (num == scoresSize) {
				// Draw the title
				if (background.get() && !placeholder) {
					RenderUtil.drawRectangle(textOffset, relativeY - 10 - topPadding.get() * 2 - 1, maxWidth,
						10 + topPadding.get() * 2, topColor.get());
				}
				float title = (renderX + (maxWidth - displayNameWidth) / 2F);
				if (shadow.get()) {
					client.f_0426313.m_1950885(text, title, (float) (relativeY - 9) - topPadding.get(), Colors.WHITE.withAlpha(textAlpha.get()).toInt());
				} else {
					client.f_0426313.m_4683674(text, (int) title, (relativeY - 9), Colors.WHITE.withAlpha(textAlpha.get()).toInt());
				}
			}
		}

		if (outline.get() && outlineColor.get().getAlpha() > 0 && !placeholder) {
			RenderUtil.drawOutline(textOffset, bounds.y(), maxWidth, fullHeight + 1, outlineColor.get());
		}
	}

	@Override
	public List<Option<?>> getConfigurationOptions() {
		List<Option<?>> options = new ArrayList<>();
		options.add(enabled);
		options.add(scale);
		options.add(textColor);
		options.add(shadow);
		options.add(background);
		options.add(backgroundColor);
		options.add(outline);
		options.add(outlineColor);
		options.add(hide);
		options.add(topColor);
		options.add(scores);
		options.add(scoreColor);
		options.add(anchor);
		options.add(topPadding);
		options.remove(textColor);
		options.add(textAlpha);
		return options;
	}

	@Override
	public C_0561170 getId() {
		return ID;
	}

	@Override
	public AnchorPoint getAnchor() {
		return anchor.get();
	}

	@Override
	public double getDefaultX() {
		return 1.0;
	}

	@Override
	public double getDefaultY() {
		return 0.5;
	}
}
