/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.hud.gui.keystrokes;

import java.util.Collection;
import java.util.List;
import java.util.function.BooleanSupplier;
import java.util.function.Function;

import com.google.common.collect.ImmutableList;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Colors;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.IntegerOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.ButtonWidget;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.ClickableWidget;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.TextFieldWidget;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.vanilla.widgets.IntegerWidget;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.vanilla.widgets.VanillaButtonWidget;
import io.github.axolotlclient.modules.hud.gui.hud.KeystrokeHud;
import io.github.axolotlclient.modules.hud.gui.layout.Justification;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import lombok.Getter;
import net.minecraft.unmapped.C_1331819;
import net.minecraft.unmapped.C_3020744;
import net.minecraft.unmapped.C_3390001;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_3831727;
import net.minecraft.unmapped.C_4976084;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.input.Keyboard;

public class ConfigureKeyBindScreen extends io.github.axolotlclient.AxolotlClientConfig.impl.ui.Screen {

	private final C_3020744 parent;
	private final KeystrokeHud hud;
	public final KeystrokeHud.Keystroke stroke;
	private final IntegerOption width;
	private final IntegerOption height;
	private final boolean isAddScreen;

	public ConfigureKeyBindScreen(C_3020744 parent, KeystrokeHud hud, KeystrokeHud.Keystroke stroke, boolean isAddScreen) {
		super(C_3390001.m_2053009("keystrokes.stroke.configure_stroke"));
		this.parent = parent;
		this.hud = hud;
		this.stroke = stroke;

		width = new IntegerOption("", stroke.getBounds().width(), v -> stroke.getBounds().width(v), 7, 100);
		height = new IntegerOption("", stroke.getBounds().height(), v -> stroke.getBounds().height(v), 7, 100);
		this.isAddScreen = isAddScreen;
	}

	@Override
	public void m_3593494() {
		if (hud.keystrokes == null) {
			hud.setKeystrokes();
		}

		int leftColX = super.f_5465691 / 2 - 4 - 150;
		int leftColY = 36 + 5;
		int rightColX = super.f_5465691 / 2 + 4;
		int rightColY = 36;

		addDrawableChild(new ClickableWidget(super.f_5465691 / 2 - 100, rightColY, 200, 40, "") {
			@Override
			protected void drawWidget(int mouseX, int mouseY, float partialTick) {
				var rect = stroke.getRenderPosition();
				C_3754158.m_8373640();
				C_3754158.m_3172490(getX(), getY(), 0);
				float scale = Math.min((float) getHeight() / rect.height(), (float) getWidth() / rect.width());
				C_3754158.m_3172490(getWidth() / 2f - (rect.width() * scale) / 2f, 0, 0);
				C_3754158.m_4552250(scale, scale, 1);
				C_3754158.m_3172490(-rect.x(), -rect.y(), 0);
				DrawUtil.fillRect(rect, Colors.WHITE.withAlpha(128));
				stroke.render();
				C_3754158.m_2041265();
			}
		}).active = false;
		leftColY += 48;
		rightColY += 48;

		ClickableWidget currentKey = addDrawable(textWidget(0, rightColY, super.f_5465691, 9, "", f_2020658));
		if (stroke.getKey() != null) {
			currentKey.setMessage(C_3390001.m_2053009("keystrokes.stroke.key", C_1331819.m_9293214(stroke.getKey().getKeyCode()), C_3390001.m_2053009(stroke.getKey().getName())));
		} else {
			currentKey.setMessage("");
		}
		leftColY += 9 + 8;
		rightColY += 9 + 8;

		if (stroke.isLabelEditable()) {
			addDrawableChild(textWidget(leftColX, leftColY, 150, 20, C_3390001.m_2053009("keystrokes.stroke.label"), f_2020658));
			leftColY += 28;
			boolean supportsSynchronization = stroke instanceof KeystrokeHud.LabelKeystroke;

			var label = addDrawableChild(new TextFieldWidget(f_2020658, rightColX, rightColY, supportsSynchronization ? 30 : 150, 20, ""));

			label.setText(stroke.getLabel());
			label.setChangedListener(stroke::setLabel);
			if (supportsSynchronization) {
				var s = (KeystrokeHud.LabelKeystroke) stroke;
				ButtonWidget synchronizeButton = addDrawableChild(new VanillaButtonWidget(rightColX + 30 + 4, rightColY, 58, 20, C_3390001.m_2053009("keystrokes.stroke.label.synchronize_with_key", s.isSynchronizeLabel() ? C_3390001.m_2053009("options.on") : C_3390001.m_2053009("options.off")), b -> {
					s.setSynchronizeLabel(!s.isSynchronizeLabel());
					b.setMessage(I18n.translate("keystrokes.stroke.label.synchronize_with_key", s.isSynchronizeLabel() ? I18n.translate("options.on") : I18n.translate("options.off")));
					label.setEditable(!s.isSynchronizeLabel());
					if (s.isSynchronizeLabel()) {
						label.setText(stroke.getLabel());
					}
				}));
				synchronizeButton.active = s.getKey() != null;
				label.setEditable(!s.isSynchronizeLabel());
				addDrawableChild(CyclingButtonWidget.<Justification>builder(j -> C_3390001.m_2053009(j.toString())).values(Justification.values())
					.initially(s.getJustification()).build(rightColX + 30 + 4 + 58 + 4, rightColY, 58, 20,
						C_3390001.m_2053009("justification"), (btn, val) -> s.setJustification(val)));
			}
			rightColY += 28;
		}
		addDrawableChild(textWidget(leftColX, leftColY, 150, 20, C_3390001.m_2053009("keystrokes.stroke.width"), f_2020658));
		leftColY += 28;
		addDrawableChild(new IntegerWidget(rightColX, rightColY, 150, 20, width));
		rightColY += 28;
		addDrawableChild(textWidget(leftColX, leftColY, 150, 20, C_3390001.m_2053009("keystrokes.stroke.height"), f_2020658));
		leftColY += 28;
		addDrawableChild(new IntegerWidget(rightColX, rightColY, 150, 20, height));

		rightColY += 28;

		addDrawableChild(new VanillaButtonWidget(super.f_5465691 / 2 - 150 - 4, rightColY, 150, 20, C_3390001.m_2053009("keystrokes.stroke.configure_key"), b -> {
			f_7153641.m_6408915(new KeyBindSelectionScreen(this, stroke));
		}));
		addDrawableChild(new VanillaButtonWidget(super.f_5465691 / 2 + 4, rightColY, 150, 20, C_3390001.m_2053009("keystrokes.stroke.configure_position"), b -> {
			f_7153641.m_6408915(new KeystrokePositioningScreen(this, hud, stroke));
		}));


		if (isAddScreen) {
			ButtonWidget addButton = addDrawableChild(new VanillaButtonWidget(super.f_5465691 / 2 - 150 - 4, super.f_3080061 - 33 / 2 - 10, 150, 20, C_3390001.m_2053009("keystrokes.stroke.add"), b -> {
				hud.keystrokes.add(stroke);
				closeScreen();
			}));
			addButton.active = stroke.getKey() != null;
		}
		addDrawableChild(new VanillaButtonWidget(isAddScreen ? super.f_5465691 / 2 + 4 : super.f_5465691 / 2 - 75, super.f_3080061 - 33 / 2 - 10, 150, 20, isAddScreen ? C_3390001.m_2053009("gui.cancel") : C_3390001.m_2053009("gui.back"), b -> closeScreen()));
	}

	@Override
	public void m_7261014(int mouseX, int mouseY, float delta) {
		super.m_7261014(mouseX, mouseY, delta);
		m_2717572(f_2020658, getTitle(), super.f_5465691 / 2, 33 / 2 - f_2020658.f_6725889 / 2, -1);
	}


	public void closeScreen() {
		f_7153641.m_6408915(parent);
		hud.saveKeystrokes();
	}

	@Override
	public boolean keyPressed(int keyCode, int scanCode, int modifiers) {
		if (keyCode == Keyboard.KEY_ESCAPE) {
			closeScreen();
			return true;
		}
		return super.keyPressed(keyCode, scanCode, modifiers);
	}

	private static ClickableWidget textWidget(int x, int y, int width, int height, String message, C_3831727 textRenderer) {
		return new ClickableWidget(x, y, width, height, message) {
			@Override
			public void render(int mouseX, int mouseY, float delta) {
				m_2717572(textRenderer, getMessage(), getX() + getWidth() / 2, getY() + getHeight() / 2 - textRenderer.f_6725889 / 2, -1);
			}
		};
	}
}

class CyclingButtonWidget<T> extends VanillaButtonWidget {
	private final String optionText;
	private int index;
	@Getter
	private T value;
	private final CyclingButtonWidget.Values<T> values;
	private final Function<T, String> valueToText;
	private final CyclingButtonWidget.UpdateCallback<T> callback;

	CyclingButtonWidget(int x, int y, int width, int height, String message, String optionText, int index, T value,
						CyclingButtonWidget.Values<T> values, Function<T, String> valueToText,
						CyclingButtonWidget.UpdateCallback<T> callback) {
		super(x, y, width, height, message, btn -> {
		});
		this.optionText = optionText;
		this.index = index;
		this.value = value;
		this.values = values;
		this.valueToText = valueToText;
		this.callback = callback;
	}

	@Override
	public void onPress() {
		if (C_3020744.m_5142737()) {
			this.cycle(-1);
		} else {
			this.cycle(1);
		}
	}

	private void cycle(int amount) {
		List<T> list = this.values.getCurrent();
		this.index = C_4976084.m_6242245(this.index + amount, list.size());
		T object = list.get(this.index);
		this.internalSetValue(object);
		this.callback.onValueChange(this, object);
	}

	@Override
	public boolean mouseScrolled(double mouseX, double mouseY, double amountX, double amount) {
		if (amount > 0.0) {
			this.cycle(-1);
		} else if (amount < 0.0) {
			this.cycle(1);
		}

		return true;
	}

	public void setValue(T value) {
		List<T> list = this.values.getCurrent();
		int i = list.indexOf(value);
		if (i != -1) {
			this.index = i;
		}

		this.internalSetValue(value);
	}

	private void internalSetValue(T value) {
		String text = this.composeText(value);
		this.setMessage(text);
		this.value = value;
	}

	private String composeText(T value) {
		return this.composeGenericOptionText(value);
	}

	private String composeGenericOptionText(T value) {
		return this.optionText + ": " + this.valueToText.apply(value);
	}

	static <T> CyclingButtonWidget.Builder<T> builder(Function<T, String> valueToText) {
		return new CyclingButtonWidget.Builder<>(valueToText);
	}

	static class Builder<T> {
		private int initialIndex;
		@Nullable
		private T value;
		private final Function<T, String> valueToText;
		private CyclingButtonWidget.Values<T> values = CyclingButtonWidget.Values.of(ImmutableList.<T>of());

		public Builder(Function<T, String> valueToText) {
			this.valueToText = valueToText;
		}

		public CyclingButtonWidget.Builder<T> values(Collection<T> values) {
			return this.values(CyclingButtonWidget.Values.of(values));
		}

		@SafeVarargs
		public final CyclingButtonWidget.Builder<T> values(T... values) {
			return this.values(ImmutableList.copyOf(values));
		}

		public CyclingButtonWidget.Builder<T> values(CyclingButtonWidget.Values<T> values) {
			this.values = values;
			return this;
		}

		public CyclingButtonWidget.Builder<T> initially(T value) {
			this.value = value;
			int i = this.values.getDefaults().indexOf(value);
			if (i != -1) {
				this.initialIndex = i;
			}

			return this;
		}

		public CyclingButtonWidget<T> build(int x, int y, int width, int height, String optionText, CyclingButtonWidget.UpdateCallback<T> callback) {
			List<T> list = this.values.getDefaults();
			if (list.isEmpty()) {
				throw new IllegalStateException("No values for cycle button");
			} else {
				T object = this.value != null ? this.value : list.get(this.initialIndex);
				String text = this.valueToText.apply(object);
				String text2 = optionText + ": " + text;
				return new CyclingButtonWidget<>(
					x,
					y,
					width,
					height,
					text2,
					optionText,
					this.initialIndex,
					object,
					this.values,
					this.valueToText,
					callback
				);
			}
		}
	}

	interface UpdateCallback<T> {
		void onValueChange(CyclingButtonWidget<T> cyclingButtonWidget, T object);
	}

	interface Values<T> {
		List<T> getCurrent();

		List<T> getDefaults();

		static <T> CyclingButtonWidget.Values<T> of(Collection<T> values) {
			final List<T> list = ImmutableList.copyOf(values);
			return new CyclingButtonWidget.Values<>() {
				@Override
				public List<T> getCurrent() {
					return list;
				}

				@Override
				public List<T> getDefaults() {
					return list;
				}
			};
		}

		static <T> CyclingButtonWidget.Values<T> of(BooleanSupplier alternativeToggle, List<T> defaults, List<T> alternatives) {
			final List<T> list = ImmutableList.copyOf(defaults);
			final List<T> list2 = ImmutableList.copyOf(alternatives);
			return new CyclingButtonWidget.Values<>() {
				@Override
				public List<T> getCurrent() {
					return alternativeToggle.getAsBoolean() ? list2 : list;
				}

				@Override
				public List<T> getDefaults() {
					return list;
				}
			};
		}
	}
}
