/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.particles;

import java.util.*;

import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.api.options.OptionCategory;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Color;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.ColorOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.IntegerOption;
import io.github.axolotlclient.mixin.ParticleAccessor;
import io.github.axolotlclient.modules.AbstractModule;
import io.github.axolotlclient.util.ClientColors;
import io.github.axolotlclient.util.Util;
import lombok.Getter;
import net.minecraft.unmapped.C_1350658;
import net.minecraft.unmapped.C_4701530;
import org.apache.commons.lang3.StringUtils;

public class Particles extends AbstractModule {

	@Getter
	private static final Particles Instance = new Particles();

	public final HashMap<C_4701530, HashMap<String, Option<?>>> particleOptions = new HashMap<>();
	public final WeakHashMap<C_1350658, C_4701530> particleMap = new WeakHashMap<>();

	private final OptionCategory cat = OptionCategory.create("particles");
	private final BooleanOption enabled = new BooleanOption("enabled", false);

	@Override
	public void init() {
		cat.add(enabled);

		addParticleOptions();
		AxolotlClient.config().rendering.add(cat);
	}

	private void addParticleOptions() {
		for (C_4701530 type : Arrays.stream(C_4701530.values()).sorted(new AlphabeticalComparator())
			.toList()) {
			OptionCategory category = OptionCategory.create(
				StringUtils.capitalize(Util.splitAtCapitalLetters(type.m_4736377().replace("_", ""))));
			HashMap<String, Option<?>> optionsByKey = new LinkedHashMap<>();

			populateMap(optionsByKey, new BooleanOption("showParticle", true),
				new IntegerOption("count", 1, 1, 20),
				new BooleanOption("customColor", false),
				new ColorOption("color", "particles.color.tooltip", ClientColors.WHITE));

			if (type == C_4701530.f_9917373 || type == C_4701530.f_5110895) {
				populateMap(optionsByKey, new BooleanOption("alwaysCrit", false));
			}

			optionsByKey.values().forEach(category::add);
			particleOptions.put(type, optionsByKey);

			cat.add(category);
		}
	}

	private void populateMap(HashMap<String, Option<?>> map, Option<?>... options) {
		for (Option<?> option : options) {
			map.put(option.getName(), option);
		}
	}

	public void applyOptions(C_1350658 particle) {
		if (enabled.get() && particleMap.containsKey(particle)) {
			C_4701530 type = particleMap.get(particle);
			if (particleOptions.containsKey(type)) {
				HashMap<String, Option<?>> options = particleOptions.get(type);

				if (((BooleanOption) options.get("customColor")).get()) {
					Color color = ((ColorOption) options.get("color")).get();
					particle.m_9240833(color.getRed() / 255F, color.getGreen() / 255F, color.getBlue() / 255F);
					((ParticleAccessor) particle).setAlpha(color.getAlpha() / 255F);
				}
			}
		}
	}

	public int getMultiplier(C_4701530 type) {
		if (enabled.get() && particleOptions.containsKey(type)) {
			HashMap<String, Option<?>> options = particleOptions.get(type);

			return ((IntegerOption) options.get("count")).get();
		}
		return 1;
	}

	public boolean getAlwaysOn(C_4701530 type) {
		return enabled.get() && particleOptions.containsKey(type)
			&& ((BooleanOption) Particles.getInstance().particleOptions.get(type).get("alwaysCrit"))
			.get();
	}

	public boolean getShowParticle(C_4701530 type) {
		return enabled.get() && particleOptions.containsKey(type)
			? ((BooleanOption) Particles.getInstance().particleOptions.get(type).get("showParticle")).get()
			: true;
	}

	protected static class AlphabeticalComparator implements Comparator<C_4701530> {

		// Function to compare
		public int compare(C_4701530 s1, C_4701530 s2) {
			if (s1.m_4736377().equals(s2.m_4736377()))
				return 0;
			String[] strings = {s1.m_4736377(), s2.m_4736377()};
			Arrays.sort(strings, Collections.reverseOrder());

			if (strings[0].equals(s1.m_4736377()))
				return 1;
			else
				return -1;
		}
	}
}
