/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.screenshotUtils;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Colors;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.api.requests.FriendRequest;
import io.github.axolotlclient.api.requests.UserRequest;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import io.github.axolotlclient.util.Util;
import io.github.axolotlclient.util.Watcher;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.unmapped.C_1630332;
import net.minecraft.unmapped.C_2348249;
import net.minecraft.unmapped.C_3020744;
import net.minecraft.unmapped.C_3390001;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_3831727;
import net.minecraft.unmapped.C_4461663;
import net.minecraft.unmapped.C_4976084;
import net.minecraft.unmapped.C_5786166;
import net.minecraft.unmapped.C_8105098;
import net.minecraft.unmapped.C_8373595;
import org.lwjgl.opengl.GL11;

public class GalleryScreen extends C_3020744 {

	public static final Path SCREENSHOTS_DIR = FabricLoader.getInstance().getGameDir().resolve("screenshots");

	private record Tab<T>(String title, Callable<List<T>> list, Map<T, ImageInstance> loadingCache,
						  Loader<T> loader) {

		private static <T> Tab<T> of(String title, Callable<List<T>> list, Loader<T> loader) {
			var cache = new HashMap<T, ImageInstance>();
			return new Tab<>(title, list, cache, o -> {
				if (cache.containsKey(o)) {
					return cache.get(o);
				}
				var val = loader.load(o);
				cache.put(o, val);
				return val;
			});
		}

		private static final Tab<Path> LOCAL = of(C_3390001.m_2053009("gallery.title.local"), () -> {
			try (Stream<Path> screenshots = Files.list(SCREENSHOTS_DIR)) {
				return screenshots.filter(Files::isRegularFile).sorted(Comparator.<Path>comparingLong(p -> {
					try {
						return Files.getLastModifiedTime(p).toMillis();
					} catch (IOException e) {
						return 0L;
					}
				}).reversed()).toList();
			}
		}, ImageInstance.LocalImpl::new);

		private static final Tab<String> SHARED = of(C_3390001.m_2053009("gallery.title.shared"), () ->
			FriendRequest.getInstance().getFriendUuids()
				.thenApply(res -> res.stream().map(UserRequest::getUploadedImages)
					.map(CompletableFuture::join)
					.filter(Optional::isPresent)
					.map(Optional::get)
					.reduce(new ArrayList<>(), (l1, l2) -> {
						l1.addAll(l2);
						return l1;
					})).join(), url -> ImageShare.getInstance().downloadImage(url).join());

		interface Loader<T> {
			ImageInstance load(T obj) throws Exception;
		}
	}

	private Tab<?> current;

	private final C_3020744 parent;
	private final Watcher watcher;
	private final String title;

	public GalleryScreen(C_3020744 parent) {
		super();
		title = C_3390001.m_2053009("gallery.title");
		this.parent = parent;
		this.current = Tab.LOCAL;
		this.watcher = Watcher.createSelfTicking(SCREENSHOTS_DIR, () -> {
			if (current == Tab.LOCAL) {
				m_0116202(f_7153641, f_5465691, f_3080061);
			}
		});
	}

	private static final int entrySpacing = 4,
		entryWidth = 100,
		entryHeight = 75,
		marginLeftRight = 10;

	private boolean isError, online;
	private ImageList area;

	@Override
	public void m_7261014(int mouseX, int mouseY, float delta) {
		m_7817195();
		area.m_9734698(mouseX, mouseY, delta);
		super.m_7261014(mouseX, mouseY, delta);

		if (online) {
			m_2717572(f_2020658, title, f_5465691 / 2, 40 / 2 - 2 - f_2020658.f_6725889, -1);
			m_2717572(f_2020658, current.title(), f_5465691 / 2, 40 / 2 + 2, -1);
		} else {
			m_2717572(f_2020658, title, f_5465691 / 2, 40 / 2 - f_2020658.f_6725889 / 2, -1);
		}

		if (isError) {
			m_2717572(f_2020658, C_3390001.m_2053009("gallery.error.loading"), f_5465691 / 2, 36, -1);
		}
	}

	@Override
	public void m_3356138() {
		super.m_3356138();
		area.m_1002325();
	}

	@Override
	protected void m_7362766(int i, int j, int k) {
		super.m_7362766(i, j, k);
		area.m_7379893(i, j, k);
	}

	@Override
	protected void m_5308748(int i, int j, int k) {
		super.m_5308748(i, j, k);
		area.m_9825864(i, j, k);
	}

	@Override
	public void m_3593494() {
		online = API.getInstance().isAuthenticated();

		int columnCount = (f_5465691 - (marginLeftRight * 2) + entrySpacing - 13) / (entryWidth + entrySpacing); // -13 to always have enough space for the scrollbar

		area = new ImageList(f_7153641, f_5465691, f_3080061, 33, f_3080061 - 40, entryHeight + entrySpacing, columnCount);


		CompletableFuture.runAsync(() -> {
			try {
				loadTab(current, columnCount, area);
			} catch (Exception e) {
				isError = true;
				f_2213969.add(new C_2348249(0, f_5465691 / 2 - 75, 36 + f_2020658.f_6725889 + 8, 150, 20, C_3390001.m_2053009("gallery.reload")));
			}
		});

		int buttonWidth = columnCount <= 5 && online ? 100 : 150;
		int footerButtonX = online ? f_5465691 / 2 - buttonWidth - buttonWidth / 2 - 4 : f_5465691 / 2 - buttonWidth - 2;
		int footerButtonY = f_3080061 - 33 / 2 - 10;
		if (online) {
			C_2348249 switchTab;
			if (current == Tab.SHARED) {
				switchTab = new C_2348249(1, footerButtonX, footerButtonY, buttonWidth, 20, C_3390001.m_2053009("gallery.tab.local"));
			} else {
				switchTab = new C_2348249(2, footerButtonX, footerButtonY, buttonWidth, 20, C_3390001.m_2053009("gallery.tab.shared"));
			}
			f_2213969.add(switchTab);
			footerButtonX += buttonWidth + 4;
		}
		f_2213969.add(new C_2348249(3, footerButtonX, footerButtonY, buttonWidth, 20, C_3390001.m_2053009("gallery.download_external")));
		footerButtonX += buttonWidth + 4;
		f_2213969.add(new C_2348249(4, footerButtonX, footerButtonY, buttonWidth, 20, C_3390001.m_2053009("gui.back")));
	}

	@Override
	protected void m_7971793(C_2348249 buttonWidget) {
		switch (buttonWidget.f_5920996) {
			case 0 -> m_0116202(f_7153641, f_5465691, f_3080061);
			case 1 -> setTab(Tab.LOCAL);
			case 2 -> setTab(Tab.SHARED);
			case 3 -> f_7153641.m_6408915(new DownloadImageScreen(this));
			case 4 -> {
				Tab.LOCAL.loadingCache().forEach((path, instance) -> {
					if (instance != null) {
						f_7153641.m_1218956().m_3775266(instance.id());
					}
				});
				Tab.LOCAL.loadingCache().clear();
				Tab.SHARED.loadingCache().forEach((s, instance) -> {
					if (instance != null) {
						f_7153641.m_1218956().m_3775266(instance.id());
					}
				});
				Tab.SHARED.loadingCache().clear();
				Watcher.close(watcher);
				f_7153641.m_6408915(parent);
			}
		}
	}

	private void setTab(Tab<?> tab) {
		current = tab;
		m_0116202(f_7153641, f_5465691, f_3080061);
	}

	private <T> void loadTab(Tab<T> tab, int columnCount, ImageList area) throws Exception {
		List<T> images = tab.list.call();
		int size = images.size();
		for (int i = 0; i < size; i += columnCount) {
			ImageListEntry row = new ImageListEntry(columnCount, area);
			area.addEntry(row);
			for (int x = 0; x < columnCount; x++) {
				if (i + x >= size) {
					break;
				}
				T p = images.get(i + x);
				var entry = new ImageEntry(entryWidth, entryHeight, () -> tab.loader.load(p), row);
				row.add(entry);
			}
		}
	}

	// This image list is loading its entries lazily! :)
	private class ImageEntry extends C_2348249 {

		private static final int bgColor = Colors.DARK_GRAY.toInt();
		private static final int accent = Colors.GRAY.withBrightness(0.5f).withAlpha(128).toInt();

		private final C_3831727 font;
		private final Callable<ImageInstance> instanceSupplier;
		private final ImageListEntry row;
		private long loadStart;
		private CompletableFuture<ImageInstance> future;

		protected ImageEntry(int width, int height, Callable<ImageInstance> instanceSupplier, ImageListEntry row) {
			super(99, 0, 0, width, height, "");
			this.instanceSupplier = instanceSupplier;
			this.row = row;
			this.font = C_8105098.m_0408063().f_0426313;
		}

		private CompletableFuture<ImageInstance> load() {
			if (future == null) {
				loadStart = C_8105098.m_6224581();
				future = CompletableFuture.supplyAsync(() -> {
					try {
						var instance = instanceSupplier.call();
						f_4865617 = instance.filename();
						return instance;
					} catch (Exception e) {
						f_7153641.m_2167033(() -> row.remove(this));
						return null;
					}
				});
			}
			return future;
		}

		public void onPress() {
			f_7153641.m_6408915(ImageScreen.create(GalleryScreen.this, load(), false));
			m_1301911(f_7153641.m_7555106());
		}

		@Override
		public void m_5350167(C_8105098 client, int mouseX, int mouseY) {
			this.f_8690685 = mouseX >= this.f_3923091 && mouseY >= this.f_8532347 && mouseX < this.f_3923091 + this.f_9527567 && mouseY < this.f_8532347 + this.f_7845401;
			if (load().isDone() && load().join() != null && client.m_1218956().m_1605001(load().join().id()) != null) {
				client.m_1218956().m_5325521(load().join().id());
				C_3754158.m_6326777(1, 1, 1);
				m_5935491(f_3923091, f_8532347, 0, 0, m_8567079(), getHeight() - font.f_6725889 - 2, m_8567079(), getHeight() - font.f_6725889 - 2);
				drawScrollingText(font, 2, -1);
			} else {
				float delta = (float) easeInOutCubic((C_8105098.m_6224581() - loadStart) % 1000f / 1000f);

				m_7865719(getX() + 2, getY() + 2, getXEnd() - 2, getYEnd() - font.f_6725889 - 2, bgColor);
				drawHorizontalGradient(getX() + 2, getY() + 2, getYEnd() - font.f_6725889 - 2, lerp(delta, getX() + 2, getXEnd() - 2));

				m_7865719(getX() + 2, getYEnd() - font.f_6725889 - 1, getXEnd() - 2, getYEnd() - 2, bgColor);
				drawHorizontalGradient(getX() + 2, getYEnd() - font.f_6725889 - 1, getYEnd() - 2, lerp(delta, getX() + 2, getXEnd() - 2));
			}
			DrawUtil.outlineRect(getX(), getY(), m_8567079(), getHeight(), row.list.isInListContent(mouseX, mouseY) && m_7131232() ? -1 : bgColor);
		}

		private int getX() {
			return f_3923091;
		}

		private int getY() {
			return f_8532347;
		}

		protected int getYEnd() {
			return f_8532347 + getHeight();
		}

		private int getHeight() {
			return f_7845401;
		}

		protected int getXEnd() {
			return f_3923091 + m_8567079();
		}

		private void drawHorizontalGradient(int x1, int y1, int y2, int x2) {
			C_8373595 consumer = C_5786166.m_2065116().m_1454391();
			consumer.m_0421390(GL11.GL_QUADS, C_4461663.f_8459667);
			consumer.m_3299851(x1, y1, 0).m_9724942(bgColor >> 16 & 255, bgColor >> 8 & 255, bgColor & 255, bgColor >> 24 & 255);
			consumer.m_3299851(x1, y2, 0).m_9724942(bgColor >> 16 & 255, bgColor >> 8 & 255, bgColor & 255, bgColor >> 24 & 255);
			consumer.m_3299851(x2, y2, 0).m_9724942(accent >> 16 & 255, accent >> 8 & 255, accent & 255, accent >> 24 & 255);
			consumer.m_3299851(x2, y1, 0).m_9724942(accent >> 16 & 255, accent >> 8 & 255, accent & 255, accent >> 24 & 255);
			C_5786166.m_2065116().m_8222644();
		}

		private double easeInOutCubic(double x) {
			return x < 0.5 ? 4 * x * x * x : 1 - Math.pow(-2 * x + 2, 3) / 2;
		}

		private int lerp(float delta, int start, int end) {
			return (int) C_4976084.m_7164829(Util.lerp(delta, start, end), start, end);
		}

		protected void drawScrollingText(C_3831727 font, int offset, int color) {
			int i = this.f_3923091 + offset;
			int j = this.f_3923091 + this.m_8567079() - offset;
			DrawUtil.drawScrollableText(font, this.f_4865617, i, this.f_8532347 + getHeight() - font.f_6725889 - 1, j, this.f_8532347 + this.getHeight(), color);
		}
	}

	private static class ImageListEntry implements C_1630332.C_8277969 {

		private final List<ImageEntry> buttons;
		private final int size;
		private final ImageList list;

		public ImageListEntry(int size, ImageList list) {
			this.size = size;
			buttons = new ArrayList<>(size);
			this.list = list;
		}

		@Override
		public void m_2013749(int index, int left, int top, int width, int height, int mouseX, int mouseY, boolean hovered) {
			if (Math.max(left, list.getRowLeft()) <= Math.min(left + width, list.getRowLeft() + list.getWidth()) - 1 &&
				Math.max(top - height, list.getY()) <= Math.min(top + height * 2, list.getY() + list.viewHeight()) - 1) {
				buttons.forEach(e -> {
					e.f_8532347 = top;
					e.m_5350167(C_8105098.m_0408063(), mouseX, mouseY);
				});
			} else {
				buttons.forEach(e -> {
					e.f_8532347 = top;
				});
			}
		}

		public void add(ImageEntry e) {
			buttons.add(e);
			repositionButtons();
		}

		public void remove(ImageEntry e) {
			buttons.remove(e);
			if (buttons.isEmpty()) {
				list.removeEntry(this);
			} else if (buttons.size() < size) {
				list.shiftEntries(this);
			}
			repositionButtons();
		}

		private void repositionButtons() {
			int x = list.getRowLeft();
			for (ImageEntry e : buttons) {
				e.f_3923091 = x;
				x += e.m_8567079() + entrySpacing;
			}
		}

		public ImageEntry pop() {
			var entry = buttons.remove(0);
			if (buttons.isEmpty()) {
				list.removeEntry(this);
				list.m_9909453(0);
			} else if (buttons.size() < size) {
				list.shiftEntries(this);
			}
			repositionButtons();
			return entry;
		}

		@Override
		public void m_1433237(int i, int j, int k) {

		}

		@Override
		public boolean m_3739705(int i, int j, int k, int l, int m, int n) {
			Optional<ImageEntry> present = buttons.stream().filter(C_2348249::m_7131232).findFirst();
			present.ifPresent(ImageEntry::onPress);
			return present.isPresent();
		}

		@Override
		public void m_4703785(int i, int j, int k, int l, int m, int n) {

		}
	}

	private static class ImageList extends C_1630332 {

		private final int rowWidth;
		private final List<ImageListEntry> entries = new ArrayList<>();

		public ImageList(C_8105098 minecraft, int screenWidth, int screenHeight, int top, int bottom, int entryHeight, int columns) {
			super(minecraft, screenWidth, screenHeight, top, bottom, entryHeight);
			this.rowWidth = columns * (entryWidth + entrySpacing) - entrySpacing;
		}

		public void addEntry(ImageListEntry entry) {
			entries.add(entry);
		}

		@Override
		public int m_6519286() {
			return rowWidth;
		}

		public int getRowLeft() {
			return this.f_8129586 + this.f_2695432 / 2 - this.m_6519286() / 2;
		}

		public void removeEntry(ImageListEntry entry) {
			entries.remove(entry);
		}

		public void shiftEntries(ImageListEntry origin) {
			int originIndex = entries.indexOf(origin);
			int lastIndex = entries.size() - 1;
			if (originIndex == lastIndex) {
				return;
			}
			ImageListEntry next = entries.get(originIndex + 1);
			origin.add(next.pop());
		}

		@Override
		protected int m_3021017() {
			return getRowLeft() + m_6519286() + 6 + 2;
		}

		public boolean isInListContent(int x, int y) {
			return x >= f_8129586 && x < f_7901480 && y >= f_9975269 && y < f_9145833;
		}

		public int getY() {
			return f_9975269;
		}

		public int getWidth() {
			return f_2695432;
		}

		@Override
		protected int m_3791031() {
			return entries.size();
		}

		public int viewHeight() {
			return f_9145833 - f_9975269;
		}

		@Override
		public C_8277969 m_7118949(int i) {
			return entries.get(i);
		}
	}
}
