/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.screenshotUtils;

import javax.imageio.ImageIO;
import net.minecraft.unmapped.C_0561170;
import net.minecraft.unmapped.C_3779718;
import net.minecraft.unmapped.C_8105098;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.util.Locale;

import com.google.common.hash.Hashing;

public interface ImageInstance {

	C_0561170 id();

	BufferedImage image();

	String filename();

	interface Local extends ImageInstance {
		Path location();

		default ImageInstance toShared(String url, String uploader, Instant sharedAt) {
			return new SharedImpl(id(), image(), filename(), location(), url, uploader, sharedAt);
		}
	}

	interface Remote extends ImageInstance {
		String url();

		String uploader();

		Instant sharedAt();

		default ImageInstance toShared(Path saved) {
			return new SharedImpl(id(), image(), filename(), saved, url(), uploader(), sharedAt());
		}
	}

	private static void register(C_0561170 id, BufferedImage img) {
		C_8105098.m_0408063().m_7581301(() -> C_8105098.m_0408063().m_1218956().m_0381205(id, new C_3779718(img)));
	}

	record LocalImpl(C_0561170 id, BufferedImage image, String filename, Path location) implements Local {
		public LocalImpl(BufferedImage image, String filename, Path location) {
			this(new C_0561170("gallery_local_" + Hashing.sha256().hashUnencodedChars(location.toString().toLowerCase(Locale.ROOT).replaceAll("[./]", "_"))),
				image, filename, location);
			register(id(), image());
		}

		public LocalImpl(Path p) throws IOException {
			this(ImageIO.read(Files.newInputStream(p)), p.getFileName().toString(), p);
		}
	}

	record SharedImpl(C_0561170 id, BufferedImage image, String filename, Path location, String url, String uploader,
					  Instant sharedAt) implements Local, Remote {

	}

	record RemoteImpl(C_0561170 id, BufferedImage image, String filename, String uploader, Instant sharedAt,
					  String url) implements Remote {
		public RemoteImpl(BufferedImage image, String filename, String uploader, Instant sharedAt, String url) {
			this(new C_0561170("axolotlclient", "gallery_remote_" + Hashing.sha256().hashUnencodedChars(url.toLowerCase(Locale.ROOT).replaceAll("[./]", "_"))),
				image, filename, uploader, sharedAt, url
			);
			register(id(), image());
		}
	}

}
