/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.screenshotUtils;

import javax.imageio.ImageIO;
import net.minecraft.unmapped.C_2348249;
import net.minecraft.unmapped.C_3020744;
import net.minecraft.unmapped.C_3390001;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_8105098;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientCommon;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.api.util.UUIDHelper;
import io.github.axolotlclient.util.GsonHelper;
import io.github.axolotlclient.util.OSUtil;
import io.github.axolotlclient.util.notifications.Notifications;

public class ImageScreen extends C_3020744 {

	private final C_3020744 parent;
	private final ImageInstance image;
	private final boolean freeOnClose;
	private final boolean isRemote;
	private final String title;
	private final CompletableFuture<String> uploader;

	static C_3020744 create(C_3020744 parent, CompletableFuture<ImageInstance> future, boolean freeOnClose) {
		if (future.isDone()) {
			if (future.join() != null) {
				return new ImageScreen(parent, future.join(), freeOnClose);
			} else {
				return parent;
			}
		}
		return new LoadingImageScreen(parent, future.thenAccept(i -> {
			if (i != null) {
				C_8105098.m_0408063().m_2167033(() -> C_8105098.m_0408063().m_6408915(new ImageScreen(parent, i, freeOnClose)));
			} else {
				C_8105098.m_0408063().m_2167033(() -> C_8105098.m_0408063().m_6408915(parent));
			}
		}), freeOnClose);
	}

	private ImageScreen(C_3020744 parent, ImageInstance instance, boolean freeOnClose) {
		super();
		this.title = instance.filename();
		this.parent = parent;
		this.image = instance;
		this.freeOnClose = freeOnClose;
		this.isRemote = image instanceof ImageInstance.Remote;
		if (isRemote) {
			this.uploader = UUIDHelper.tryGetUsernameAsync(((ImageInstance.Remote) image).uploader());
		} else {
			this.uploader = null;
		}
	}

	@Override
	public void m_7261014(int mouseX, int mouseY, float delta) {
		m_7817195();
		super.m_7261014(mouseX, mouseY, delta);
		if (isRemote && uploader.isDone()) {
			ImageInstance.Remote r = (ImageInstance.Remote) image;
			m_2717572(f_2020658, title, f_5465691 / 2, 38 / 2 - f_2020658.f_6725889 - 2, -1);
			m_2717572(f_2020658,
				C_3390001.m_2053009("gallery.image.upload_details", uploader.join(),
					r.sharedAt().atZone(ZoneId.systemDefault()).format(AxolotlClientCommon.getInstance().getConfig().getDateTimeFormatter())),
				f_5465691 / 2, 38 / 2 + 2, -1);
		} else {
			m_2717572(f_2020658, title, f_5465691 / 2, 33 / 2 - f_2020658.f_6725889 / 2, -1);
		}
	}

	@Override
	public void m_3593494() {

		int contentsHeight = f_3080061 - ((isRemote ? 38 : 33) + 33);
		int buttonWidth = 75;
		double imgAspectRatio = image.image().getWidth() / (double) image.image().getHeight();
		int imageWidth = Math.min((int) (contentsHeight * imgAspectRatio), f_5465691 - buttonWidth - 4 - 20);
		int imageHeight = (int) (imageWidth / imgAspectRatio);


		var element = new ImageElement(imageWidth, imageHeight);
		f_2213969.add(element);
		if (f_5465691 / 2 > (imageWidth / 2) + buttonWidth + 4) {
			element.setPosition(f_5465691 / 2 - imageWidth / 2, 36);
		} else {
			element.setPosition(f_5465691 / 2 - imageWidth / 2 - buttonWidth / 2 - 2, 36);
		}
		int actionX = element.f_3923091 + imageWidth + 4;
		var actions = new ArrayList<C_2348249>();
		if (image instanceof ImageInstance.Local) {
			if (API.getInstance().isAuthenticated() && !(image instanceof ImageInstance.Remote)) {
				actions.add(new C_2348249(0, 0, 0, buttonWidth, 20, C_3390001.m_2053009("gallery.image.upload")));
			}
			actions.add(new C_2348249(1, 0, 0, buttonWidth, 20, C_3390001.m_2053009("gallery.image.copy")));
			actions.add(new C_2348249(2, 0, 0, buttonWidth, 20, C_3390001.m_2053009("gallery.image.open.external")));
		}
		if (image instanceof ImageInstance.Remote) {
			if (!(image instanceof ImageInstance.Local)) {
				actions.add(new C_2348249(3, 0, 0, buttonWidth, 20, C_3390001.m_2053009("gallery.image.save")));
				actions.add(new C_2348249(4, 0, 0, buttonWidth, 20, C_3390001.m_2053009("gallery.image.copy")));
			}
			actions.add(new C_2348249(5, 0, 0, buttonWidth, 20, C_3390001.m_2053009("gallery.image.open.external.browser")));
			actions.add(new C_2348249(6, 0, 0, buttonWidth, 20, C_3390001.m_2053009("gallery.image.copy_url")));
		}
		int actionY = element.f_8532347;
		for (C_2348249 w : actions) {
			w.f_3923091 = actionX;
			w.f_8532347 = actionY;
			f_2213969.add(w);
			actionY += 4 + 20;
		}

		f_2213969.add(new C_2348249(7, f_5465691 / 2 - 75, f_3080061 - 33 + 33 / 2 - 10, 150, 20, C_3390001.m_2053009("gui.back")));
	}

	@Override
	protected void m_7971793(C_2348249 b) {
		if (b.f_5920996 == 0) {
			b.f_3831222 = false;
			ImageInstance.Local local = (ImageInstance.Local) image;
			ImageShare.getInstance().upload(local.location()).thenAccept(s -> {
				if (s.isEmpty()) {
					Notifications.getInstance().addStatus("gallery.image.upload.failure", "gallery.image.upload.failure.description");
				} else {
					minecraft.submit(() -> minecraft.openScreen(new ImageScreen(parent, local.toShared(s, API.getInstance().getSelf().getUuid(), Instant.now()), freeOnClose)));
					setClipboard(s);
					Notifications.getInstance().addStatus("gallery.image.upload.success", "gallery.image.upload.success.description", s);
				}
			});
		} else if (b.f_5920996 == 1) {
			ImageInstance.Local local = (ImageInstance.Local) image;
			ScreenshotCopying.copy(local.location());
		} else if (b.f_5920996 == 2) {
			ImageInstance.Local local = (ImageInstance.Local) image;
			OSUtil.getOS().open(local.location().toUri());
		} else if (b.f_5920996 == 3) {
			ImageInstance.Remote remote = (ImageInstance.Remote) image;
			b.f_3831222 = false;
			try {
				Path out = saveSharedImage(remote);
				f_7153641.m_6408915(new ImageScreen(parent, remote.toShared(out), freeOnClose));
			} catch (IOException e) {
				Notifications.getInstance().addStatus("gallery.image.save.failure", "gallery.image.save.failure.description", e.getMessage());
				AxolotlClient.LOGGER.warn("Failed to save shared image!", e);
			}
		} else if (b.f_5920996 == 4) {
			try (ByteArrayOutputStream baos = new ByteArrayOutputStream()) {
				ImageIO.write(image.image(), "png", baos);
				ScreenshotCopying.copy(baos.toByteArray());
			} catch (IOException e) {
				Notifications.getInstance().addStatus("gallery.image.copy.failure", "gallery.image.copy.failure.description", e.getMessage());
				AxolotlClient.LOGGER.warn("Failed to copy shared image!", e);
			}

		} else if (b.f_5920996 == 5) {
			ImageInstance.Remote remote = (ImageInstance.Remote) image;
			OSUtil.getOS().open(remote.url());

		} else if (b.f_5920996 == 6) {
			ImageInstance.Remote remote = (ImageInstance.Remote) image;
			m_5360526(remote.url());

		} else if (b.f_5920996 == 7) {
			if (freeOnClose) {
				f_7153641.m_1218956().m_3775266(image.id());
			}
			f_7153641.m_6408915(parent);

		}
	}

	private Path saveSharedImage(ImageInstance.Remote remote) throws IOException {
		Path out = GalleryScreen.SCREENSHOTS_DIR.resolve("shared")
			.resolve(remote.uploader())
			.resolve(remote.filename());
		Path infoJson = out.resolveSibling(remote.filename() + ".json");

		Files.createDirectories(out.getParent());
		try (OutputStream s = Files.newOutputStream(out)) {
			ImageIO.write(remote.image(), "png", s);
		}
		Object json = Map.of("uploader", remote.uploader(), "shared_at", remote.sharedAt());
		Files.writeString(infoJson, GsonHelper.GSON.toJson(json));
		return out;
	}

	private class ImageElement extends C_2348249 {

		public ImageElement(int width, int height) {
			super(99, 0, 0, width, height, "");
			f_3831222 = false;
		}

		@Override
		public void m_5350167(C_8105098 client, int mouseX, int mouseY) {
			C_3754158.m_3323122(1, 1, 1, 1);
			client.m_1218956().m_5325521(image.id());
			m_5935491(f_3923091, f_8532347, 0, 0, m_8567079(), getHeight(), m_8567079(), getHeight());
		}

		private int getHeight() {
			return f_7845401;
		}

		public void setPosition(int x, int y) {
			this.f_3923091 = x;
			this.f_8532347 = y;
		}
	}
}
