/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.screenshotUtils;

import javax.imageio.ImageIO;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;

import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import io.github.axolotlclient.util.Util;
import lombok.Getter;
import net.minecraft.text.*;
import net.minecraft.unmapped.C_1160932;
import net.minecraft.unmapped.C_1716360;
import net.minecraft.unmapped.C_1945050;
import net.minecraft.unmapped.C_3020744;
import net.minecraft.unmapped.C_3390001;
import net.minecraft.unmapped.C_5008376;
import net.minecraft.unmapped.C_6237110;

public class ImageShare extends ImageNetworking {

	@Getter
	private static final ImageShare Instance = new ImageShare();

	private ImageShare() {
	}

	public void uploadImage(Path file) {
		Util.addMessageToChatHud(new C_1160932("imageUploadStarted"));
		upload(file).whenCompleteAsync((downloadUrl, throwable) -> {
			if (downloadUrl.isEmpty()) {
				Util.addMessageToChatHud(new C_1160932("imageUploadFailure"));
			} else {
				Util.addMessageToChatHud(new C_1716360(C_3390001.m_2053009("imageUploadSuccess") + " ")
					.m_8059675(new C_1716360(downloadUrl)
						.m_7551367(new C_6237110()
							.m_9739860(true)
							.m_0282611(C_1945050.f_8162691)
							.m_7701784(new ScreenshotUtils.CustomClickEvent(null, null) {
											   @Override
											   public void doAction() {
												   C_3020744.m_5360526(downloadUrl);
											   }
										   }
							)
							.m_3975477(new C_5008376(C_5008376.C_4842797.f_5996706, new C_1716360(C_3390001.m_2053009("clickToCopy")))))));
			}
		});
	}

	public CompletableFuture<ImageInstance> downloadImage(String url) {
		return download(url).thenApply(data -> {
			if (data != ImageData.EMPTY) {
				try (var in = new ByteArrayInputStream(data.data())) {
					ImageInstance.Remote remote = new ImageInstance.RemoteImpl(ImageIO.read(in), data.name(), data.uploader(), data.sharedAt(), ensureUrl(url).orElseThrow());
					try {
						Path local = GalleryScreen.SCREENSHOTS_DIR.resolve(remote.filename());
						HashFunction hash = Hashing.goodFastHash(32);
						if (Files.exists(local) && hash.hashBytes(data.data()).equals(hash.hashBytes(Files.readAllBytes(local)))) {
							return remote.toShared(local);
						}
					} catch (IOException ignored) {
					}
					return remote;
				} catch (IOException ignored) {
				}
			}
			return null;
		});
	}
}
