/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.screenshotUtils;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.BooleanSupplier;

import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientConfig.api.options.OptionCategory;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.StringArrayOption;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.modules.AbstractModule;
import io.github.axolotlclient.util.OSUtil;
import io.github.axolotlclient.util.Util;
import io.github.axolotlclient.util.options.GenericOption;
import lombok.AllArgsConstructor;
import lombok.Getter;
import net.minecraft.text.*;
import net.minecraft.unmapped.C_1716360;
import net.minecraft.unmapped.C_1945050;
import net.minecraft.unmapped.C_3390001;
import net.minecraft.unmapped.C_5008376;
import net.minecraft.unmapped.C_6237110;
import net.minecraft.unmapped.C_8105098;
import net.minecraft.unmapped.C_8529493;
import net.minecraft.unmapped.C_9550253;
import org.jetbrains.annotations.Nullable;

public class ScreenshotUtils extends AbstractModule {

	@Getter
	private static final ScreenshotUtils Instance = new ScreenshotUtils();
	private final OptionCategory category = OptionCategory.create("screenshotUtils");
	private final BooleanOption enabled = new BooleanOption("enabled", false);
	private final GenericOption openViewer = new GenericOption("imageViewer", "openViewer",
		() -> C_8105098.m_0408063().m_6408915(new GalleryScreen(C_8105098.m_0408063().f_0723335)));

	private final Map<BooleanSupplier, Action> actions = new LinkedHashMap<>();

	private StringArrayOption autoExec;

	@Override
	public void init() {

		actions.put(() -> true, new Action("copyAction", C_1945050.f_0440484,
			"copy_image",
			ScreenshotCopying::copy));

		actions.put(() -> true, new Action("deleteAction", C_1945050.f_3614776,
			"delete_image",
			(file) -> {
				try {
					Files.delete(file);
					Util.addMessageToChatHud(
						new LiteralText(I18n.translate("screenshot_deleted").replace("<name>", file.getFileName().toString())));
				} catch (Exception e) {
					AxolotlClient.LOGGER.warn("Couldn't delete Screenshot " + file.getFileName().toString());
				}
			}));

		actions.put(() -> true, new Action("openAction", C_1945050.f_5887713,
			"open_image",
			(file) -> OSUtil.getOS().open(file.toUri())));

		actions.put(() -> true, new Action("viewInGalleryAction", C_1945050.f_3614776, "view_in_gallery",
			file -> {
				try {
					ImageInstance instance = new ImageInstance.LocalImpl(file);
					Minecraft.getInstance().submit(() -> Minecraft.getInstance().openScreen(ImageScreen.create(null, CompletableFuture.completedFuture(instance), true)));
				} catch (Exception ignored) {
					Util.addMessageToChatHud(new TranslatableText("screenshot.gallery.view.error"));
				}
			}));

		actions.put(() -> API.getInstance().isAuthenticated(), new Action("uploadAction", C_1945050.f_0440484,
			"upload_image",
			ImageShare.getInstance()::uploadImage));

		autoExec = new StringArrayOption("autoExec", Util.make(() -> {
			List<String> names = new ArrayList<>();
			names.add("off");
			actions.forEach((condition, action) -> names.add(action.getName()));
			return names.toArray(new String[0]);

		}), "off");

		category.add(enabled, autoExec, openViewer);

		AxolotlClient.config().general.add(category);
	}

	public C_9550253 onScreenshotTaken(C_9550253 text, File shot) {
		if (enabled.get()) {
			C_9550253 t = getUtilsText(shot.toPath());
			if (t != null) {
				return text.m_1643573("\n").m_8059675(t);
			}
		}
		return text;
	}

	private @Nullable C_9550253 getUtilsText(Path file) {
		if (!autoExec.get().equals("off")) {
			actions.forEach((condition, action) -> {
				if (condition.getAsBoolean() && autoExec.get().equals(action.getName())) {
					action.getClickEvent(file).doAction();
				}
			});
			return null;
		}

		C_9550253 message = new C_1716360("");
		actions.forEach((condition, action) -> {
			if (condition.getAsBoolean()) {
				message.m_8059675(action.getText(file)).m_1643573(" ");
			}
		});
		return message;
	}

	public interface OnActionCall {

		void doAction(Path file);
	}

	@AllArgsConstructor
	public static class Action {

		private final String translationKey;
		private final C_1945050 formatting;
		private final String hoverTextKey;
		private final OnActionCall clickEvent;

		public C_9550253 getText(Path file) {
			return new C_1716360(C_3390001.m_2053009(translationKey))
				.m_7551367(new C_6237110()
					.m_0282611(formatting)
					.m_7701784(getClickEvent(file))
					.m_3975477(new C_5008376(C_5008376.C_4842797.f_5996706, new C_1716360(C_3390001.m_2053009(hoverTextKey)))));
		}

		public String getName() {
			return translationKey;
		}

		public CustomClickEvent getClickEvent(Path file) {
			return new CustomClickEvent(clickEvent, file);
		}
	}

	public static class CustomClickEvent extends C_8529493 {

		private final OnActionCall action;
		private final Path file;

		public CustomClickEvent(OnActionCall action, Path file) {
			super(C_4892378.m_6936186(""), "");
			this.action = action;
			this.file = file;
		}

		public void doAction() {
			if (file != null) {
				action.doAction(file);
			} else {
				AxolotlClient.LOGGER.warn("How'd you manage to do this? " +
					"Now there's a screenshot ClickEvent without a File attached to it!");
			}
		}
	}
}
