/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.sky;

import java.util.Locale;
import java.util.Objects;
import net.minecraft.unmapped.C_0561170;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_4461663;
import net.minecraft.unmapped.C_4976084;
import net.minecraft.unmapped.C_5091950;
import net.minecraft.unmapped.C_5786166;
import net.minecraft.unmapped.C_8105098;
import net.minecraft.unmapped.C_8373595;
import com.google.gson.JsonObject;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.mixin.TextureManagerAccessor;
import io.github.axolotlclient.mixin.WorldRendererAccessor;
import io.github.axolotlclient.util.Util;
import org.lwjgl.opengl.GL11;
import org.lwjgl.opengl.GL14;

/**
 * This implementation of custom skies is based on the FabricSkyBoxes mod by AMereBagatelle
 * <a href="https://github.com/AMereBagatelle/FabricSkyBoxes">Github Link.</a>
 *
 * <p>License: MIT</p>
 **/

// TODO fix rotation & blending issues with shooting stars, implement more missing features like worlds, weather, biomes...

public abstract class SkyboxInstance {

	protected final C_0561170 MOON_PHASES = new C_0561170("textures/environment/moon_phases.png");
	protected final C_0561170 SUN = new C_0561170("textures/environment/sun.png");
	protected int blendMode = 1;
	// ! These are the options variables.  Do not mess with these.
	protected boolean alwaysOn;
	protected float maxAlpha = 1f;
	protected boolean manualBlend = false;
	protected int blendSrcFactor = 1;
	protected int blendDstFactor = 1;
	protected int blendEquation;
	protected boolean rotate = false;
	protected float rotationSpeed = 1F;
	protected float[] rotationStatic = new float[]{0, 0, 0};
	protected float[] rotationAxis = new float[]{0, 0, 0};
	protected boolean showSun = true;
	protected boolean showMoon = true;
	protected boolean showStars = true;
	JsonObject object;
	float alpha = 1F;
	C_0561170[] textures = new C_0561170[6];
	int[] fade = new int[4];

	public SkyboxInstance(JsonObject json) {
		this.object = json;
	}

	public float getAlpha() {
		if (alwaysOn) {
			return 1F;
		}

		int currentTime = (int) Objects.requireNonNull(C_8105098.m_0408063().f_4601986).m_2665637() % 24000; // modulo so that it's bound to 24000
		int durationIn = Util.getTicksBetween(fade[0], fade[1]);
		int durationOut = Util.getTicksBetween(fade[2], fade[3]);

		int startFadeIn = fade[0] % 24000;
		int endFadeIn = fade[1] % 24000;

		if (endFadeIn < startFadeIn) {
			endFadeIn += 24000;
		}

		int startFadeOut = fade[2] % 24000;
		int endFadeOut = fade[3] % 24000;

		if (startFadeOut < endFadeIn) {
			startFadeOut += 24000;
		}

		if (endFadeOut < startFadeOut) {
			endFadeOut += 24000;
		}

		int tempInTime = currentTime;

		if (tempInTime < startFadeIn) {
			tempInTime += 24000;
		}

		int tempFullTime = currentTime;

		if (tempFullTime < endFadeIn) {
			tempFullTime += 24000;
		}

		int tempOutTime = currentTime;

		if (tempOutTime < startFadeOut) {
			tempOutTime += 24000;
		}

		float maxPossibleAlpha;

		if (startFadeIn < tempInTime && endFadeIn >= tempInTime) {
			maxPossibleAlpha = 1f - (((float) (endFadeIn - tempInTime)) / durationIn); // fading in
		} else if (endFadeIn < tempFullTime && startFadeOut >= tempFullTime) {
			maxPossibleAlpha = 1f; // fully faded in
		} else if (startFadeOut < tempOutTime && endFadeOut >= tempOutTime) {
			maxPossibleAlpha = (float) (endFadeOut - tempOutTime) / durationOut; // fading out
		} else {
			maxPossibleAlpha = 0f; // default not showing
		}

		return alpha = C_4976084.m_7164829(maxPossibleAlpha * maxAlpha, 0, 1);
	}

	protected int parseBlend(String str) {
		if (str == null) {
			return 1;
		} else {
			return switch (str.toLowerCase(Locale.ENGLISH).trim()) {
				case "alpha" -> 0;
				case "add" -> 1;
				case "subtract" -> 2;
				case "multiply" -> 3;
				case "dodge" -> 4;
				case "burn" -> 5;
				case "screen" -> 6;
				case "overlay" -> 7;
				case "replace" -> 8;
				default -> {
					AxolotlClient.LOGGER.warn("Unknown blend: " + str);
					yield 1;
				}
			};
		}
	}

	public void render(float delta, float brightness) {
		C_3754158.m_6191982();
		C_3754158.m_9671730();
		C_3754158.m_8373640();

		setupBlend(brightness);
		setupRotate(delta, brightness);
		renderSkybox();
		renderDecorations(delta, brightness);
		clearBlend(brightness);
		clearRotate();

		C_3754158.m_2041265();
		C_3754158.m_2754767();
		C_3754158.m_5313301();
	}

	protected void setupBlend(float brightness) {
		C_3754158.m_6191982();
		C_3754158.m_5313301();
		if (manualBlend) {
			C_3754158.m_9671730();
			C_3754158.m_2076358(blendSrcFactor, blendDstFactor);
			GL14.glBlendEquation(blendEquation);
			C_3754158.m_7547086();
			return;
		}

		switch (blendMode) {
			case 0:

				C_3754158.m_9671730();
				C_3754158.m_2076358(770, 771);
				break;

			case 1:

				C_3754158.m_9671730();
				C_3754158.m_2076358(770, 1);
				break;

			case 2:
				C_3754158.m_9671730();
				C_3754158.m_2076358(775, 0);
				break;

			case 3:
				C_3754158.m_9671730();
				C_3754158.m_2076358(774, 771);
				break;

			case 4:
				C_3754158.m_9671730();
				C_3754158.m_2076358(1, 1);
				break;

			case 5:
				C_3754158.m_9671730();
				C_3754158.m_2076358(0, 769);
				break;

			case 6:
				C_3754158.m_9671730();
				C_3754158.m_2076358(1, 769);
				break;

			case 7:
				C_3754158.m_9671730();
				C_3754158.m_2076358(774, 768);
				break;

			case 8:
				C_3754158.m_5313301();
				break;
		}
		C_3754158.m_3323122(1.0F, 1.0F, 1.0F, brightness);

		C_3754158.m_7547086();
	}

	protected void setupRotate(float delta, float brightness) {
		C_3754158.m_8616673(0, rotationStatic[0], rotationStatic[1], rotationStatic[2]);
		if (rotate) {
			C_3754158.m_8616673(0, rotationAxis[0], rotationAxis[1], rotationAxis[2]);
			C_3754158.m_3323122(1.0F, 1.0F, 1.0F, brightness);
			//GlStateManager.rotatef(-90.0F, 0.0F, 1.0F, 0.0F);
			C_3754158.m_8616673(C_8105098.m_0408063().f_4601986.m_0386667(delta) * rotationSpeed, 0.0F, 1.0F,
				0.0F);
			C_3754158.m_8616673(0, -rotationAxis[0], -rotationAxis[1], -rotationAxis[2]);
		}
	}

	public abstract void renderSkybox();

	protected void renderDecorations(float delta, float brightness) {
		C_3754158.m_7547086();
		C_3754158.m_9671730();
		C_3754158.m_0833259(770, 1, 1, 0);
		C_3754158.m_8373640();
		C_3754158.m_3323122(1.0F, 1.0F, 1.0F, brightness);
		C_3754158.m_8616673(-90.0F, 0.0F, 1.0F, 0.0F);
		C_3754158.m_8616673(C_8105098.m_0408063().f_4601986.m_0386667(delta) * 360.0F, 1.0F, 0.0F, 0.0F);

		C_5786166 tessellator = C_5786166.m_2065116();
		C_8373595 bufferBuilder = tessellator.m_1454391();

		if (showSun) {
			float o = 30.0F;
			C_8105098.m_0408063().m_1218956().m_5325521(SUN);
			bufferBuilder.m_0421390(GL11.GL_QUADS, C_4461663.f_9223614);
			bufferBuilder.m_3299851((-o), 100.0, (-o)).m_4749889(0.0, 0.0).m_4365807();
			bufferBuilder.m_3299851(o, 100.0, (-o)).m_4749889(1.0, 0.0).m_4365807();
			bufferBuilder.m_3299851(o, 100.0, o).m_4749889(1.0, 1.0).m_4365807();
			bufferBuilder.m_3299851(-o, 100.0, o).m_4749889(0.0, 1.0).m_4365807();
			tessellator.m_8222644();
		}
		if (showMoon) {
			float o = 20.0F;
			C_8105098.m_0408063().m_1218956().m_5325521(MOON_PHASES);
			int x = C_8105098.m_0408063().f_4601986.m_3322854();
			int t = x % 4;
			int u = x / 4 % 2;
			float s = (float) (t) / 4.0F;
			float v = (float) (u) / 2.0F;
			float w = (float) (t + 1) / 4.0F;
			float y = (float) (u + 1) / 2.0F;
			bufferBuilder.m_0421390(7, C_4461663.f_9223614);
			bufferBuilder.m_3299851((-o), -100.0, o).m_4749889(w, y).m_4365807();
			bufferBuilder.m_3299851(o, -100.0, o).m_4749889(s, y).m_4365807();
			bufferBuilder.m_3299851(o, -100.0, (-o)).m_4749889(s, v).m_4365807();
			bufferBuilder.m_3299851((-o), -100.0, (-o)).m_4749889(w, v).m_4365807();
			tessellator.m_8222644();
		}
		if (showStars) {
			C_3754158.m_2090124();
			float z = C_8105098.m_0408063().f_4601986.m_9739314(delta) * brightness;
			if (z > 0.0F) {
				C_3754158.m_3323122(z, z, z, z);
				if (((WorldRendererAccessor) C_8105098.m_0408063().f_4021716).getVbo()) {
					C_5091950 starsBuffer = ((WorldRendererAccessor) C_8105098.m_0408063().f_4021716)
						.getStarsBuffer();
					starsBuffer.m_4899212();
					GL11.glEnableClientState(32884);
					GL11.glVertexPointer(3, 5126, 12, 0L);
					starsBuffer.m_2500802(7);
					starsBuffer.m_7277379();
					GL11.glDisableClientState(32884);
				} else {
					C_3754158.m_1175393(
						((WorldRendererAccessor) C_8105098.m_0408063().f_4021716).getStarsList());
				}
			}
		}
		C_3754158.m_5313301();
		C_3754158.m_2754767();
		C_3754158.m_2041265();
		C_3754158.m_7547086();
	}

	protected void clearBlend(float brightness) {
		C_3754158.m_6191982();
		C_3754158.m_9671730();
		C_3754158.m_2076358(0, 1);
		C_3754158.m_3323122(1.0F, 1.0F, 1.0F, brightness);
	}

	protected void clearRotate() {
		C_3754158.m_8616673(0, -rotationStatic[0], -rotationStatic[1], -rotationStatic[2]);
	}

	public void remove() {
		for (C_0561170 id : textures) {
			try {
				C_8105098.m_0408063().m_1218956().m_3775266(id);
				((TextureManagerAccessor) C_8105098.m_0408063().m_1218956()).getTextures().remove(id);
			} catch (Exception ignored) {
			}
		}
	}
}
