/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.util;

import javax.imageio.ImageIO;
import net.minecraft.unmapped.C_0561170;
import net.minecraft.unmapped.C_0814381;
import net.minecraft.unmapped.C_1945050;
import net.minecraft.unmapped.C_3526030;
import net.minecraft.unmapped.C_3779718;
import net.minecraft.unmapped.C_3945557;
import net.minecraft.unmapped.C_4143404;
import net.minecraft.unmapped.C_7799337;
import net.minecraft.unmapped.C_8105098;
import net.minecraft.unmapped.C_9550253;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.*;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Color;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Graphics;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.GraphicsOption;
import io.github.axolotlclient.mixin.MinecraftClientAccessor;
import org.jetbrains.annotations.ApiStatus;

public class Util {

	public static final Util.GlColor GlColor = new GlColor();
	public static String lastgame;
	public static String game;

	@ApiStatus.Internal
	public static C_7799337 window;

	/**
	 * Gets the amount of ticks in between start and end, on a 24000 tick system.
	 *
	 * @param start The start of the time you wish to measure
	 * @param end   The end of the time you wish to measure
	 * @return The amount of ticks in between start and end
	 */
	public static int getTicksBetween(int start, int end) {
		if (end < start)
			end += 24000;
		return end - start;
	}

	public static int toGlCoordsX(int x) {
		if (window == null) {
			window = new C_7799337(C_8105098.m_0408063());
		}
		return x * window.m_8052133();
	}

	public static int toGlCoordsY(int y) {
		if (window == null) {
			window = new C_7799337(C_8105098.m_0408063());
		}
		int scale = window.m_8052133();
		return C_8105098.m_0408063().f_5990000 - y * scale - scale;
	}

	public static int toMCCoordsX(int x) {
		if (window == null) {
			window = new C_7799337(C_8105098.m_0408063());
		}
		return x * window.m_1713651() / C_8105098.m_0408063().f_0545414;
	}

	public static int toMCCoordsY(int y) {
		if (window == null) {
			window = new C_7799337(C_8105098.m_0408063());
		}
		return window.m_5421693() - y * window.m_5421693() / C_8105098.m_0408063().f_5990000 - 1;
	}

	public static C_7799337 getWindow() {
		if (window == null) {
			try {
				window = new C_7799337(C_8105098.m_0408063());
			} catch (Exception e) {
				return null;
			}
		}
		return window;
	}

	public static void sendChatMessage(String msg) {
		C_8105098.m_0408063().f_7663840.m_2226526(msg);
	}

	public static void addMessageToChatHud(C_9550253 msg) {
		C_8105098.m_0408063().f_3501374.m_8449802().m_7344462(msg);
	}

	public static String splitAtCapitalLetters(String string) {
		if (string == null || string.isEmpty()) {
			return "";
		}
		StringBuilder builder = new StringBuilder();
		for (char c : string.toCharArray()) {
			if (Character.isUpperCase(c) && c != string.charAt(0)) {
				builder.append(" ");
			}
			builder.append(c);
		}
		return builder.toString();
	}

	public static String getGame() {
		List<String> sidebar = getSidebar();

		if (sidebar.isEmpty())
			game = "";
		else if (Util.getCurrentServerAddress() != null
			&& Util.getCurrentServerAddress().toLowerCase().contains(sidebar.get(0).toLowerCase())) {
			if (sidebar.get(sidebar.size() - 1).toLowerCase(Locale.ROOT)
				.contains(Util.getCurrentServerAddress().toLowerCase(Locale.ROOT))
				|| sidebar.get(sidebar.size() - 1).contains("Playtime")) {
				game = "In Lobby";
			} else {
				if (sidebar.get(sidebar.size() - 1).contains("--------")) {
					game = "Playing Bridge Practice";
				} else {
					game = "Playing " + sidebar.get(sidebar.size() - 1);
				}
			}
		} else {
			game = "Playing " + sidebar.get(0);
		}

		if (!Objects.equals(lastgame, game) && game.isEmpty())
			game = lastgame;
		else
			lastgame = game;

		if (game == null) {
			game = "";
		}

		return C_1945050.m_4710995(game);
	}

	public static List<String> getSidebar() {
		List<String> lines = new ArrayList<>();
		C_8105098 client = C_8105098.m_0408063();
		if (client.f_4601986 == null)
			return lines;

		C_0814381 scoreboard = client.f_4601986.m_5847414();
		if (scoreboard == null)
			return lines;
		C_3945557 sidebar = scoreboard.m_8757270(1);
		if (sidebar == null)
			return lines;

		Collection<C_3526030> scores = scoreboard.m_0477108(sidebar);
		List<C_3526030> list = scores.stream().filter(
				input -> input != null && input.m_2012228() != null && !input.m_2012228().startsWith("#"))
			.collect(Collectors.toList());

		if (list.size() > 15) {
			scores = Lists.newArrayList(Iterables.skip(list, scores.size() - 15));
		} else {
			scores = list;
		}

		for (C_3526030 score : scores) {
			C_4143404 team = scoreboard.m_2322207(score.m_2012228());
			if (team == null)
				return lines;
			String text = team.m_1445072() + team.m_6964130();
			if (!text.trim().isEmpty())
				lines.add(text);
		}

		lines.add(sidebar.m_0646168());
		Collections.reverse(lines);

		return lines;
	}

	public static String getCurrentServerAddress() {
		if (C_8105098.m_0408063().m_9805322()) {
			return null;
		}

		if (C_8105098.m_0408063().m_1862495() != null) {
			return C_8105098.m_0408063().m_1862495().f_7619058;
		}
		return ((MinecraftClientAccessor) C_8105098.m_0408063()).getServerAddress() != null
			? ((MinecraftClientAccessor) C_8105098.m_0408063()).getServerAddress()
			: null;
	}

	public static <T> T make(Supplier<T> factory) {
		return factory.get();
	}

	public static boolean currentServerAddressContains(String address) {
		if (C_8105098.m_0408063().m_9805322()
			|| C_8105098.m_0408063().m_3844208()) {
			return false;
		}
		if (C_8105098.m_0408063().m_1862495() != null) {
			return C_8105098.m_0408063().m_1862495().f_7619058.contains(address);
		}
		return ((MinecraftClientAccessor) C_8105098.m_0408063()).getServerAddress() != null
			&& ((MinecraftClientAccessor) C_8105098.m_0408063()).getServerAddress().contains(address);
	}

	public static float lerp(float start, float end, float percent) {
		return start + ((end - start) * percent);
	}

	public static C_0561170 getTexture(GraphicsOption option) {
		return getTexture(option.get(), "graphics_" + option.getName());
	}

	public static C_0561170 getTexture(Graphics graphics, String name) {
		C_0561170 id = new C_0561170("axolotlclient", name.toLowerCase(Locale.ROOT));
		return getTexture(graphics, id);
	}

	public static C_0561170 getTexture(Graphics graphics, C_0561170 id) {
		try {
			C_3779718 texture;
			if (C_8105098.m_0408063().m_1218956().m_1605001(id) == null) {
				texture = new C_3779718(ImageIO.read(new ByteArrayInputStream(graphics.getPixelData())));
				C_8105098.m_0408063().m_1218956().m_0381205(id, texture);
			} else {
				texture = (C_3779718) C_8105098.m_0408063().m_1218956().m_1605001(id);
				int[] pix = texture.m_1823577();
				for (int x = 0; x < graphics.getWidth(); x++) {
					for (int y = 0; y < graphics.getHeight(); y++) {
						int rows = (y) * graphics.getWidth() + x;
						pix[rows] = graphics.getPixelColor(x, y);
					}
				}
			}

			texture.m_4420833();
		} catch (IOException e) {
			AxolotlClient.LOGGER.error("Failed to bind texture for " + id + ": ", e);
		}
		return id;
	}

	public static void bindTexture(GraphicsOption option) {
		C_0561170 id = getTexture(option);
		C_8105098.m_0408063().m_1218956().m_5325521(id);
	}

	public static class GlColor {

		public float red = 1.0F;
		public float green = 1.0F;
		public float blue = 1.0F;
		public float alpha = 1.0F;

		public GlColor() {
		}

		public GlColor(float red, float green, float blue, float alpha) {
			this.red = red;
			this.green = green;
			this.blue = blue;
			this.alpha = alpha;
		}
	}

	public static String getFormatCode(Color color) {
		return String.format("§#%06X", color.getRed() << 16 | color.getGreen() << 8 | color.getBlue());
	}
}
