/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.util.notifications;

import java.util.List;

import com.google.common.collect.ImmutableList;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import io.github.axolotlclient.util.notifications.toasts.Toast;
import io.github.axolotlclient.util.notifications.toasts.ToastManager;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.unmapped.C_0561170;
import net.minecraft.unmapped.C_2899740;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_3831727;
import net.minecraft.unmapped.C_8105098;
import org.jetbrains.annotations.Nullable;

@Environment(EnvType.CLIENT)
public class AxolotlClientToast extends DrawUtil implements Toast {
	private static final C_0561170 BACKGROUND_SPRITE = new C_0561170("axolotlclient", "textures/gui/sprites/toast/axolotlclient.png");
	private static final int DISPLAY_TIME_MILLIS = 5000;
	private static final int MAX_LINE_SIZE = 200;
	private static final int LINE_SPACING = 12;
	private static final int MARGIN = 10;
	private final String title;
	private final List<String> messageLines;
	private final int width;
	private Visibility wantedVisibility = Visibility.HIDE;

	public AxolotlClientToast(String title, @Nullable String message) {
		this(
			title,
			nullToEmpty(message),
			Math.max(DEFAULT_WIDTH, 15 + (2 * MARGIN) + Math.max(C_8105098.m_0408063().f_0426313.m_0040387(title), message == null ? 0 : C_8105098.m_0408063().f_0426313.m_0040387(message)))
		);
	}

	public static AxolotlClientToast multiline(C_8105098 minecraft, String title, String message) {
		C_3831727 font = minecraft.f_0426313;
		List<String> list = font.m_0466830(message, MAX_LINE_SIZE);
		int i = Math.min(MAX_LINE_SIZE, Math.max(font.m_0040387(title), list.stream().mapToInt(font::m_0040387).max().orElse(MAX_LINE_SIZE)));
		return new AxolotlClientToast(title, list, i + (2 * MARGIN) + 15);
	}

	private AxolotlClientToast(String title, List<String> messageLines, int width) {
		this.title = title;
		this.messageLines = messageLines;
		this.width = width;
	}

	private static ImmutableList<String> nullToEmpty(@Nullable String message) {
		return message == null ? ImmutableList.of() : ImmutableList.of(message);
	}

	@Override
	public int width() {
		return this.width;
	}

	@Override
	public int height() {
		return (2 * MARGIN) + Math.max(this.messageLines.size(), 1) * LINE_SPACING;
	}

	@Override
	public Visibility getWantedVisibility() {
		return this.wantedVisibility;
	}

	@Override
	public void update(ToastManager toastManager, long l) {
		double d = (double) DISPLAY_TIME_MILLIS * toastManager.getNotificationDisplayTimeMultiplier();
		this.wantedVisibility = (double) l < d ? Visibility.SHOW : Visibility.HIDE;
	}

	@Override
	public void render(C_3831727 font, long startTime) {
		C_3754158.m_5313301();
		C_2899740.m_8989115();
		blitSprite(BACKGROUND_SPRITE, 0, 0, width(), height(), new NineSlice(160, 64, new Border(17, 30, 4, 4), false));
		C_8105098.m_0408063().m_1218956().m_5325521(AxolotlClient.badgeIcon);
		m_5935491(4, 4, 0, 0, 15, 15, 15, 15);
		int textOffset = 22;
		if (this.messageLines.isEmpty()) {
			drawString(title, textOffset, LINE_SPACING, -256, false);
		} else {
			drawString(title, textOffset, 7, -256, false);

			for (int i = 0; i < this.messageLines.size(); i++) {
				drawString(this.messageLines.get(i), textOffset, 18 + i * LINE_SPACING, -1, false);
			}
		}
		C_2899740.m_3229588();
	}
}
