/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.api.chat;

import java.util.List;

import com.mojang.blaze3d.systems.RenderSystem;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.api.ContextMenu;
import io.github.axolotlclient.api.requests.ChannelRequest;
import io.github.axolotlclient.api.requests.FriendRequest;
import io.github.axolotlclient.api.types.Channel;
import io.github.axolotlclient.api.types.Relation;
import io.github.axolotlclient.api.types.User;
import io.github.axolotlclient.api.util.AlphabeticalComparator;
import io.github.axolotlclient.modules.auth.Auth;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import lombok.Getter;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_2588;
import net.minecraft.class_310;
import net.minecraft.class_4280;
import net.minecraft.class_4587;

public class ChatUserListWidget extends class_4280<ChatUserListWidget.UserListEntry> {

	private final ChatScreen screen;

	public ChatUserListWidget(ChatScreen screen, class_310 client, int width, int height, int top, int bottom, int entryHeight) {
		super(client, width, bottom - top, top, bottom, entryHeight);
		this.screen = screen;
	}

	public void setUsers(List<User> users, Channel channel) {
		users.stream().sorted((u1, u2) -> new AlphabeticalComparator().compare(u1.getName(), u2.getName())).forEach(user -> addEntry(new UserListEntry(user, channel)));
	}

	@Override
	public int method_25322() {
		return field_22742 - 5;
	}

	public int addEntry(UserListEntry entry) {
		return super.method_25321(entry.init(screen));
	}

	@Override
	protected int method_25329() {
		return method_25342() + field_22742 - 8;
	}

	@Override
	public boolean method_25316() {
		return this.screen.method_25399() == this;
	}

	public class UserListEntry extends Entry<UserListEntry> {

		@Getter
		private final User user;
		private final class_310 client;
		private final Channel channel;
		private long time;
		private ChatScreen screen;

		public UserListEntry(User user, Channel channel) {
			this.client = class_310.method_1551();
			this.user = user;
			this.channel = channel;
		}

		public UserListEntry init(ChatScreen screen) {
			this.screen = screen;
			return this;
		}


		@Override
		public void render(class_4587 graphics, int index, int y, int x, int entryWidth, int entryHeight, int mouseX, int mouseY, boolean hovered, float tickDelta) {
			if (hovered && !screen.hasContextMenu()) {
				method_25294(graphics, x - 2, y - 1, x + entryWidth - 3, y + entryHeight + 1, 0x55ffffff);
			}
			DrawUtil.drawScrollableText(graphics, client.field_1772, class_2561.method_30163(user.getName()), x + 3 + entryHeight,
				y + 1, x + entryWidth - 6, y + 1 + client.field_1772.field_2000 + 2, -1);
			DrawUtil.drawScrollableText(graphics, client.field_1772, class_2561.method_30163(user.getStatus().getTitle()), x + 3 + entryHeight,
				y + 12, x + entryWidth - 6, y + 12 + client.field_1772.field_2000 + 2, 8421504);

			client.method_1531().method_22813(Auth.getInstance().getSkinTexture(user.getUuid(), user.getName()));
			RenderSystem.enableBlend();
			method_25293(graphics, x, y, entryHeight, entryHeight, 8, 8, 8, 8, 64, 64);
			method_25293(graphics, x, y, entryHeight, entryHeight, 40, 8, 8, 8, 64, 64);
			RenderSystem.disableBlend();
		}

		@Override
		public boolean mouseClicked(double mouseX, double mouseY, int button) {
			ChatUserListWidget.this.method_25313(this);
			if (button == 0) { // left click
				if (class_156.method_658() - this.time < 250L && client.field_1687 == null) { // left *double* click

				}
				this.time = class_156.method_658();
			} else if (button == 1) { // right click

				if (!user.equals(API.getInstance().getSelf())) {
					ContextMenu.Builder menu = ContextMenu.builder().title(class_2561.method_30163(user.getName())).spacer();
					if (!channel.isDM()) {
						menu.entry(new class_2588("api.friends.chat"), buttonWidget -> {
							ChannelRequest.getOrCreateDM(user).whenCompleteAsync(((channel, throwable) ->
								client.execute(() -> client.method_1507(new ChatScreen(screen.getParent(), channel)))));
						}).spacer();
					}
					if (user.getRelation() != Relation.BLOCKED) {
						if (user.getRelation() != Relation.FRIEND) {
							menu.entry(new class_2588("api.friends.add"), b -> FriendRequest.getInstance().addFriend(user.getUuid())).spacer();
						}
						menu.entry(new class_2588("api.users.block"), buttonWidget ->
							FriendRequest.getInstance().blockUser(user));
					} else {
						menu.entry(new class_2588("api.users.unblock"), buttonWidget ->
							FriendRequest.getInstance().unblockUser(user));
					}
					if (channel.getOwner().equals(API.getInstance().getSelf())) {
						menu.spacer().entry(new class_2588("api.channel.remove_user"), b -> ChannelRequest.removeUserFromChannel(channel, user));
					}
					screen.setContextMenu(menu.build());
					return true;
				}
			}

			return false;
		}
	}
}
