/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.api.chat;

import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import com.mojang.blaze3d.systems.RenderSystem;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.api.ContextMenu;
import io.github.axolotlclient.api.ContextMenuScreen;
import io.github.axolotlclient.api.handlers.ChatHandler;
import io.github.axolotlclient.api.requests.ChannelRequest;
import io.github.axolotlclient.api.types.Channel;
import io.github.axolotlclient.api.types.ChatMessage;
import io.github.axolotlclient.modules.auth.Auth;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import io.github.axolotlclient.util.ClientColors;
import lombok.Getter;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2585;
import net.minecraft.class_2588;
import net.minecraft.class_287;
import net.minecraft.class_289;
import net.minecraft.class_290;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_410;
import net.minecraft.class_4280;
import net.minecraft.class_437;
import net.minecraft.class_4493;
import net.minecraft.class_4587;
import net.minecraft.class_5481;
import net.minecraft.text.*;

public class ChatWidget extends class_4280<ChatWidget.ChatLine> {

	private final List<ChatMessage> messages = new ArrayList<>();
	private final Channel channel;
	private final class_310 field_22740;
	private final ContextMenuScreen screen;

	public ChatWidget(Channel channel, int x, int y, int width, int height, ContextMenuScreen screen) {
		super(class_310.method_1551(), width, height, y, y + height, 13);
		this.channel = channel;
		this.field_22740 = class_310.method_1551();
		method_25333(x + 5);

		method_25315(false, 0);
		this.screen = screen;
		this.field_22742 = width;
		this.field_22743 = height;
		channel.getMessages().forEach(this::addMessage);

		ChatHandler.getInstance().setMessagesConsumer(chatMessages -> chatMessages.forEach(this::addMessage));
		ChatHandler.getInstance().setMessageConsumer(this::addMessage);
		ChatHandler.getInstance().setEnableNotifications(message -> !message.channelId().equals(channel.getId()));

		method_25307(method_25331());
		method_29344(false);
	}

	public int getX() {
		return field_19088;
	}

	protected int method_25331() {
		return Math.max(0, this.method_25317() - (this.field_19086 - this.field_19085 - 4));
	}

	@Override
	protected int method_25329() {
		return field_19088 + field_22742 - 6;
	}

	@Override
	public int method_25322() {
		return field_22742 - 60;
	}

	private void addMessage(ChatMessage message) {
		List<class_5481> list = field_22740.field_1772.method_1728(class_2561.method_30163(message.content()), method_25322());

		boolean scrollToBottom = method_25341() == method_25331();

		if (!messages.isEmpty()) {
			ChatMessage prev = messages.get(messages.size() - 1);
			if (!(prev.sender().equals(message.sender()) && prev.senderDisplayName().equals(message.senderDisplayName()))) {
				method_25321(new NameChatLine(message));
			} else {
				if (message.timestamp().getEpochSecond() - prev.timestamp().getEpochSecond() > 150) {
					method_25321(new NameChatLine(message));
				}
			}
		} else {
			method_25321(new NameChatLine(message));
		}

		list.forEach(t -> method_25321(new ChatLine(t, message)));
		messages.add(message);

		method_25396().sort(Comparator.comparingLong(c -> c.getOrigin().timestamp().getEpochSecond()));

		if (scrollToBottom) {
			method_25307(method_25331());
		}
		messages.sort(Comparator.comparingLong(value -> value.timestamp().getEpochSecond()));
	}

	private void loadMessages() {
		long before;
		if (!messages.isEmpty()) {
			before = messages.get(0).timestamp().getEpochSecond();
		} else {
			before = Instant.now().getEpochSecond();
		}
		ChatHandler.getInstance().getMessagesBefore(channel, before);
	}

	@Override
	public boolean method_25401(double mouseX, double mouseY, double amountY) {
		double scrollAmount = (this.method_25341() - amountY * (double) this.field_22741 / 2.0);
		if (scrollAmount < 0) {
			loadMessages();
		}
		method_25307(scrollAmount);
		return true;
	}

	public void remove() {
		ChatHandler.getInstance().setMessagesConsumer(ChatHandler.DEFAULT_MESSAGES_CONSUMER);
		ChatHandler.getInstance().setMessageConsumer(ChatHandler.DEFAULT_MESSAGE_CONSUMER);
		ChatHandler.getInstance().setEnableNotifications(ChatHandler.DEFAULT);
	}

	@Override
	protected void method_25311(class_4587 matrices, int x, int y, int mouseX, int mouseY, float delta) {
		DrawUtil.enableScissor(this.field_19088, this.field_19085, this.field_19088 + field_22742, this.field_19085 + field_22743);
		super.method_25311(matrices, x, y, mouseX, mouseY, delta);
		DrawUtil.disableScissor();
	}

	@Override
	public void method_25394(class_4587 matrices, int mouseX, int mouseY, float delta) {
		int i = this.method_25329();
		int j = i + 6;
		class_289 tessellator = class_289.method_1348();
		class_287 bufferBuilder = tessellator.method_1349();
		this.field_22740.method_1531().method_22813(class_332.field_22735);
		RenderSystem.color4f(1.0F, 1.0F, 1.0F, 1.0F);
		if (field_22740.field_1687 == null) {
			bufferBuilder.method_1328(7, class_290.field_1575);
			bufferBuilder.method_22912(this.field_19088, this.field_19086, 0.0)
				.method_22913((float) this.field_19088 / 32.0F, (float) (this.field_19086 + (int) this.method_25341()) / 32.0F)
				.method_1336(32, 32, 32, 255)
				.method_1344();
			bufferBuilder.method_22912(this.field_19087, this.field_19086, 0.0)
				.method_22913((float) this.field_19087 / 32.0F, (float) (this.field_19086 + (int) this.method_25341()) / 32.0F)
				.method_1336(32, 32, 32, 255)
				.method_1344();
			bufferBuilder.method_22912(this.field_19087, this.field_19085, 0.0)
				.method_22913((float) this.field_19087 / 32.0F, (float) (this.field_19085 + (int) this.method_25341()) / 32.0F)
				.method_1336(32, 32, 32, 255)
				.method_1344();
			bufferBuilder.method_22912(this.field_19088, this.field_19085, 0.0)
				.method_22913((float) this.field_19088 / 32.0F, (float) (this.field_19085 + (int) this.method_25341()) / 32.0F)
				.method_1336(32, 32, 32, 255)
				.method_1344();
			tessellator.method_1350();
		}
		int k = this.method_25342();
		int l = this.field_19085 + 4 - (int) this.method_25341();

		this.method_25311(matrices, k, l, mouseX, mouseY, delta);
		this.field_22740.method_1531().method_22813(class_332.field_22735);
		RenderSystem.enableDepthTest();
		RenderSystem.depthFunc(519);
		bufferBuilder.method_1328(7, class_290.field_1575);
		bufferBuilder.method_22912(this.field_19088, this.field_19085, -100.0).method_22913(0.0F, (float) this.field_19085 / 32.0F).method_1336(64, 64, 64, 255).method_1344();
		bufferBuilder.method_22912(this.field_19088 + this.field_22742, this.field_19085, -100.0)
			.method_22913((float) this.field_22742 / 32.0F, (float) this.field_19085 / 32.0F)
			.method_1336(64, 64, 64, 255)
			.method_1344();
		bufferBuilder.method_22912(this.field_19088 + this.field_22742, 0.0, -100.0).method_22913((float) this.field_22742 / 32.0F, 0.0F).method_1336(64, 64, 64, 255).method_1344();
		bufferBuilder.method_22912(this.field_19088, 0.0, -100.0).method_22913(0.0F, 0.0F).method_1336(64, 64, 64, 255).method_1344();
		bufferBuilder.method_22912(this.field_19088, this.field_22743, -100.0).method_22913(0.0F, (float) this.field_22743 / 32.0F).method_1336(64, 64, 64, 255).method_1344();
		bufferBuilder.method_22912(this.field_19088 + this.field_22742, this.field_22743, -100.0)
			.method_22913((float) this.field_22742 / 32.0F, (float) this.field_22743 / 32.0F)
			.method_1336(64, 64, 64, 255)
			.method_1344();
		bufferBuilder.method_22912(this.field_19088 + this.field_22742, this.field_19086, -100.0)
			.method_22913((float) this.field_22742 / 32.0F, (float) this.field_19086 / 32.0F)
			.method_1336(64, 64, 64, 255)
			.method_1344();
		bufferBuilder.method_22912(this.field_19088, this.field_19086, -100.0).method_22913(0.0F, (float) this.field_19086 / 32.0F).method_1336(64, 64, 64, 255).method_1344();
		tessellator.method_1350();
		RenderSystem.depthFunc(515);
		RenderSystem.disableDepthTest();
		RenderSystem.enableBlend();
		RenderSystem.blendFuncSeparate(
			class_4493.class_4535.field_22541, class_4493.class_4534.field_22523, class_4493.class_4535.field_22544, class_4493.class_4534.field_22518
		);
		RenderSystem.disableAlphaTest();
		RenderSystem.shadeModel(7425);
		RenderSystem.disableTexture();
		int n = 4;
		bufferBuilder.method_1328(7, class_290.field_1575);
		bufferBuilder.method_22912(this.field_19088, this.field_19085 + 4, 0.0).method_22913(0.0F, 1.0F).method_1336(0, 0, 0, 0).method_1344();
		bufferBuilder.method_22912(this.field_19087, this.field_19085 + 4, 0.0).method_22913(1.0F, 1.0F).method_1336(0, 0, 0, 0).method_1344();
		bufferBuilder.method_22912(this.field_19087, this.field_19085, 0.0).method_22913(1.0F, 0.0F).method_1336(0, 0, 0, 255).method_1344();
		bufferBuilder.method_22912(this.field_19088, this.field_19085, 0.0).method_22913(0.0F, 0.0F).method_1336(0, 0, 0, 255).method_1344();
		bufferBuilder.method_22912(this.field_19088, this.field_19086, 0.0).method_22913(0.0F, 1.0F).method_1336(0, 0, 0, 255).method_1344();
		bufferBuilder.method_22912(this.field_19087, this.field_19086, 0.0).method_22913(1.0F, 1.0F).method_1336(0, 0, 0, 255).method_1344();
		bufferBuilder.method_22912(this.field_19087, this.field_19086 - 4, 0.0).method_22913(1.0F, 0.0F).method_1336(0, 0, 0, 0).method_1344();
		bufferBuilder.method_22912(this.field_19088, this.field_19086 - 4, 0.0).method_22913(0.0F, 0.0F).method_1336(0, 0, 0, 0).method_1344();
		tessellator.method_1350();
		int o = this.method_25331();
		if (o > 0) {
			int p = (int) ((float) ((this.field_19086 - this.field_19085) * (this.field_19086 - this.field_19085)) / (float) this.method_25317());
			p = class_3532.method_15340(p, 32, this.field_19086 - this.field_19085 - 8);
			int q = (int) this.method_25341() * (this.field_19086 - this.field_19085 - p) / o + this.field_19085;
			if (q < this.field_19085) {
				q = this.field_19085;
			}

			bufferBuilder.method_1328(7, class_290.field_1575);
			bufferBuilder.method_22912(i, this.field_19086, 0.0).method_22913(0.0F, 1.0F).method_1336(0, 0, 0, 255).method_1344();
			bufferBuilder.method_22912(j, this.field_19086, 0.0).method_22913(1.0F, 1.0F).method_1336(0, 0, 0, 255).method_1344();
			bufferBuilder.method_22912(j, this.field_19085, 0.0).method_22913(1.0F, 0.0F).method_1336(0, 0, 0, 255).method_1344();
			bufferBuilder.method_22912(i, this.field_19085, 0.0).method_22913(0.0F, 0.0F).method_1336(0, 0, 0, 255).method_1344();
			bufferBuilder.method_22912(i, q + p, 0.0).method_22913(0.0F, 1.0F).method_1336(128, 128, 128, 255).method_1344();
			bufferBuilder.method_22912(j, q + p, 0.0).method_22913(1.0F, 1.0F).method_1336(128, 128, 128, 255).method_1344();
			bufferBuilder.method_22912(j, q, 0.0).method_22913(1.0F, 0.0F).method_1336(128, 128, 128, 255).method_1344();
			bufferBuilder.method_22912(i, q, 0.0).method_22913(0.0F, 0.0F).method_1336(128, 128, 128, 255).method_1344();
			bufferBuilder.method_22912(i, q + p - 1, 0.0).method_22913(0.0F, 1.0F).method_1336(192, 192, 192, 255).method_1344();
			bufferBuilder.method_22912(j - 1, q + p - 1, 0.0).method_22913(1.0F, 1.0F).method_1336(192, 192, 192, 255).method_1344();
			bufferBuilder.method_22912(j - 1, q, 0.0).method_22913(1.0F, 0.0F).method_1336(192, 192, 192, 255).method_1344();
			bufferBuilder.method_22912(i, q, 0.0).method_22913(0.0F, 0.0F).method_1336(192, 192, 192, 255).method_1344();
			tessellator.method_1350();
		}

		RenderSystem.enableTexture();
		RenderSystem.shadeModel(7424);
		RenderSystem.enableAlphaTest();
		RenderSystem.disableBlend();
	}

	public class ChatLine extends Entry<ChatLine> {
		protected final class_310 client = class_310.method_1551();
		@Getter
		private final class_5481 content;
		@Getter
		private final ChatMessage origin;

		public ChatLine(class_5481 content, ChatMessage origin) {
			super();
			this.content = content;
			this.origin = origin;
		}

		@Override
		public boolean mouseClicked(double mouseX, double mouseY, int button) {
			if (button == 0) {
				ChatWidget.this.method_25313(this);
				return true;
			}
			if (button == 1) {
				ContextMenu.Builder builder = ContextMenu.builder()
					.title(class_2561.method_30163(origin.sender().getName()))
					.spacer();
				if (!origin.sender().equals(API.getInstance().getSelf())) {
					builder.entry(new class_2588("api.friends.chat"), buttonWidget -> {
							ChannelRequest.getOrCreateDM(origin.sender())
								.whenCompleteAsync((channel, throwable) -> client.execute(() -> client.method_1507(new ChatScreen(screen.getParent(), channel))));
						})
						.spacer();
				}
				builder.entry(new class_2588("api.chat.report.message"), buttonWidget -> {
						class_437 previous = client.field_1755;
						client.method_1507(new class_410(b -> {
							if (b) {
								ChatHandler.getInstance().reportMessage(origin);
							}
							client.method_1507(previous);
						}, new class_2588("api.channels.confirm_report"), new class_2588("api.channels.confirm_report.desc", origin.content())));
					})
					.spacer()
					.entry(new class_2588("action.copy"), buttonWidget -> {
						client.field_1774.method_1455(origin.content());
					});
				screen.setContextMenu(builder.build());
				return true;
			}
			return false;
		}

		protected void renderExtras(class_4587 graphics, int x, int y, int mouseX, int mouseY) {
		}

		@Override
		public void render(class_4587 graphics, int index, int y, int x, int entryWidth, int entryHeight, int mouseX, int mouseY, boolean hovered, float tickDelta) {
			for (ChatLine l : method_25396()) {
				if (l.getOrigin().equals(origin)) {
					if (l.isMouseOver(mouseX, mouseY) && Objects.equals(method_25308(mouseX, mouseY), l)) {
						hovered = true;
						break;
					}
				}
			}
			if (hovered && !screen.hasContextMenu()) {
				method_25294(graphics, x - 2 - 22, y - 2, x + entryWidth + 20, y + entryHeight - 1, 0x33FFFFFF);
				if (index < method_25396().size() - 1 && method_25396().get(index + 1).getOrigin().equals(origin)) {
					method_25294(graphics, -2 - 22, y + entryHeight - 1, x + entryWidth + 20, y + entryHeight + 2, 0x33FFFFFF);
				}
				if ((index < method_25396().size() - 1 && !method_25396().get(index + 1).getOrigin().equals(origin)) || index == method_25396().size() - 1) {
					method_25294(graphics, -2 - 22, y + entryHeight - 1, x + entryWidth + 20, y + entryHeight, 0x33FFFFFF);
				}
			}
			renderExtras(graphics, x, y, mouseX, mouseY);
			client.field_1772.method_27528(graphics, content, x, y, -1);
		}
	}

	public class NameChatLine extends ChatLine {

		private final String formattedTime;

		public NameChatLine(ChatMessage message) {
			super(new class_2585(message.senderDisplayName())
				.method_10862(class_2583.field_24360.method_10982(true)).method_30937(), message);

			DateTimeFormatter DATE_FORMAT = DateTimeFormatter.ofPattern("d/M/yyyy H:mm");
			formattedTime = DATE_FORMAT.format(message.timestamp().atZone(ZoneId.systemDefault()));
		}

		@Override
		protected void renderExtras(class_4587 graphics, int x, int y, int mouseX, int mouseY) {
			RenderSystem.disableBlend();
			class_2960 texture = Auth.getInstance().getSkinTexture(getOrigin().sender().getUuid(),
				getOrigin().sender().getName());
			client.method_1531().method_22813(texture);
			method_25293(graphics, x - 22, y, 18, 18, 8, 8, 8, 8, 64, 64);
			method_25293(graphics, x - 22, y, 18, 18, 40, 8, 8, 8, 64, 64);
			RenderSystem.enableBlend();
			client.field_1772.method_1729(graphics, formattedTime, client.field_1772.method_1727(getContent()) + x + 5, y, ClientColors.GRAY.toInt());
		}
	}
}
