/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.api.multiplayer;

import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.stream.Collectors;

import com.google.common.hash.Hashing;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.mojang.blaze3d.systems.RenderSystem;
import io.github.axolotlclient.api.types.PkSystem;
import io.github.axolotlclient.api.types.Status;
import io.github.axolotlclient.api.types.User;
import io.github.axolotlclient.modules.auth.Auth;
import lombok.Getter;
import lombok.Setter;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1011;
import net.minecraft.class_1043;
import net.minecraft.class_124;
import net.minecraft.class_140;
import net.minecraft.class_155;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2585;
import net.minecraft.class_2588;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_4280;
import net.minecraft.class_4587;
import net.minecraft.class_5250;
import net.minecraft.class_5251;
import net.minecraft.class_5481;
import net.minecraft.class_642;
import net.minecraft.text.*;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.Nullable;

public class FriendsMultiplayerSelectionList extends class_4280<FriendsMultiplayerSelectionList.Entry> {
	private static final class_2960 UNKNOWN_SERVER_TEXTURE = new class_2960("textures/misc/unknown_server.png");
	static final class_2960 SERVER_SELECTION_TEXTURE = new class_2960("textures/gui/server_selection.png");
	static final class_2960 ICONS = new class_2960("textures/gui/icons.png");
	static final Logger LOGGER = LogManager.getLogger();
	static final ThreadPoolExecutor THREAD_POOL = new ScheduledThreadPoolExecutor(
		5,
		new ThreadFactoryBuilder()
			.setNameFormat("Friends Server Pinger #%d")
			.setDaemon(true)
			.setUncaughtExceptionHandler(new class_140(LOGGER))
			.build()
	);
	static final class_2561 CANT_RESOLVE_TEXT = new class_2588("multiplayer.status.cannot_resolve").method_27694(style -> style.method_27703(class_5251.method_27717(-65536)));
	static final class_2561 CANT_CONNECT_TEXT = new class_2588("multiplayer.status.cannot_connect").method_27694(style -> style.method_27703(class_5251.method_27717(-65536)));
	static final class_2561 INCOMPATIBLE_STATUS = new class_2588("multiplayer.status.incompatible");
	static final class_2561 NO_CONNECTION_STATUS = new class_2588("multiplayer.status.no_connection");
	static final class_2561 PINGING_STATUS = new class_2588("multiplayer.status.pinging");
	static final class_2561 NOT_PUBLISHED_STATUS = new class_2588("api.worldhost.joinability.not_published").method_27692(class_124.field_1061);
	private final FriendsMultiplayerScreen screen;
	private final List<Entry> friendEntries = new ArrayList<>();
	private final LoadingHeader loadingHeader = new LoadingHeader();

	public FriendsMultiplayerSelectionList(FriendsMultiplayerScreen screen, class_310 minecraft, int width, int height, int y, int itemHeight) {
		super(minecraft, screen.field_22789, screen.field_22790, y, y + height, itemHeight);
		this.screen = screen;
		method_25321(loadingHeader);
	}

	private void refreshEntries() {
		this.method_25339();
		this.friendEntries.forEach(this::method_25321);
	}

	public void setSelected(@Nullable FriendsMultiplayerSelectionList.Entry entry) {
		super.method_25313(entry);
		this.screen.onSelectedChange();
	}

	@Override
	public boolean method_25404(int keyCode, int scanCode, int modifiers) {
		Entry entry = this.method_25334();
		return entry != null && entry.method_25404(keyCode, scanCode, modifiers) || super.method_25404(keyCode, scanCode, modifiers);
	}

	public void updateList(List<User> friends) {
		this.friendEntries.clear();

		for (User friend : friends) {
			if (friend.getStatus().isOnline()) {
				this.friendEntries.add(createEntry(friend));
			}
		}

		this.refreshEntries();
	}

	private Entry createEntry(User friend) {
		if (friend.getStatus().getActivity() != null && friend.getStatus().getActivity().hasMetadata()) {
			if (friend.getStatus().getActivity().hasMetadata(Status.Activity.ExternalServerMetadata.ID)) {
				return externalServerEntry(this.screen, friend);
			} else {
				return e4mcServerFriendEntry(this.screen, friend);
			}
		}
		return new StatusFriendEntry(screen, friend);
	}

	public void updateEntry(User user) {
		this.friendEntries.stream().filter(e1 -> {
			if (e1 instanceof StatusFriendEntry statusFriendEntry) {
				return statusFriendEntry.getUser().equals(user);
			} else if (e1 instanceof ServerEntry serverEntry) {
				return serverEntry.getUser().equals(user);
			}
			return false;
		}).findFirst().ifPresent(e -> {
			this.friendEntries.set(friendEntries.indexOf(e), createEntry(user));
			refreshEntries();
		});
	}

	@Override
	public int method_25322() {
		return 305;
	}

	@Environment(EnvType.CLIENT)
	public abstract static class Entry extends class_4280.class_4281<Entry> implements AutoCloseable {
		public void close() {
		}

		public boolean canJoin() {
			return false;
		}

		public class_642 getServerData() {
			return null;
		}
	}

	@Getter
	public class StatusFriendEntry extends Entry {

		private final User user;

		protected StatusFriendEntry(final FriendsMultiplayerScreen screen, final User friend) {
			this.user = friend;
		}

		@Override
		public void method_25343(class_4587 graphics, int index, int top, int left, int width, int height, int mouseX, int mouseY, boolean hovering, float partialTick) {
			if (user.isSystem()) {
				class_5250 fronters = new class_2585(
					user.getSystem().getFronters().stream().map(PkSystem.Member::getDisplayName)
						.collect(Collectors.joining("/")));
				class_2561 tag = new class_2585("(" + user.getSystem().getName() + "/" + user.getName() + ")")
					.method_10862(class_2583.field_24360.method_10978(true).method_10977(class_124.field_1080));
				field_22740.field_1772.method_30883(graphics, fronters.method_10852(tag), left + 3, top + 1, -1);
			} else {
				field_22740.field_1772.method_1729(graphics, user.getName(), left + 3 + 32, top + 1, -1);
			}

			if (user.getStatus().isOnline() && user.getStatus().getActivity() != null) {
				field_22740.field_1772.method_1729(graphics, user.getStatus().getTitle(), left + 3 + 32, top + 12, 8421504);
				field_22740.field_1772.method_1729(graphics, user.getStatus().getDescription(), left + 3 + 40, top + 23, 8421504);
			} else if (user.getStatus().getLastOnline() != null) {
				field_22740.field_1772.method_1729(graphics, user.getStatus().getLastOnline(), left + 3 + 32, top + 12, 8421504);
			}

			field_22740.method_1531().method_22813(Auth.getInstance().getSkinTexture(user));
			RenderSystem.enableBlend();
			method_25293(graphics, left - 1, top - 1, 32, 32, 8, 8, 8, 8, 64, 64);
			method_25293(graphics, left - 1, top - 1, 32, 32, 40, 8, 8, 8, 64, 64);
			RenderSystem.disableBlend();
		}
	}

	private static final int STATUS_ICON_HEIGHT = 8;
	private static final int STATUS_ICON_WIDTH = 10;

	protected class ServerEntry extends Entry {
		private static final int ICON_WIDTH = 32;
		private static final int ICON_HEIGHT = 32;
		private static final int SPACING = 5;
		private static final int STATUS_ICON_WIDTH = 10;
		private static final int STATUS_ICON_HEIGHT = 8;
		private final FriendsMultiplayerScreen screen;
		private final class_310 minecraft;
		protected final ServerInfoEx serverData;
		private final class_2960 iconId;
		private class_1043 icon;
		private String lastIconBytes;
		private long lastClickTime;
		@Nullable
		private List<class_2561> onlinePlayersTooltip;
		private Sprite statusIcon;
		@Nullable
		private class_2561 statusIconTooltip;
		@Getter
		protected final User user;

		@SuppressWarnings("UnstableApiUsage")
		protected ServerEntry(FriendsMultiplayerScreen screen, class_642 serverData, User user) {
			this.screen = screen;
			this.minecraft = class_310.method_1551();
			this.serverData = new ServerInfoEx(serverData);
			this.iconId = new class_2960("servers/" + Hashing.sha1().hashUnencodedChars(serverData.field_3761 != null ? serverData.field_3761 : user.getUuid() + "_" + serverData.field_3752) + "/icon");
			this.user = user;
		}

		@Override
		public class_642 getServerData() {
			return serverData.serverInfo();
		}

		protected void refreshStatus() {
			this.onlinePlayersTooltip = null;
			if (!isPublished()) {
				this.serverData.setPingResult(PingResult.UNREACHABLE);
			}
			switch (this.serverData.pingResult()) {
				case INITIAL:
				case PINGING:
					this.statusIcon = Sprite.PING_1_SPRITE;
					this.statusIconTooltip = FriendsMultiplayerSelectionList.PINGING_STATUS;
					break;
				case INCOMPATIBLE:
					this.statusIcon = Sprite.INCOMPATIBLE_SPRITE;
					this.onlinePlayersTooltip = this.serverData.serverInfo.field_3762;
					this.statusIconTooltip = FriendsMultiplayerSelectionList.INCOMPATIBLE_STATUS;
					break;
				case UNREACHABLE:
					this.statusIcon = Sprite.UNREACHABLE_SPRITE;
					if (!isPublished()) {
						break;
					}
					this.statusIconTooltip = FriendsMultiplayerSelectionList.NO_CONNECTION_STATUS;
					break;
				case SUCCESSFUL:
					if (this.serverData.serverInfo.field_3758 < 150L) {
						this.statusIcon = Sprite.PING_5_SPRITE;
					} else if (this.serverData.serverInfo.field_3758 < 300L) {
						this.statusIcon = Sprite.PING_4_SPRITE;
					} else if (this.serverData.serverInfo.field_3758 < 600L) {
						this.statusIcon = Sprite.PING_3_SPRITE;
					} else if (this.serverData.serverInfo.field_3758 < 1000L) {
						this.statusIcon = Sprite.PING_2_SPRITE;
					} else {
						this.statusIcon = Sprite.PING_1_SPRITE;
					}

					this.statusIconTooltip = new class_2588("multiplayer.status.ping", this.serverData.serverInfo.field_3758);
					this.onlinePlayersTooltip = this.serverData.serverInfo.field_3762;
			}
		}

		@Override
		public void method_25343(class_4587 guiGraphics, int index, int top, int left, int width, int height, int mouseX, int mouseY, boolean hovering, float partialTick) {
			if (this.serverData.pingResult() == PingResult.INITIAL) {
				this.serverData.setPingResult(PingResult.PINGING);
				this.serverData.serverInfo.field_3757 = class_2585.field_24366;
				this.serverData.serverInfo.field_3753 = class_2585.field_24366;
				FriendsMultiplayerSelectionList.THREAD_POOL
					.submit(
						() -> {
							try {
								this.screen
									.getPinger()
									.add(
										this.serverData.serverInfo,
										() -> {
										}
									);
							} catch (UnknownHostException var2) {
								this.serverData.setPingResult(PingResult.UNREACHABLE);
								this.serverData.serverInfo.field_3757 = FriendsMultiplayerSelectionList.CANT_RESOLVE_TEXT;
								this.minecraft.execute(this::refreshStatus);
							} catch (Exception var3) {
								this.serverData.setPingResult(PingResult.UNREACHABLE);
								this.serverData.serverInfo.field_3757 = FriendsMultiplayerSelectionList.CANT_CONNECT_TEXT;
								this.minecraft.execute(this::refreshStatus);
							}
						}
					);
			}

			if (serverData.pingResult == PingResult.PINGING && serverData.serverInfo.field_3758 != -2) {
				this.serverData.setPingResult(
					this.serverData.serverInfo.field_3756 == class_155.method_16673().getProtocolVersion() ? PingResult.SUCCESSFUL : PingResult.INCOMPATIBLE
				);
			}
			refreshStatus();

			method_25303(guiGraphics, minecraft.field_1772, this.serverData.serverInfo.field_3752, left + ICON_WIDTH + 3, top + 1, -1);
			List<class_5481> list = this.minecraft.field_1772.method_1728(this.serverData.serverInfo.field_3757, width - ICON_WIDTH - 2);

			for (int i = 0; i < Math.min(list.size(), 2); i++) {
				minecraft.field_1772.method_27517(guiGraphics, list.get(i), left + ICON_WIDTH + 3, top + 12 + 9 * i, -8355712);
			}

			field_22740.method_1531().method_22813(this.icon != null ? this.iconId : UNKNOWN_SERVER_TEXTURE);
			method_25290(guiGraphics, left, top, 0.0F, 0.0F, 32, 32, 32, 32);
			field_22740.method_1531().method_22813(Auth.getInstance().getSkinTexture(user));
			RenderSystem.enableBlend();
			method_25293(guiGraphics, left + ICON_WIDTH - 10, top + ICON_HEIGHT - 10, 10, 10, 8, 8, 8, 8, 64, 64);
			method_25293(guiGraphics, left + ICON_WIDTH - 10, top + ICON_HEIGHT - 10, 10, 10, 40, 8, 8, 8, 64, 64);
			RenderSystem.disableBlend();
			if (this.serverData.pingResult() == PingResult.PINGING) {
				int i = (int) (class_156.method_658() / 100L + index * 2 & 7L);
				if (i > 4) {
					i = 8 - i;
				}
				this.statusIcon = switch (i) {
					case 1 -> Sprite.PINGING_2_SPRITE;
					case 2 -> Sprite.PINGING_3_SPRITE;
					case 3 -> Sprite.PINGING_4_SPRITE;
					case 4 -> Sprite.PINGING_5_SPRITE;
					default -> Sprite.PINGING_1_SPRITE;
				};
			}

			int i = left + width - STATUS_ICON_WIDTH - SPACING;
			if (this.statusIcon != null) {
				statusIcon.draw(guiGraphics, i, top);
			}

			String bs = this.serverData.serverInfo.method_2991();
			if (!Objects.equals(bs, this.lastIconBytes)) {
				if (this.uploadIcon(bs)) {
					this.lastIconBytes = bs;
				} else {
					this.serverData.serverInfo.method_2989(null);
				}
			}

			class_2561 component;
			if (!isPublished()) {
				component = NOT_PUBLISHED_STATUS;
			} else {
				if (this.serverData.pingResult() == PingResult.INCOMPATIBLE) {
					component = this.serverData.serverInfo.field_3760.method_27662().method_27692(class_124.field_1061);
				} else {
					component = this.serverData.serverInfo.field_3753;
				}
			}
			int j = this.minecraft.field_1772.method_27525(component);
			int k = i - j - SPACING;
			minecraft.field_1772.method_30881(guiGraphics, component, k, top + 1, -8355712);
			if (this.statusIconTooltip != null && mouseX >= i && mouseX <= i + STATUS_ICON_WIDTH && mouseY >= top && mouseY <= top + STATUS_ICON_HEIGHT) {
				this.screen.setDeferredTooltip(this.statusIconTooltip);
			} else if (this.onlinePlayersTooltip != null && mouseX >= k && mouseX <= k + j && mouseY >= top && mouseY <= top - 1 + 9) {
				this.screen.setDeferredTooltip(this.onlinePlayersTooltip);
			}

			if (this.minecraft.field_1690.field_1854 || hovering) {
				int l = mouseX - left;
				int m = mouseY - top;
				if (this.canJoin()) {
					method_25294(guiGraphics, left, top, left + ICON_WIDTH, top + ICON_HEIGHT, -1601138544);
					if (l < ICON_WIDTH && l > ICON_WIDTH / 2) {
						Sprite.JOIN_HIGHLIGHTED_SPRITE.draw(guiGraphics, left, top);
					} else {
						Sprite.JOIN_SPRITE.draw(guiGraphics, left, top);
					}
				}
			}
		}

		protected boolean isPublished() {
			return true;
		}

		@Override
		public boolean canJoin() {
			return serverData.pingResult() == PingResult.SUCCESSFUL && isPublished();
		}

		private boolean uploadIcon(String iconBytes) {
			if (iconBytes == null) {
				this.minecraft.method_1531().method_4615(this.iconId);
				if (this.icon != null && this.icon.method_4525() != null) {
					this.icon.method_4525().close();
				}

				this.icon = null;
			} else {
				try {
					class_1011 nativeImage = class_1011.method_15990(iconBytes);
					if (nativeImage.method_4307() != 64 || nativeImage.method_4323() != 64) {
						throw new IllegalArgumentException("Icon must be 64x64, but was " + nativeImage.method_4307() + "x" + nativeImage.method_4323());
					}
					if (this.icon == null) {
						this.icon = new class_1043(nativeImage);
					} else {
						this.icon.method_4526(nativeImage);
						this.icon.method_4524();
					}

					this.minecraft.method_1531().method_4616(this.iconId, this.icon);
				} catch (Throwable var3) {
					FriendsMultiplayerSelectionList.LOGGER.error("Invalid icon for server {} ({})", this.serverData.serverInfo.field_3752, this.serverData.serverInfo().field_3761, var3);
					return false;
				}
			}

			return true;
		}

		@Override
		public boolean method_25402(double mouseX, double mouseY, int button) {
			double d = mouseX - FriendsMultiplayerSelectionList.this.method_25342();
			double e = mouseY - FriendsMultiplayerSelectionList.this.method_25337(FriendsMultiplayerSelectionList.this.method_25396().indexOf(this));
			if (d <= 32.0) {
				if (d < 32.0 && d > 16.0 && this.canJoin()) {
					this.screen.setSelected(this);
					this.screen.joinSelectedServer();
					return true;
				}
			}

			this.screen.setSelected(this);
			if (class_156.method_658() - this.lastClickTime < 250L && canJoin()) {
				this.screen.joinSelectedServer();
			}

			this.lastClickTime = class_156.method_658();
			return super.method_25402(mouseX, mouseY, button);
		}

		@Override
		public void close() {
			this.icon.close();
		}
	}

	private ExternalServerFriendEntry externalServerEntry(FriendsMultiplayerScreen screen, User friend) {
		Status.Activity.ExternalServerMetadata metadata = (Status.Activity.ExternalServerMetadata) friend.getStatus().getActivity().metadata().attributes();
		return new ExternalServerFriendEntry(screen, metadata, new class_642(metadata.serverName(), metadata.address(), false), friend);
	}

	public class ExternalServerFriendEntry extends ServerEntry {
		private final Status.Activity.ExternalServerMetadata statusDescription;

		private ExternalServerFriendEntry(FriendsMultiplayerScreen screen, Status.Activity.ExternalServerMetadata statusDescription, class_642 serverData, User friend) {
			super(screen, serverData, friend);
			this.statusDescription = statusDescription;
			refreshStatus();
		}

		@Override
		public boolean canJoin() {
			return statusDescription.address() != null;
		}

	}

	private E4mcServerFriendEntry e4mcServerFriendEntry(FriendsMultiplayerScreen screen, User friend) {
		var activity = friend.getStatus().getActivity();
		Status.Activity.E4mcMetadata metadata;
		if (activity.hasMetadata(Status.Activity.WorldHostMetadata.ID)) {
			metadata = ((Status.Activity.WorldHostMetadata) activity.metadata().attributes()).asE4mcMetadata();
		} else {
			metadata = (Status.Activity.E4mcMetadata) activity.metadata().attributes();
		}
		return new E4mcServerFriendEntry(screen, metadata, ServerInfoUtil.getServerData(friend.getName(), metadata), friend);
	}

	public class E4mcServerFriendEntry extends ServerEntry {

		private final Status.Activity.E4mcMetadata statusDescription;

		protected E4mcServerFriendEntry(FriendsMultiplayerScreen screen, Status.Activity.E4mcMetadata statusDescription, class_642 serverData, User friend) {
			super(screen, serverData, friend);
			this.statusDescription = statusDescription;
			refreshStatus();
		}

		@Override
		protected boolean isPublished() {
			return statusDescription.domain() != null;
		}

		@Override
		protected void refreshStatus() {
			super.refreshStatus();
			serverData.serverInfo.field_3757 = class_2561.method_30163(statusDescription.serverInfo().levelName());
		}
	}

	@Environment(EnvType.CLIENT)
	public static class LoadingHeader extends Entry {
		private final class_310 minecraft = class_310.method_1551();

		@Override
		public void method_25343(class_4587 guiGraphics, int index, int top, int left, int width, int height, int mouseX, int mouseY, boolean hovering, float partialTick) {
			int i = top + height / 2 - 9 / 2;

			String string = switch ((int) (class_156.method_658() / 300L % 4L)) {
				case 1, 3 -> "o O o";
				case 2 -> "o o O";
				default -> "O o o";
			};
			minecraft.field_1772.method_1720(guiGraphics, string, this.minecraft.field_1755.field_22789 / 2f - this.minecraft.field_1772.method_1727(string) / 2f, i, -8355712);
		}
	}

	protected static final class ServerInfoEx {
		private final class_642 serverInfo;
		@Setter
		private PingResult pingResult;

		private ServerInfoEx(class_642 serverInfo) {
			this.serverInfo = serverInfo;
			this.pingResult = PingResult.INITIAL;
		}

		public class_642 serverInfo() {
			return serverInfo;
		}

		public PingResult pingResult() {
			return pingResult;
		}

	}

	protected enum PingResult {
		INITIAL,
		PINGING,
		UNREACHABLE,
		INCOMPATIBLE,
		SUCCESSFUL
	}

	private enum Sprite {
		UNREACHABLE_SPRITE(0, 5),
		INCOMPATIBLE_SPRITE(0, 5),
		PING_1_SPRITE(0, 4),
		PING_2_SPRITE(0, 3),
		PING_3_SPRITE(0, 2),
		PING_4_SPRITE(0, 3),
		PING_5_SPRITE(0, 0),
		PINGING_1_SPRITE(10, 0),
		PINGING_2_SPRITE(10, 1),
		PINGING_3_SPRITE(10, 2),
		PINGING_4_SPRITE(10, 3),
		PINGING_5_SPRITE(10, 4),
		JOIN_HIGHLIGHTED_SPRITE(0, 32, 32, 32, SERVER_SELECTION_TEXTURE),
		JOIN_SPRITE(0, 0, 32, 32, SERVER_SELECTION_TEXTURE),

		;
		private final int u, v, texWidth, texHeight;
		private final class_2960 atlas;

		Sprite(int u, int v, int texWidth, int texHeight, class_2960 texture) {
			this.u = u;
			this.v = v;
			this.texWidth = texWidth;
			this.texHeight = texHeight;
			this.atlas = texture;
		}

		Sprite(int u, int v) {
			this(u, 176 + v * 8, STATUS_ICON_WIDTH, STATUS_ICON_HEIGHT, ICONS);
		}

		public void draw(class_4587 graphics, int x, int y) {
			class_310.method_1551().method_1531().method_22813(atlas);
			class_332.method_25290(graphics, x, y, u, v, texWidth, texHeight, 256, 256);
		}

	}
}
