/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.auth.skin;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.util.*;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.mojang.blaze3d.systems.RenderSystem;
import io.github.axolotlclient.AxolotlClientCommon;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Colors;
import io.github.axolotlclient.api.SimpleTextInputScreen;
import io.github.axolotlclient.api.util.UUIDHelper;
import io.github.axolotlclient.modules.auth.Account;
import io.github.axolotlclient.modules.auth.Auth;
import io.github.axolotlclient.modules.auth.MSApi;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import io.github.axolotlclient.util.ButtonWidgetTextures;
import io.github.axolotlclient.util.ClientColors;
import io.github.axolotlclient.util.Watcher;
import io.github.axolotlclient.util.notifications.Notifications;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_2585;
import net.minecraft.class_2588;
import net.minecraft.class_289;
import net.minecraft.class_290;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_339;
import net.minecraft.class_364;
import net.minecraft.class_4068;
import net.minecraft.class_4069;
import net.minecraft.class_410;
import net.minecraft.class_4185;
import net.minecraft.class_4265;
import net.minecraft.class_437;
import net.minecraft.class_4587;
import net.minecraft.class_5244;
import net.minecraft.class_5250;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class SkinManagementScreen extends class_437 {
	private static final Path SKINS_DIR = FabricLoader.getInstance().getGameDir().resolve("skins");
	private static final int LIST_SKIN_WIDTH = 75;
	private static final int LIST_SKIN_HEIGHT = 110;
	private static final class_2561 TEXT_EQUIPPING = new class_2588("skins.manage.equipping");
	private final class_437 parent;
	private final Account account;
	private MSApi.MCProfile cachedProfile;
	private SkinListWidget skinList;
	private SkinListWidget capesList;
	private boolean capesTab;
	private SkinWidget current;
	private final Watcher skinDirWatcher;
	private final List<class_4068> drawables = new ArrayList<>();
	private final CompletableFuture<MSApi.MCProfile> loadingFuture;
	private class_2561 tooltip;

	public SkinManagementScreen(class_437 parent, Account account) {
		super(new class_2588("skins.manage"));
		this.parent = parent;
		this.account = account;
		skinDirWatcher = Watcher.createSelfTicking(SKINS_DIR, () -> {
			AxolotlClientCommon.getInstance().getLogger().info("Reloading screen as local files changed!");
			loadSkinsList();
		});
		loadingFuture = (account.needsRefresh() ? account.refresh(Auth.getInstance().getMsApi())
			: CompletableFuture.completedFuture(null))
			.thenComposeAsync(unused -> Auth.getInstance().getMsApi().getProfile(account));
	}

	@Override
	public void method_25426() {
		int headerHeight = 33;
		int contentHeight = field_22790 - headerHeight * 2;

		var back = addDrawableChild(new class_4185(field_22789 / 2 - 75, field_22790 - headerHeight / 2 - 10, 150, 20, class_5244.field_24339, btn -> method_25419()));

		var loadingPlaceholder = new class_339(0, headerHeight, field_22789, contentHeight, new class_2588("skins.loading")) {
			@Override
			public void method_25359(class_4587 graphics, int mouseX, int mouseY, float delta) {
				int centerX = this.field_22760 + this.method_25368() / 2;
				int centerY = this.field_22761 + this.method_25364() / 2;
				class_2561 text = this.method_25369();
				field_22793.method_30883(graphics, text, centerX - field_22793.method_27525(text) / 2f, centerY - 9, -1);
				String string = switch ((int) (class_156.method_658() / 300L % 4L)) {
					case 1, 3 -> "o O o";
					case 2 -> "o o O";
					default -> "O o o";
				};
				field_22793.method_1729(graphics, string, centerX - field_22793.method_1727(string) / 2f, centerY + 9, 0xFF808080);
			}

			@Override
			protected class_5250 method_25360() {
				return class_2585.field_24366.method_27662();
			}
		};
		loadingPlaceholder.field_22763 = false;
		addDrawableChild(loadingPlaceholder);
		addDrawableChild(back);

		skinList = new SkinListWidget(field_22787, field_22789 / 2, contentHeight - 24, headerHeight + 24, LIST_SKIN_HEIGHT + 34);
		capesList = new SkinListWidget(field_22787, field_22789 / 2, contentHeight - 24, headerHeight + 24, skinList.getEntryContentsHeight() + 24);
		skinList.method_25333(field_22789 / 2);
		capesList.method_25333(field_22789 / 2);
		var currentHeight = Math.min((field_22789 / 2f) * 120 / 85, contentHeight);
		var currentWidth = currentHeight * 85 / 120;
		current = new SkinWidget((int) currentWidth, (int) currentHeight, null, account);
		current.setPosition((int) (field_22789 / 4f - currentWidth / 2), (int) (field_22790 / 2f - currentHeight / 2));

		if (!capesTab) {
			capesList.visible = capesList.active = false;
		} else {
			skinList.visible = skinList.active = false;
		}
		List<class_339> navBar = new ArrayList<>();
		var skinsTab = new class_4185(Math.max(field_22789 * 3 / 4 - 102, field_22789 / 2 + 2), headerHeight, Math.min(100, field_22789 / 4 - 2), 20, new class_2588("skins.nav.skins"), btn -> {
			navBar.forEach(w -> {
				if (w != btn) w.field_22763 = true;
			});
			btn.field_22763 = false;
			skinList.visible = skinList.active = true;
			capesList.visible = capesList.active = false;
			capesTab = false;
		});
		navBar.add(skinsTab);
		var capesTab = new class_4185(field_22789 * 3 / 4 + 2, headerHeight, Math.min(100, field_22789 / 4 - 2), 20, new class_2588("skins.nav.capes"), btn -> {
			navBar.forEach(w -> {
				if (w != btn) w.field_22763 = true;
			});
			btn.field_22763 = false;
			skinList.visible = skinList.active = false;
			capesList.visible = capesList.active = true;
			this.capesTab = true;
		});
		navBar.add(capesTab);
		var importButton = new SpriteButton(new class_2588("skins.manage.import.local"), btn -> {
			btn.field_22763 = false;
			SkinImportUtil.openImportSkinDialog().thenAccept(this::method_29638).thenRun(() -> btn.field_22763 = true);
		}, new class_2960("axolotlclient", "textures/gui/sprites/folder.png"));
		var downloadButton = new SpriteButton(new class_2588("skins.manage.import.online"), btn -> {
			btn.field_22763 = false;
			promptForSkinDownload();
		}, new class_2960("axolotlclient", "textures/gui/sprites/download.png"));
		downloadButton.field_22760 = importButton.field_22760 - 2 - 11;
		downloadButton.field_22761 = capesTab.field_22761 - 13;
		if (field_22789 - (capesTab.field_22760 + capesTab.method_25368()) > 28) {
			importButton.field_22760 = field_22789 - importButton.method_25368() - 2;
			downloadButton.field_22760 = importButton.field_22760 - downloadButton.method_25368() - 2;
			importButton.field_22761 = capesTab.field_22761 + capesTab.method_25364() - 11;
			downloadButton.field_22761 = importButton.field_22761;
		} else {
			importButton.field_22760 = capesTab.field_22760 + capesTab.method_25368() - 11;
			importButton.field_22761 = capesTab.field_22761 - 13;
			downloadButton.field_22760 = importButton.field_22760 - 2 - 11;
			downloadButton.field_22761 = importButton.field_22761;
		}
		skinsTab.field_22763 = this.capesTab;
		capesTab.field_22763 = !this.capesTab;
		Runnable addWidgets = () -> {
			clear();
			addDrawableChild(current);
			addDrawableChild(skinsTab);
			addDrawableChild(capesTab);
			addDrawableChild(downloadButton);
			addDrawableChild(importButton);
			addDrawableChild(skinList);
			addDrawableChild(capesList);
			addDrawableChild(back);
		};
		if (cachedProfile != null) {
			initDisplay();
			addWidgets.run();
			return;
		}
		loadingFuture.thenAcceptAsync(profile -> {
				cachedProfile = profile;
				initDisplay();
				addWidgets.run();
			}).exceptionally(t -> {
				if (t.getCause() instanceof CancellationException) {
					field_22787.method_1507(parent);
					return null;
				}
				AxolotlClientCommon.getInstance().getLogger().error("Failed to load skins!", t);
				var error = new class_2588("skins.error.failed_to_load");
				var errorDesc = new class_2588("skins.error.failed_to_load_desc");
				clear();
				addDrawableChild(back);
				class TextWidget extends class_339 {

					public TextWidget(int x, int y, int width, int height, class_2561 message, class_327 textRenderer) {
						super(x, y, width, height, message);
						field_22763 = false;
					}

					@Override
					public void method_25359(class_4587 matrices, int mouseX, int mouseY, float delta) {
						method_27534(matrices, field_22793, method_25369(), field_22760 + method_25368() / 2, field_22761 + method_25364() / 2 - field_22793.field_2000 / 2, -1);
					}
				}
				addDrawableChild(new TextWidget(field_22789 / 2 - field_22793.method_27525(error) / 2, field_22790 / 2 - field_22793.field_2000 - 2, field_22793.method_27525(error), field_22793.field_2000, error, field_22793));
				addDrawableChild(new TextWidget(field_22789 / 2 - field_22793.method_27525(errorDesc) / 2, field_22790 / 2 + 1, field_22793.method_27525(errorDesc), field_22793.field_2000, errorDesc, field_22793));
				return null;
			});
	}

	private void promptForSkinDownload() {
		field_22787.method_1507(new SimpleTextInputScreen(this, new class_2588("skins.manage.import.online"), new class_2588("skins.manage.import.online.input"), s ->
			UUIDHelper.ensureUuidOpt(s).thenAccept(o -> {
				if (o.isPresent()) {
					AxolotlClientCommon.getInstance().getLogger().info("Downloading skin of {} ({})", s, o.get());
					Auth.getInstance().getMsApi().getTextures(o.get())
						.exceptionally(th -> {
							AxolotlClientCommon.getInstance().getLogger().info("Failed to download skin of {} ({})", s, o.get(), th);
							return null;
						}).thenAccept(t -> {
							if (t == null) {
								Notifications.getInstance().addStatus("skins.notification.title", "skins.notification.import.online.failed_to_download", s);
								return;
							}
							try {
								var bytes = t.skin().join();
								var out = ensureNonexistent(SKINS_DIR.resolve(t.skinKey()));
								Skin.LocalSkin.writeMetadata(out, Map.of(Skin.LocalSkin.CLASSIC_METADATA_KEY, t.classicModel(), "name", t.name(), "uuid", t.id(), "download_time", Instant.now()));
								Files.write(out, bytes);
								client.execute(this::loadSkinsList);
								Notifications.getInstance().addStatus("skins.notification.title", "skins.notification.import.online.downloaded", t.name());
								AxolotlClientCommon.getInstance().getLogger().info("Downloaded skin of {} ({})", t.name(), o.get());
							} catch (IOException e) {
								AxolotlClientCommon.getInstance().getLogger().warn("Failed to write skin file", e);
								Notifications.getInstance().addStatus("skins.notification.title", "skins.notification.import.online.failed_to_save", t.name());
							}
						});
				} else {
					Notifications.getInstance().addStatus("skins.notification.title", "skins.notification.import.online.not_found", s);
				}
			})));
	}

	private <T extends class_4068 & class_364> T addDrawableChild(T child) {
		drawables.add(child);
		return method_25429(child);
	}

	private void clear() {
		field_22786.clear();
		field_22791.clear();
		drawables.clear();
	}

	@Override
	public void method_25394(class_4587 graphics, int mouseX, int mouseY, float delta) {
		tooltip = null;
		method_25420(graphics);
		drawables.forEach(d -> d.method_25394(graphics, mouseX, mouseY, delta));
		method_27534(graphics, field_22793, method_25440(), field_22789 / 2, 33 / 2 - field_22793.field_2000 / 2, -1);
		if (tooltip != null) {
			method_25424(graphics, tooltip, mouseX, mouseY + 20);
		}
	}

	private void initDisplay() {
		loadSkinsList();
		loadCapesList();
	}

	private void refreshCurrentList() {
		if (capesTab) {
			var scroll = capesList.method_25341();
			loadCapesList();
			capesList.method_25307(scroll);
		} else {
			var scroll = skinList.method_25341();
			loadSkinsList();
			skinList.method_25307(scroll);
		}
	}

	private void loadCapesList() {
		List<Row> rows = new ArrayList<>();
		var profile = cachedProfile;
		int columns = Math.max(2, (field_22789 / 2 - 25) / LIST_SKIN_WIDTH);
		var capes = profile.capes();
		var deselectCape = createWidgetForCape(current.getSkin(), null);
		var activeCape = capes.stream().filter(Cape::active).findFirst();
		current.setCape(activeCape.orElse(null));
		deselectCape.noCape(activeCape.isEmpty());
		for (int i = 0; i < capes.size() + 1; i += columns) {
			Entry widget;
			if (i == 0) {
				widget = createEntry(capesList.getEntryContentsHeight(), deselectCape, new class_2588("skins.capes.no_cape"));
			} else {
				var cape = capes.get(i - 1);
				widget = createEntryForCape(current.getSkin(), cape, capesList.getEntryContentsHeight());
			}
			List<class_339> widgets = new ArrayList<>();
			widgets.add(widget);
			for (int c = 1; c < columns; c++) {
				if (!(i < capes.size() + 1 - c)) continue;
				var cape2 = capes.get(i + c - 1);
				Entry widget2 = createEntryForCape(current.getSkin(), cape2, capesList.getEntryContentsHeight());

				widgets.add(widget2);
			}
			rows.add(new Row(widgets));
		}
		field_22787.execute(() -> capesList.method_25314(rows));
	}

	private void loadSkinsList() {
		var profile = cachedProfile;
		int columns = Math.max(2, (field_22789 / 2 - 25) / LIST_SKIN_WIDTH);
		List<Skin> skins = new ArrayList<>(profile.skins());
		var hashes = skins.stream().map(Asset::sha256).collect(Collectors.toSet());
		var defaultSkin = Skin.getDefaultSkin(account);
		var local = new ArrayList<>(loadLocalSkins());
		var localHashes = local.stream().collect(Collectors.toMap(Asset::sha256, Function.identity(), (skin, skin2) -> skin));
		local.removeIf(s -> !localHashes.containsValue(s));
		skins.replaceAll(s -> {
			if (s instanceof MSApi.MCProfile.OnlineSkin online) {
				if (localHashes.containsKey(s.sha256()) && localHashes.get(s.sha256()) instanceof Skin.LocalSkin file) {
					local.remove(localHashes.remove(s.sha256()));
					return new Skin.Shared(file, online);
				}
			}
			return s;
		});
		skins.addAll(local);
		if (!hashes.contains(defaultSkin.sha256())) {
			skins.add(defaultSkin);
		}
		populateSkinList(skins, columns);
	}

	private List<Skin> loadLocalSkins() {
		try {
			Files.createDirectories(SKINS_DIR);
			try (Stream<Path> skins = Files.list(SKINS_DIR)) {
				return skins.filter(Files::isRegularFile).sorted(Comparator.<Path>comparingLong(p -> {
					try {
						return Files.getLastModifiedTime(p).toMillis();
					} catch (IOException e) {
						return 0L;
					}
				}).reversed()).map(Auth.getInstance().getSkinManager()::read).filter(Objects::nonNull).toList();
			}
		} catch (IOException e) {
			AxolotlClientCommon.getInstance().getLogger().warn("Failed to read skins dir!", e);
		}
		return Collections.emptyList();
	}

	private void populateSkinList(List<? extends Skin> skins, int columns) {
		int entryHeight = skinList.getEntryContentsHeight();
		List<Row> rows = new ArrayList<>();
		for (int i = 0; i < skins.size(); i += columns) {
			var s = skins.get(i);
			if (s != null && s.active()) {
				current.setSkin(s);
			}
			var widget = createEntryForSkin(s, entryHeight);
			List<class_339> widgets = new ArrayList<>();
			widgets.add(widget);
			for (int c = 1; c < columns; c++) {
				if (!(i < skins.size() - c)) continue;
				var s2 = skins.get(i + c);
				if (s2 != null && s2.active()) {
					current.setSkin(s2);
				}
				var widget2 = createEntryForSkin(s2, entryHeight);
				widgets.add(widget2);
			}
			rows.add(new Row(widgets));
		}
		field_22787.execute(() -> skinList.method_25314(rows));
	}

	private Path ensureNonexistent(Path p) {
		if (Files.exists(p)) {
			int counter = 0;
			do {
				counter++;
				p = p.resolveSibling(p.getFileName().toString() + "_" + counter);
			} while (Files.exists(p));
		}
		return p;
	}

	@Override
	public void method_29638(List<Path> packs) {
		if (packs.isEmpty()) return;

		CompletableFuture<?>[] futs = new CompletableFuture[packs.size()];
		for (int i = 0; i < packs.size(); i++) {
			Path p = packs.get(i);
			futs[i] = CompletableFuture.runAsync(() -> {
				try {
					var target = ensureNonexistent(SKINS_DIR.resolve(p.getFileName()));
					var skin = Auth.getInstance().getSkinManager().read(p, false);
					if (skin != null) {
						Files.write(target, skin.image());
					} else {
						AxolotlClientCommon.getInstance().getLogger().info("Skipping dragged file {} because it does not seem to be a valid skin!", p);
						Notifications.getInstance().addStatus("skins.notification.title", "skins.notification.not_copied", p.getFileName());
					}
				} catch (IOException e) {
					AxolotlClientCommon.getInstance().getLogger().warn("Failed to copy skin file: ", e);
				}
			}, field_22787);
		}
		CompletableFuture.allOf(futs).thenRun(this::loadSkinsList);
	}

	private @NotNull Entry createEntryForSkin(Skin skin, int entryHeight) {
		return createEntry(entryHeight, new SkinWidget(LIST_SKIN_WIDTH, LIST_SKIN_HEIGHT, skin, account));
	}

	private @NotNull Entry createEntryForCape(Skin currentSkin, Cape cape, int entryHeight) {
		return createEntry(entryHeight, createWidgetForCape(currentSkin, cape), new class_2585(cape.alias()));
	}

	private SkinWidget createWidgetForCape(Skin currentSkin, Cape cape) {
		SkinWidget widget2 = new SkinWidget(LIST_SKIN_WIDTH, LIST_SKIN_HEIGHT, currentSkin, cape, account);
		widget2.setRotationY(210);
		return widget2;
	}

	@Override
	public void method_25410(class_310 client, int width, int height) {
		Auth.getInstance().getSkinManager().releaseAll();
		super.method_25410(client, width, height);
	}

	@Override
	public void method_25432() {
		Auth.getInstance().getSkinManager().releaseAll();
		Watcher.close(skinDirWatcher);
	}

	@Override
	public void method_25419() {
		field_22787.method_1507(parent);
	}

	private SkinListWidget getCurrentList() {
		return capesTab ? capesList : skinList;
	}

	private class SkinListWidget extends class_4265<Row> {
		public boolean active = true, visible = true;

		public SkinListWidget(class_310 minecraft, int width, int height, int y, int entryHeight) {
			super(minecraft, width, SkinManagementScreen.this.field_22790, y, y + height, entryHeight);
			method_25315(false, 0);
			//setRenderBackground(false);
		}

		@Override
		protected int method_25329() {
			return field_19087 - 8;
		}

		@Override
		public int method_25342() {
			return field_19088 + 3;
		}

		@Override
		public int method_25322() {
			if (!(method_25331() > 0)) {
				return field_22742 - 4;
			}
			return field_22742 - 14;
		}

		private int method_25331() {
			return Math.max(0, this.method_25317() - (this.field_19086 - this.field_19085 - 4));
		}

		public int getEntryContentsHeight() {
			return field_22741 - 4;
		}

		@Override
		public void method_25314(Collection<Row> newEntries) {
			super.method_25314(newEntries);
		}

		@Override
		public void centerScrollOn(Row entry) {
			super.method_25324(entry);
		}

		@Override
		public boolean method_25401(double mouseX, double mouseY, double amountY) {
			if (!visible) return false;
			return super.method_25401(mouseX, mouseY, amountY);
		}

		@Override
		public boolean method_25405(double mouseX, double mouseY) {
			return active && visible && super.method_25405(mouseX, mouseY);
		}

		@Override
		public void method_25394(class_4587 graphics, int mouseX, int mouseY, float delta) {
			if (!visible) return;
			super.method_25394(graphics, mouseX, mouseY, delta);
		}
	}

	private class Row extends class_4265.class_4266<Row> {
		private final List<class_339> widgets;

		public Row(List<class_339> entries) {
			this.widgets = entries;
		}

		@Override
		public void method_25343(class_4587 guiGraphics, int index, int top, int left, int width, int height, int mouseX, int mouseY, boolean hovering, float partialTick) {
			int x = left;
			if (widgets.isEmpty()) return;
			int count = widgets.size();
			int padding = ((width - 5 * (count - 1)) / count);
			for (var w : widgets) {
				w.field_22760 = x;
				w.field_22761 = top;
				w.method_25358(padding);
				w.method_25394(guiGraphics, mouseX, mouseY, partialTick);
				x += w.method_25368() + 5;
			}
		}

		@Override
		public @NotNull List<? extends class_364> method_25396() {
			return widgets;
		}

		@Override
		public void method_25395(@Nullable class_364 focused) {
			super.method_25395(focused);
			if (focused != null) {
				getCurrentList().centerScrollOn(this);
			}
		}
	}

	Entry createEntry(int height, SkinWidget widget) {
		return createEntry(height, widget, null);
	}

	Entry createEntry(int height, SkinWidget widget, class_2561 label) {
		return new Entry(height, widget, label);
	}

	private class Entry extends class_339 implements class_4069 {
		private final SkinWidget skinWidget;
		private final @Nullable class_339 label;
		private final List<class_339> actionButtons = new ArrayList<>();
		private final class_339 equipButton;
		private boolean equipping;
		private long equippingStart;
		@Nullable
		private class_364 focused;
		private boolean dragging;

		public Entry(int height, SkinWidget widget, @Nullable class_2561 label) {
			super(0, 0, widget.method_25368(), height, class_2585.field_24366);
			widget.method_25358(method_25368() - 4);
			var asset = widget.getFocusedAsset();
			if (asset instanceof Skin skin) {
				var wideSprite = new class_2960("axolotlclient", "textures/gui/sprites/wide.png");
				var slimSprite = new class_2960("axolotlclient", "textures/gui/sprites/slim.png");
				var slimText = new class_2588("skins.manage.variant.classic");
				var wideText = new class_2588("skins.manage.variant.slim");
				actionButtons.add(new SpriteButton(skin.classicVariant() ? wideText : slimText, btn -> {
					var self = (SpriteButton) btn;
					skin.classicVariant(!skin.classicVariant());
					self.sprite = skin.classicVariant() ? slimSprite : wideSprite;
					self.method_25355(skin.classicVariant() ? wideText : slimText);
				}, skin.classicVariant() ? slimSprite : wideSprite));
			}
			if (asset != null) {
				if (asset instanceof Asset.Local local) {
					this.actionButtons.add(new SpriteButton(new class_2588("skins.manage.delete"), btn -> {
						btn.field_22763 = false;
						field_22787.method_1507(new class_410(confirmed -> {
							field_22787.method_1507(new LoadingScreen(method_25440(), new class_2588("menu.working")));
							if (confirmed) {
								try {
									Files.delete(local.file());
									Skin.LocalSkin.deleteMetadata(local.file());
								} catch (IOException e) {
									AxolotlClientCommon.getInstance().getLogger().warn("Failed to delete: ", e);
								}
							}
							field_22787.method_1507(SkinManagementScreen.this);
							btn.field_22763 = true;
						}, new class_2588("skins.manage.delete.confirm"), (class_2561) (asset.active() ?
							new class_2588("skins.manage.delete.confirm.desc_active") :
							new class_2588("skins.manage.delete.confirm.desc")
						).br$color(Colors.RED.toInt())));
					}, new class_2960("axolotlclient", "textures/gui/sprites/delete.png")));
				}
				if (asset instanceof Asset.Online online && online.supportsDownload() && !(asset instanceof Asset.Local)) {
					this.actionButtons.add(new SpriteButton(new class_2588("skins.manage.download"), btn -> {
						btn.field_22763 = false;
						download(asset).thenRun(() -> {
							refreshCurrentList();
							btn.field_22763 = true;
						});
					}, new class_2960("axolotlclient", "textures/gui/sprites/download.png")));
				}
			}
			if (label != null) {
				this.label = new class_339(0, 0, widget.method_25368(), 16, label) {
					@Override
					public void method_25359(class_4587 guiGraphics, int mouseX, int mouseY, float partialTick) {
						DrawUtil.drawScrollableText(guiGraphics, field_22793, method_25369(), field_22760 + 2, field_22761, field_22760 + field_22758 - 2, field_22761 + field_22759, -1);
					}
				};
				this.label.field_22763 = false;
			} else {
				this.label = null;
			}
			this.equipButton = new class_4185(0, 0, widget.method_25368(), 20, new class_2588(
				widget.isEquipped() ? "skins.manage.equipped" : "skins.manage.equip"),
				btn -> {
					equippingStart = class_156.method_658();
					equipping = true;
					btn.method_25355(TEXT_EQUIPPING);
					btn.field_22763 = false;
					Consumer<CompletableFuture<MSApi.MCProfile>> consumer = f -> f.thenAcceptAsync(p -> {
						cachedProfile = p;
						if (field_22787.field_1755 == SkinManagementScreen.this) {
							refreshCurrentList();
						} else {
							field_22787.execute(() -> field_22787.method_1507(SkinManagementScreen.this));
						}
					}).exceptionally(t -> {
						AxolotlClientCommon.getInstance().getLogger().warn("Failed to equip asset!", t);
						equipping = false;
						return null;
					});
					if (asset instanceof Skin && !(current.getSkin() instanceof Skin.Local)) {
						field_22787.method_1507(new class_410(confirmed -> {
							field_22787.method_1507(new LoadingScreen(method_25440(), TEXT_EQUIPPING));
							if (confirmed) {
								consumer.accept(download(current.getSkin()).thenCompose(a -> widget.equip()));
							} else {
								consumer.accept(widget.equip());
							}
						}, new class_2588("skins.manage.equip.confirm"), new class_2588("skins.manage.equip.download_current")));
					} else {
						consumer.accept(widget.equip());
					}
				});
			this.equipButton.field_22763 = !widget.isEquipped();
			this.skinWidget = widget;
		}

		private @NotNull CompletableFuture<?> download(Asset asset) {
			return CompletableFuture.runAsync(() -> {
				try {
					var out = SKINS_DIR.resolve(asset.sha256());
					Files.createDirectories(out.getParent());
					Files.write(out, asset.image());
					if (asset instanceof Skin skin) {
						Skin.LocalSkin.writeMetadata(out, Map.of(Skin.LocalSkin.CLASSIC_METADATA_KEY, skin.classicVariant()));
					}
				} catch (IOException e) {
					AxolotlClientCommon.getInstance().getLogger().warn("Failed to download: ", e);
				}
			});
		}

		@Override
		public final boolean method_25397() {
			return this.dragging;
		}

		@Override
		public final void method_25398(boolean dragging) {
			this.dragging = dragging;
		}

		@Nullable
		@Override
		public class_364 method_25399() {
			return this.focused;
		}

		@Override
		public void method_25395(@Nullable class_364 child) {
			this.focused = child;
		}

		@Override
		public boolean method_25402(double mouseX, double mouseY, int button) {
			return class_4069.super.method_25402(mouseX, mouseY, button);
		}

		@Override
		public boolean method_25406(double mouseX, double mouseY, int button) {
			return class_4069.super.method_25406(mouseX, mouseY, button);
		}

		@Override
		public boolean method_25403(double mouseX, double mouseY, int button, double deltaX, double deltaY) {
			return class_4069.super.method_25403(mouseX, mouseY, button, deltaX, deltaY);
		}

		@Override
		public boolean method_25370() {
			return method_25399() != null;
		}

		@Override
		public void method_25365(boolean focused) {

		}

		@Override
		public @NotNull List<? extends class_364> method_25396() {
			return Stream.concat(actionButtons.stream(), Stream.of(skinWidget, label, equipButton)).filter(Objects::nonNull).toList();
		}

		private float applyEasing(float x) {
			return x * x * x;
		}

		@Override
		public void method_25359(class_4587 guiGraphics, int mouseX, int mouseY, float partialTick) {
			int y = this.field_22761 + 4;
			int x = this.field_22760 + 2;
			skinWidget.setPosition(x, y);
			skinWidget.method_25358(method_25368() - 4);
			if (skinWidget.isEquipped() || equipping) {
				long prog;
				if (Auth.getInstance().skinManagerAnimations.get()) {
					if (equipping) prog = (class_156.method_658() - equippingStart) / 20 % 100;
					else prog = Math.abs((class_156.method_658() / 30 % 200) - 100);
				} else prog = 100;
				var percent = (prog / 100f);
				float gradientWidth;
				if (equipping) {
					gradientWidth = percent * Math.min(method_25368() / 3f, method_25364() / 3f);
				} else {
					gradientWidth = Math.min(method_25368() / 15f, method_25364() / 6f) + applyEasing(percent) * Math.min(method_25368() * 2 / 15f, method_25364() / 6f);
				}
				GradientHoleRectangleRenderState.render(guiGraphics, this.field_22760 + 2, this.field_22761 + 2, this.field_22760 + method_25368() - 2,
					skinWidget.getY() + skinWidget.method_25364() + 2,
					gradientWidth,
					equipping ? 0xFFFF0088 : ClientColors.SELECTOR_GREEN.toInt(), 0);
			}
			skinWidget.method_25394(guiGraphics, mouseX, mouseY, partialTick);
			int actionButtonY = this.field_22761 + 2;
			for (var button : actionButtons) {
				button.field_22760 = skinWidget.getX() + skinWidget.method_25368() - button.method_25368();
				button.field_22761 = actionButtonY;
				if (method_25367() || button.method_25367()) {
					button.method_25394(guiGraphics, mouseX, mouseY, partialTick);
				}
				actionButtonY += button.method_25364() + 2;
			}
			if (label != null) {
				label.field_22760 = x;
				label.field_22761 = skinWidget.getY() + skinWidget.method_25364() + 6;
				label.method_25394(guiGraphics, mouseX, mouseY, partialTick);
				label.method_25358(method_25368() - 4);
				equipButton.field_22760 = x;
				equipButton.field_22761 = label.field_22761 + label.method_25364() + 2;
			} else {
				equipButton.field_22760 = x;
				equipButton.field_22761 = skinWidget.getY() + skinWidget.method_25364() + 4;
			}
			equipButton.method_25358(method_25368() - 4);
			equipButton.method_25394(guiGraphics, mouseX, mouseY, partialTick);

			if (method_25367()) {
				guiGraphics.br$outlineRect(this.field_22760, this.field_22761, method_25368(), method_25364(), -1);
			}
		}

		private static class GradientHoleRectangleRenderState {

			public static void render(class_4587 graphics, int x0, int y0, int x1, int y1, float gradientWidth, int col1, int col2) {
				RenderSystem.disableTexture();
				RenderSystem.enableBlend();
				RenderSystem.disableAlphaTest();
				RenderSystem.defaultBlendFunc();
				RenderSystem.shadeModel(7425);
				var tessellator = class_289.method_1348();
				var vertexConsumer = tessellator.method_1349();
				float z = 0;
				//top
				var pose = graphics.method_23760().method_23761();
				vertexConsumer.method_1328(7, class_290.field_1576);
				vertexConsumer.method_22918(pose, x0, y0, z).method_1336(col1 >> 16 & 255, col1 >> 8 & 255, col1 & 255, col1 >> 24 & 255).method_1344();
				vertexConsumer.method_22918(pose, x0 + gradientWidth, y0 + gradientWidth, z).method_1336(col2 >> 16 & 255, col2 >> 8 & 255, col2 & 255, col2 >> 24 & 255).method_1344();
				vertexConsumer.method_22918(pose, x1 - gradientWidth, y0 + gradientWidth, z).method_1336(col2 >> 16 & 255, col2 >> 8 & 255, col2 & 255, col2 >> 24 & 255).method_1344();
				vertexConsumer.method_22918(pose, x1, y0, z).method_1336(col1 >> 16 & 255, col1 >> 8 & 255, col1 & 255, col1 >> 24 & 255).method_1344();
				//left
				vertexConsumer.method_22918(pose, x0, y1, z).method_1336(col1 >> 16 & 255, col1 >> 8 & 255, col1 & 255, col1 >> 24 & 255).method_1344();
				vertexConsumer.method_22918(pose, x0 + gradientWidth, y1 - gradientWidth, z).method_1336(col2 >> 16 & 255, col2 >> 8 & 255, col2 & 255, col2 >> 24 & 255).method_1344();
				vertexConsumer.method_22918(pose, x0 + gradientWidth, y0 + gradientWidth, z).method_1336(col2 >> 16 & 255, col2 >> 8 & 255, col2 & 255, col2 >> 24 & 255).method_1344();
				vertexConsumer.method_22918(pose, x0, y0, z).method_1336(col1 >> 16 & 255, col1 >> 8 & 255, col1 & 255, col1 >> 24 & 255).method_1344();
				//bottom
				vertexConsumer.method_22918(pose, x1, y1, z).method_1336(col1 >> 16 & 255, col1 >> 8 & 255, col1 & 255, col1 >> 24 & 255).method_1344();
				vertexConsumer.method_22918(pose, x1 - gradientWidth, y1 - gradientWidth, z).method_1336(col2 >> 16 & 255, col2 >> 8 & 255, col2 & 255, col2 >> 24 & 255).method_1344();
				vertexConsumer.method_22918(pose, x0 + gradientWidth, y1 - gradientWidth, z).method_1336(col2 >> 16 & 255, col2 >> 8 & 255, col2 & 255, col2 >> 24 & 255).method_1344();
				vertexConsumer.method_22918(pose, x0, y1, z).method_1336(col1 >> 16 & 255, col1 >> 8 & 255, col1 & 255, col1 >> 24 & 255).method_1344();
				//right
				vertexConsumer.method_22918(pose, x1, y0, z).method_1336(col1 >> 16 & 255, col1 >> 8 & 255, col1 & 255, col1 >> 24 & 255).method_1344();
				vertexConsumer.method_22918(pose, x1 - gradientWidth, y0 + gradientWidth, z).method_1336(col2 >> 16 & 255, col2 >> 8 & 255, col2 & 255, col2 >> 24 & 255).method_1344();
				vertexConsumer.method_22918(pose, x1 - gradientWidth, y1 - gradientWidth, z).method_1336(col2 >> 16 & 255, col2 >> 8 & 255, col2 & 255, col2 >> 24 & 255).method_1344();
				vertexConsumer.method_22918(pose, x1, y1, z).method_1336(col1 >> 16 & 255, col1 >> 8 & 255, col1 & 255, col1 >> 24 & 255).method_1344();
				tessellator.method_1350();
				RenderSystem.shadeModel(7424);
				RenderSystem.disableBlend();
				RenderSystem.enableAlphaTest();
				RenderSystem.enableTexture();
			}
		}
	}

	private class SpriteButton extends class_4185 {
		private class_2960 sprite;

		public SpriteButton(class_2561 message, class_4241 onPress, class_2960 sprite) {
			super(0, 0, 11, 11, message, onPress);
			this.sprite = sprite;
		}

		@Override
		public void method_25359(class_4587 graphics, int mouseX, int mouseY, float delta) {
			class_2960 tex = ButtonWidgetTextures.get(method_25356(field_22762));
			DrawUtil.blitSprite(tex, field_22760, field_22761, field_22758, field_22759, new DrawUtil.NineSlice(200, 20, 3));
			field_22787.method_1531().method_22813(sprite);
			method_25290(graphics, field_22760 + 2, field_22761 + 2, 0, 0, 7, 7, 7, 7);
			if (this.method_25367()) {
				tooltip = method_25369();
			}
		}
	}
}
