/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.hud.gui.hud;

import java.nio.file.Files;
import java.util.*;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientCommon;
import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Color;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.ColorOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.IntegerOption;
import io.github.axolotlclient.bridge.render.AxoRenderContext;
import io.github.axolotlclient.config.profiles.ProfileAware;
import io.github.axolotlclient.mixin.KeyBindAccessor;
import io.github.axolotlclient.modules.hud.ClickInputTracker;
import io.github.axolotlclient.modules.hud.gui.entry.TextHudEntry;
import io.github.axolotlclient.modules.hud.gui.keystrokes.KeystrokePositioningScreen;
import io.github.axolotlclient.modules.hud.gui.keystrokes.KeystrokesScreen;
import io.github.axolotlclient.modules.hud.gui.layout.Justification;
import io.github.axolotlclient.modules.hud.util.DrawPosition;
import io.github.axolotlclient.modules.hud.util.Rectangle;
import io.github.axolotlclient.util.ClientColors;
import io.github.axolotlclient.util.GsonHelper;
import io.github.axolotlclient.util.events.Events;
import io.github.axolotlclient.util.options.GenericOption;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.Setter;
import net.minecraft.class_1074;
import net.minecraft.class_124;
import net.minecraft.class_156;
import net.minecraft.class_2960;
import net.minecraft.class_304;
import net.minecraft.class_310;
import net.minecraft.class_3532;
import net.minecraft.class_3675;
import net.minecraft.class_4587;
import org.lwjgl.glfw.GLFW;

import static io.github.axolotlclient.modules.hud.util.DrawUtil.*;

/**
 * This implementation of Hud modules is based on KronHUD.
 * <a href="https://github.com/DarkKronicle/KronHUD">Github Link.</a>
 *
 * <p>License: GPL-3.0</p>
 */

public class KeystrokeHud extends TextHudEntry implements ProfileAware {

	private static final String KEYSTROKE_SAVE_FILE_NAME = "keystrokes.json";
	public static final class_2960 ID = new class_2960("kronhud", "keystrokehud");

	private final class_310 client = (class_310) super.client;

	private final ColorOption pressedTextColor = new ColorOption("heldtextcolor", new Color(0xFF000000));
	private final ColorOption pressedBackgroundColor = new ColorOption("heldbackgroundcolor", new Color(0x64FFFFFF));
	private final ColorOption pressedOutlineColor = new ColorOption("heldoutlinecolor", ClientColors.BLACK);

	private final GenericOption keystrokesOption = new GenericOption("keystrokes", "keystrokes.configure", () -> client.method_1507(new KeystrokesScreen(KeystrokeHud.this, client.field_1755)));
	private final GenericOption configurePositions = new GenericOption("keystrokes.positions", "keystrokes.positions.configure",
		() -> client.method_1507(new KeystrokePositioningScreen(client.field_1755, this)));
	private final IntegerOption animationTime = new IntegerOption("keystrokes.animation_time", 100, 0, 500);
	public ArrayList<Keystroke> keystrokes;

	public KeystrokeHud() {
		super(53, 61, true);
		Events.KEYBIND_CHANGE.register(key -> {
			if (class_310.method_1551().method_22683() != null) {
				class_304.method_1437();
				class_304.method_1424();
			}
		});
	}

	public static Optional<String> getMouseKeyBindName(class_304 keyBinding) {
		if (keyBinding.method_1428().equalsIgnoreCase(
			class_3675.class_307.field_1672.method_1447(GLFW.GLFW_MOUSE_BUTTON_1).method_1441())) {
			return Optional.of("LMB");
		} else if (keyBinding.method_1428().equalsIgnoreCase(
			class_3675.class_307.field_1672.method_1447(GLFW.GLFW_MOUSE_BUTTON_2).method_1441())) {
			return Optional.of("RMB");
		} else if (keyBinding.method_1428().equalsIgnoreCase(
			class_3675.class_307.field_1672.method_1447(GLFW.GLFW_MOUSE_BUTTON_3).method_1441())) {
			return Optional.of("MMB");
		}
		return Optional.empty();
	}

	public void setDefaultKeystrokes() {
		DrawPosition pos = getPos();
		// LMB
		keystrokes.add(createFromKey(new Rectangle(0, 36, 26, 17), pos, client.field_1690.field_1886));
		// RMB
		keystrokes.add(createFromKey(new Rectangle(27, 36, 26, 17), pos, client.field_1690.field_1904));
		// W
		keystrokes.add(createFromKey(new Rectangle(18, 0, 17, 17), pos, client.field_1690.field_1894));
		// A
		keystrokes.add(createFromKey(new Rectangle(0, 18, 17, 17), pos, client.field_1690.field_1913));
		// S
		keystrokes.add(createFromKey(new Rectangle(18, 18, 17, 17), pos, client.field_1690.field_1881));
		// D
		keystrokes.add(createFromKey(new Rectangle(36, 18, 17, 17), pos, client.field_1690.field_1849));

		// Space
		keystrokes.add(new CustomRenderKeystroke(SpecialKeystroke.SPACE));
	}

	public void setKeystrokes() {
		if (client.method_22683() == null) {
			keystrokes = null;
			return;
			// Wait until render is called
		}
		keystrokes = new ArrayList<>();
		setDefaultKeystrokes();
		loadKeystrokes();
		class_304.method_1437();
		class_304.method_1424();
	}

	public Keystroke createFromKey(Rectangle bounds, DrawPosition offset, class_304 key) {
		String name = getMouseKeyBindName(key).orElse(key.method_16007().getString().toUpperCase());
		if (name.length() > 4) {
			name = name.substring(0, 2);
		}
		return createFromString(bounds, offset, key, name);
	}

	public Keystroke createFromString(Rectangle bounds, DrawPosition offset, class_304 key, String word) {
		return new LabelKeystroke(bounds, offset, key, word);
	}

	@Override
	public void render(AxoRenderContext graphics, float delta) {
		graphics.br$pushMatrix();
		scale(graphics);
		renderComponent(graphics, delta);
		graphics.br$popMatrix();
	}

	@Override
	public void renderComponent(AxoRenderContext graphics, float delta) {
		if (keystrokes == null) {
			setKeystrokes();
		}
		for (Keystroke stroke : keystrokes) {
			stroke.render((class_4587) graphics);
		}
	}

	@Override
	public void renderPlaceholderComponent(AxoRenderContext graphics, float delta) {
		renderComponent(graphics, delta);
	}

	@Override
	public boolean tickable() {
		return true;
	}

	@Override
	public void tick() {
		DrawPosition pos = getPos();
		if (keystrokes == null) {
			setKeystrokes();
		}
		for (Keystroke stroke : keystrokes) {
			stroke.offset = pos;
		}
	}

	@Override
	protected boolean getShadowDefault() {
		return false;
	}

	@Override
	public List<Option<?>> getConfigurationOptions() {
		// We want a specific order since this is a more complicated entry
		List<Option<?>> options = new ArrayList<>();
		options.add(enabled);
		options.add(scale);
		options.add(textColor);
		options.add(pressedTextColor);
		options.add(shadow);
		options.add(background);
		options.add(backgroundColor);
		options.add(pressedBackgroundColor);
		options.add(outline);
		options.add(outlineColor);
		options.add(pressedOutlineColor);
		options.add(animationTime);
		options.add(keystrokesOption);
		options.add(configurePositions);
		return options;
	}

	@Override
	public class_2960 getId() {
		return ID;
	}

	@Override
	public void reloadConfig() {
		keystrokes = null;
	}

	@Override
	public void saveConfig() {
		saveKeystrokes();
	}

	public interface KeystrokeRenderer {

		void render(Keystroke stroke, class_4587 graphics);
	}

	public abstract class Keystroke {

		@Getter
		@Setter
		protected class_304 key;
		protected KeystrokeRenderer render;
		@Getter
		protected final Rectangle bounds;
		protected DrawPosition offset;
		private long start = -1;
		private boolean wasPressed = false;

		public Keystroke(Rectangle bounds, DrawPosition offset, class_304 key, KeystrokeRenderer render) {
			this.bounds = bounds;
			this.offset = offset;
			this.key = key;
			this.render = render;
		}

		public void setX(int x) {
			bounds.x(x - offset.x());
		}

		public void setY(int y) {
			bounds.y(y - offset.y());
		}

		public Rectangle getRenderPosition() {
			return bounds.offset(offset);
		}

		public Color getFGColor() {
			return isKeyDown() ? ClientColors.blend(textColor.get(), pressedTextColor.get(), getPercentPressed())
				: ClientColors.blend(pressedTextColor.get(), textColor.get(), getPercentPressed());
		}

		private float getPercentPressed() {
			return start == -1 ? 1 : class_3532.method_15363((float) (class_156.method_658() - start) / getAnimTime(), 0, 1);
		}

		public void render(class_4587 matrices) {
			renderStroke(matrices);
			render.render(this, matrices);
		}

		public void renderStroke(class_4587 matrices) {
			if (isKeyDown() != wasPressed) {
				start = class_156.method_658();
			}
			Rectangle rect = getRenderPosition();
			if (background.get()) {
				fillRect(matrices, rect, getColor());
			}
			if (outline.get()) {
				outlineRect(matrices, rect, getOutlineColor());
			}
			if ((float) (class_156.method_658() - start) / getAnimTime() >= 1) {
				start = -1;
			}
			wasPressed = isKeyDown();
		}

		private int getAnimTime() {
			return animationTime.get();
		}

		private boolean isKeyDown() {
			return key != null && key.method_1434();
		}

		public Color getColor() {
			return isKeyDown()
				? ClientColors.blend(backgroundColor.get(), pressedBackgroundColor.get(), getPercentPressed())
				: ClientColors.blend(pressedBackgroundColor.get(), backgroundColor.get(), getPercentPressed());
		}

		public Color getOutlineColor() {
			return isKeyDown() ? ClientColors.blend(outlineColor.get(), pressedOutlineColor.get(), getPercentPressed())
				: ClientColors.blend(pressedOutlineColor.get(), outlineColor.get(), getPercentPressed());
		}

		public Map<String, Object> serialize() {
			Map<String, Object> map = new HashMap<>();
			map.put("key", key.method_1428());
			map.put("bounds", Map.of("x", bounds.x(), "y", bounds.y(), "width", bounds.width(), "height", bounds.height()));
			return map;
		}

		public abstract String getLabel();

		public abstract void setLabel(String label);

		public abstract boolean isLabelEditable();
	}

	@SuppressWarnings("unchecked")
	private Keystroke deserializeKey(Map<String, Object> json) {
		if ("option".equals(json.get("type"))) {
			class_304 key = KeyBindAccessor.getAllKeyBinds().get((String) json.getOrDefault("key_name", json.get("option")));
			return new CustomRenderKeystroke(SpecialKeystroke.byId.get(((String) json.get("special_name")).toLowerCase(Locale.ROOT)),
				getRectangle((Map<String, ?>) json.get("bounds")), getPos(), key);
		} else {
			var key = KeyBindAccessor.getAllKeyBinds().get((String) json.get("key_name"));
			return new LabelKeystroke(getRectangle((Map<String, ?>) json.get("bounds")), getPos(), key, (String) json.get("label"), (boolean) json.get("synchronize_label"),
				Justification.valueOf((String) json.getOrDefault("justification", "CENTER")));
		}
	}

	private static Rectangle getRectangle(Map<String, ?> json) {
		return new Rectangle((int) (long) json.get("x"), (int) (long) json.get("y"), (int) (long) json.get("width"), (int) (long) json.get("height"));
	}

	public class CustomRenderKeystroke extends Keystroke {

		private static final Supplier<String> label = () -> class_124.field_1056 + class_1074.method_4662("keystrokes.stroke.custom_renderer");

		private final SpecialKeystroke parent;

		public CustomRenderKeystroke(SpecialKeystroke stroke, Rectangle bounds, DrawPosition offset, class_304 key) {
			super(bounds, offset, key, (s, g) -> stroke.getRenderer().render(KeystrokeHud.this, s, g));
			this.parent = stroke;
		}

		public CustomRenderKeystroke(SpecialKeystroke stroke) {
			this(stroke, stroke.getRect().copy(), KeystrokeHud.this.getPos(), stroke.getKey());
		}

		@Override
		public Map<String, Object> serialize() {
			Map<String, Object> json = super.serialize();
			json.put("type", "option");
			json.put("key_name", key.method_1431());
			json.put("special_name", parent.getId());
			return json;
		}

		@Override
		public String getLabel() {
			return label.get();
		}

		@Override
		public void setLabel(String label) {

		}

		@Override
		public boolean isLabelEditable() {
			return false;
		}
	}

	public Keystroke newSpecialStroke(SpecialKeystroke stroke) {
		return new CustomRenderKeystroke(stroke);
	}

	public LabelKeystroke newStroke() {
		return new LabelKeystroke(new Rectangle(0, 0, 17, 17), getPos(), null, "", false, Justification.CENTER);
	}

	@Setter
	public class LabelKeystroke extends Keystroke {

		private String label;
		@Getter
		private boolean synchronizeLabel;
		@Getter
		private Justification justification;

		public LabelKeystroke(Rectangle bounds, DrawPosition offset, class_304 key, String label) {
			this(bounds, offset, key, label, true, Justification.CENTER);
		}

		public LabelKeystroke(Rectangle bounds, DrawPosition offset, class_304 key, String label, boolean synchronizeLabel, Justification justification) {
			super(bounds, offset, key, (stroke, matrices) -> {
			});
			this.label = label;
			this.render = (stroke, matrices) -> {
				Rectangle strokeBounds = stroke.bounds;
				int x = strokeBounds.x() + stroke.offset.x() + 2 + this.justification.getXOffset(getLabel(), strokeBounds.width() - 3);
				float y = strokeBounds.y() + stroke.offset.y() + ((float) strokeBounds.height() / 2) - 4;

				drawString(matrices, getLabel(), x, (int) y, stroke.getFGColor().toInt(), shadow.get());
			};
			setSynchronizeLabel(synchronizeLabel);
			this.justification = justification;
		}

		@Override
		public Map<String, Object> serialize() {
			Map<String, Object> json = super.serialize();
			json.put("type", "custom");
			json.put("key_name", key.method_1431());
			json.put("label", label);
			json.put("synchronize_label", synchronizeLabel);
			json.put("justification", justification.name());
			return json;
		}

		public void setSynchronizeLabel(boolean synchronizeLabel) {
			if (synchronizeLabel) {
				String name = getMouseKeyBindName(key).orElse(key.method_16007().getString().toUpperCase());
				if (name.length() > 4) {
					name = name.substring(0, 2);
				}
				this.label = name;
			}
			this.synchronizeLabel = synchronizeLabel;
		}

		@Override
		public void setKey(class_304 key) {
			if (synchronizeLabel) {
				String name = getMouseKeyBindName(key).orElse(key.method_16007().getString().toUpperCase());
				if (name.length() > 4) {
					name = name.substring(0, 2);
				}
				this.label = name;
			}
			super.setKey(key);
		}

		@Override
		public String getLabel() {
			return label;
		}

		@Override
		public boolean isLabelEditable() {
			return true;
		}
	}

	public void saveKeystrokes() {
		if (keystrokes == null) return;
		try {
			var path = AxolotlClientCommon.resolveProfileConfigFile(KEYSTROKE_SAVE_FILE_NAME);
			Files.createDirectories(path.getParent());
			Files.writeString(path, GsonHelper.GSON.toJson(keystrokes.stream().map(Keystroke::serialize).toList()));
		} catch (Exception e) {
			AxolotlClient.LOGGER.warn("Failed to save keystroke configuration!", e);
		}
	}

	@SuppressWarnings("unchecked")
	public void loadKeystrokes() {
		try {
			var path = AxolotlClientCommon.resolveProfileConfigFile(KEYSTROKE_SAVE_FILE_NAME);
			if (Files.exists(path)) {
				List<?> entries = (List<?>) GsonHelper.read(Files.readString(path));
				var loaded = entries.stream().map(e -> (Map<String, Object>) e)
					.map(KeystrokeHud.this::deserializeKey)
					.toList();
				if (keystrokes == null) {
					keystrokes = new ArrayList<>();
				} else {
					keystrokes.clear();
				}
				keystrokes.addAll(loaded);
			} else {
				saveKeystrokes();
			}
		} catch (Exception e) {
			AxolotlClient.LOGGER.warn("Failed to load keystroke configuration, using defaults!", e);
		}
	}

	@AllArgsConstructor
	@Getter
	public enum SpecialKeystroke {
		SPACE("space", new Rectangle(0, 54, 53, 7), class_310.method_1551().field_1690.field_1903, (hud, stroke, matrices) -> {
			Rectangle bounds = stroke.bounds;
			Rectangle spaceBounds = new Rectangle(bounds.x() + stroke.offset.x() + 4,
				bounds.y() + stroke.offset.y() + bounds.height() / 2 - 1, bounds.width() - 8, 1);
			fillRect(matrices, spaceBounds, stroke.getFGColor());
			if (hud.shadow.get()) {
				fillRect(matrices, spaceBounds.offset(1, 1), new Color(
					(stroke.getFGColor().toInt() & 16579836) >> 2 | stroke.getFGColor().toInt() & -16777216));
			}
		}),
		LMB_CPS("lmb_cps", new Rectangle(0, 36, 26, 17), class_310.method_1551().field_1690.field_1886, (hud, stroke, graphics) -> {
			Rectangle bounds = stroke.bounds;
			int centerX = bounds.x() + stroke.offset.x() + bounds.width() / 2;
			int y = bounds.y() + stroke.offset.y() + 3;
			int nameY = y + bounds.height() / 4 - hud.client.textRenderer.fontHeight / 2;
			drawCenteredString(graphics, hud.client.textRenderer, "LMB", centerX, nameY, stroke.getFGColor(), hud.shadow.get());
			int cpsY = y + bounds.height() * 3 / 4 - hud.client.textRenderer.fontHeight / 2;
			graphics.push();
			graphics.translate(centerX, cpsY, 0);
			graphics.scale(0.5f, 0.5f, 1);
			// TODO have fromKeybinds here configured seperately
			String cpsText = ClickInputTracker.getInstance().leftMouse.clicks() + " CPS";
			graphics.translate(-hud.client.textRenderer.getWidth(cpsText) / 2f, 0, 0);
			drawString(graphics, cpsText, 0, 0, stroke.getFGColor(), hud.shadow.get());
			graphics.pop();
		}),
		RMB_CPS("rmb_cps", new Rectangle(27, 36, 26, 17), class_310.method_1551().field_1690.field_1904, (hud, stroke, graphics) -> {
			Rectangle bounds = stroke.bounds;
			int centerX = bounds.x() + stroke.offset.x() + bounds.width() / 2;
			int y = bounds.y() + stroke.offset.y() + 3;
			int nameY = y + bounds.height() / 4 - hud.client.textRenderer.fontHeight / 2;
			drawCenteredString(graphics, hud.client.textRenderer, "RMB", centerX, nameY, stroke.getFGColor(), hud.shadow.get());
			int cpsY = y + bounds.height() * 3 / 4 - hud.client.textRenderer.fontHeight / 2;
			graphics.push();
			graphics.translate(centerX, cpsY, 0);
			graphics.scale(0.5f, 0.5f, 1);
			String cpsText = ClickInputTracker.getInstance().rightMouse.clicks() + " CPS";
			graphics.translate(-hud.client.textRenderer.getWidth(cpsText) / 2f, 0, 0);
			drawString(graphics, cpsText, 0, 0, stroke.getFGColor(), hud.shadow.get());
			graphics.pop();
		});
		private static final Map<String, SpecialKeystroke> byId = Arrays.stream(values()).collect(Collectors.toMap(SpecialKeystroke::getId, Function.identity()));

		private final String id;
		private final Rectangle rect;
		private final class_304 key;
		private final SpecialKeystrokeRenderer renderer;

		public interface SpecialKeystrokeRenderer {
			void render(KeystrokeHud hud, Keystroke stroke, class_4587 graphics);
		}
	}
}
