/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.hud.gui.hud.vanilla;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.mojang.datafixers.util.Pair;
import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Color;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Colors;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.ColorOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.EnumOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.IntegerOption;
import io.github.axolotlclient.bridge.render.AxoRenderContext;
import io.github.axolotlclient.modules.hud.gui.component.DynamicallyPositionable;
import io.github.axolotlclient.modules.hud.gui.entry.TextHudEntry;
import io.github.axolotlclient.modules.hud.gui.layout.AnchorPoint;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import io.github.axolotlclient.modules.hud.util.Rectangle;
import io.github.axolotlclient.modules.hud.util.RenderUtil;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_2585;
import net.minecraft.class_266;
import net.minecraft.class_267;
import net.minecraft.class_268;
import net.minecraft.class_269;
import net.minecraft.class_274;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_4587;
import net.minecraft.class_5250;
import net.minecraft.scoreboard.*;

/**
 * This implementation of Hud modules is based on KronHUD.
 * <a href="https://github.com/DarkKronicle/KronHUD">Github Link.</a>
 *
 * <p>License: GPL-3.0</p>
 */

public class ScoreboardHud extends TextHudEntry implements DynamicallyPositionable {

	public static final class_2960 ID = new class_2960("kronhud", "scoreboardhud");
	public static final class_266 placeholder = class_156.method_656(() -> {
		class_269 placeScore = new class_269();
		class_266 objective = placeScore.method_1168("placeholder", class_274.field_1468,
			new class_2585("Scoreboard"), class_274.class_275.field_1472);
		class_267 dark = placeScore.method_1180("DarkKronicle", objective);
		dark.method_1128(8780);

		class_267 moeh = placeScore.method_1180("moehreag", objective);
		moeh.method_1128(743);

		class_267 kode = placeScore.method_1180("TheKodeToad", objective);
		kode.method_1128(2948);

		placeScore.method_1158(1, objective);
		return objective;
	});

	private final ColorOption backgroundColor = new ColorOption("backgroundcolor", new Color(0x4C000000));
	private final ColorOption topColor = new ColorOption("topbackgroundcolor", new Color(0x66000000));
	private final IntegerOption topPadding = new IntegerOption("toppadding", 0, 0, 4);
	private final BooleanOption scores = new BooleanOption("scores", true);
	private final ColorOption scoreColor = new ColorOption("scorecolor", new Color(0xFFFF5555));
	private final IntegerOption textAlpha = new IntegerOption("text_alpha", 255, 0, 255);
	private final EnumOption<AnchorPoint> anchor = new EnumOption<>("anchorpoint", AnchorPoint.class,
		AnchorPoint.MIDDLE_RIGHT);

	private final class_310 client = (class_310) super.client;

	public ScoreboardHud() {
		super(200, 146, true);
	}

	@Override
	public void render(AxoRenderContext matrices, float delta) {
		matrices.br$pushMatrix();
		scale(matrices);
		renderComponent(matrices, delta);
		matrices.br$popMatrix();
	}

	@Override
	public void renderComponent(AxoRenderContext matrices, float delta) {
		//noinspection DataFlowIssue
		class_269 scoreboard = this.client.field_1687.method_8428();
		class_266 scoreboardObjective = null;
		//noinspection DataFlowIssue
		class_268 team = scoreboard.method_1164(this.client.field_1724.method_5820());
		if (team != null) {
			int t = team.method_1202().method_536();
			if (t >= 0) {
				scoreboardObjective = scoreboard.method_1189(3 + t);
			}
		}

		class_266 scoreboardObjective2 = scoreboardObjective != null ? scoreboardObjective
			: scoreboard.method_1189(1);
		if (scoreboardObjective2 != null) {
			this.renderScoreboardSidebar((class_4587) matrices, scoreboardObjective2);
		}
	}

	@Override
	public void renderPlaceholderComponent(AxoRenderContext matrices, float delta) {
		renderScoreboardSidebar((class_4587) matrices, placeholder);
	}

	// Abusing this could break some stuff/could allow for unfair advantages. The goal is not to do this, so it won't
	// show any more information than it would have in vanilla.
	private void renderScoreboardSidebar(class_4587 matrices, class_266 objective) {
		class_269 scoreboard = objective.method_1117();
		Collection<class_267> scores = scoreboard.method_1184(objective);
		List<class_267> filteredScores = scores.stream()
			.filter((testScore) -> testScore.method_1129() != null && !testScore.method_1129().startsWith("#"))
			.collect(Collectors.toList());

		if (filteredScores.size() > 15) {
			scores = Lists.newArrayList(Iterables.skip(filteredScores, scores.size() - 15));
		} else {
			scores = filteredScores;
		}

		List<Pair<class_267, class_2561>> scoresWText = Lists.newArrayListWithCapacity(scores.size());
		class_2561 text = objective.method_1114();
		int displayNameWidth = client.field_1772.method_27525(text);
		int maxWidth = displayNameWidth;
		int spacerWidth = client.field_1772.method_1727(": ");

		class_267 scoreboardPlayerScore;
		class_5250 formattedText;
		for (Iterator<class_267> scoresIterator = scores.iterator(); scoresIterator
			.hasNext(); maxWidth = Math.max(maxWidth, client.field_1772.method_27525(formattedText) + spacerWidth
			+ client.field_1772.method_1727(Integer.toString(scoreboardPlayerScore.method_1126())))) {
			scoreboardPlayerScore = scoresIterator.next();
			class_268 team = scoreboard.method_1164(scoreboardPlayerScore.method_1129());
			formattedText = class_268.method_1142(team, new class_2585(scoreboardPlayerScore.method_1129()));
			scoresWText.add(Pair.of(scoreboardPlayerScore, formattedText));
		}
		maxWidth = maxWidth + 6;

		int scoresSize = scores.size();
		int scoreHeight = scoresSize * 9;
		int fullHeight = scoreHeight + 11 + topPadding.get() * 2;

		boolean updated = false;
		if (fullHeight + 1 != height) {
			setHeight(fullHeight + 1);
			updated = true;
		}
		if (maxWidth + 1 != width) {
			setWidth(maxWidth + 1);
			updated = true;
		}
		if (updated) {
			onBoundsUpdate();
		}

		Rectangle bounds = getBounds();

		int renderX = bounds.x() + bounds.width() - maxWidth;
		int renderY = bounds.y() + (bounds.height() / 2 - fullHeight / 2) + 1;

		int scoreX = renderX + 4;
		int scoreY = renderY + scoreHeight + 10;
		int num = 0;
		int textOffset = scoreX - 4;

		for (Pair<class_267, class_2561> scoreboardPlayerScoreTextPair : scoresWText) {
			++num;
			class_267 scoreboardPlayerScore2 = scoreboardPlayerScoreTextPair.getFirst();
			class_2561 scoreText = scoreboardPlayerScoreTextPair.getSecond();
			String score = String.valueOf(scoreboardPlayerScore2.method_1126());
			int relativeY = scoreY - num * 9 + topPadding.get() * 2;

			if (background.get() && backgroundColor.get().toInt() > 0) {
				if (num == scoresSize) {
					RenderUtil.drawRectangle(matrices, textOffset, relativeY - 1, maxWidth, 10,
						backgroundColor.get().toInt());
				} else if (num == 1) {
					RenderUtil.drawRectangle(matrices, textOffset, relativeY, maxWidth, 10,
						backgroundColor.get().toInt());
				} else {
					RenderUtil.drawRectangle(matrices, textOffset, relativeY, maxWidth, 9,
						backgroundColor.get().toInt());
				}
			}

			if (shadow.get()) {
				client.field_1772.method_30881(matrices, scoreText, (float) scoreX, (float) relativeY, Colors.WHITE.withAlpha(textAlpha.get()).toInt());
			} else {
				client.field_1772.method_30883(matrices, scoreText, (float) scoreX, (float) relativeY, Colors.WHITE.withAlpha(textAlpha.get()).toInt());
			}
			if (this.scores.get()) {
				DrawUtil.drawString(matrices, score, (float) (scoreX + maxWidth - client.field_1772.method_1727(score) - 6),
					(float) relativeY, scoreColor.get().toInt(), shadow.get());
			}
			if (num == scoresSize) {
				// Draw the title
				if (background.get()) {
					RenderUtil.drawRectangle(matrices, textOffset, relativeY - 10 - topPadding.get() * 2 - 1, maxWidth,
						10 + topPadding.get() * 2, topColor.get());
				}
				float title = (float) (renderX + (maxWidth - displayNameWidth) / 2);
				if (shadow.get()) {
					client.field_1772.method_30881(matrices, text, title,
						(float) (relativeY - 9) - topPadding.get(), Colors.WHITE.withAlpha(textAlpha.get()).toInt());
				} else {
					client.field_1772.method_30883(matrices, text, title, (float) (relativeY - 9), Colors.WHITE.withAlpha(textAlpha.get()).toInt());
				}
			}
		}

		if (outline.get() && outlineColor.get().getAlpha() > 0) {
			RenderUtil.drawOutline(matrices, textOffset - 1, bounds.y(), maxWidth + 1, fullHeight + 1, outlineColor.get());
		}
	}

	@Override
	public List<Option<?>> getConfigurationOptions() {
		List<Option<?>> options = super.getConfigurationOptions();
		options.set(options.indexOf(super.backgroundColor), backgroundColor);
		options.add(hide);
		options.add(topColor);
		options.add(scores);
		options.add(scoreColor);
		options.add(anchor);
		options.add(topPadding);
		options.remove(textColor);
		options.add(textAlpha);
		return options;
	}

	@Override
	public class_2960 getId() {
		return ID;
	}

	@Override
	public AnchorPoint getAnchor() {
		return anchor.get();
	}

	@Override
	public double getDefaultX() {
		return 1.0;
	}

	@Override
	public double getDefaultY() {
		return 0.5;
	}
}
