/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.particles;

import java.util.*;
import java.util.stream.Collectors;

import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientConfig.api.options.Option;
import io.github.axolotlclient.AxolotlClientConfig.api.options.OptionCategory;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Color;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.ColorOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.IntegerOption;
import io.github.axolotlclient.mixin.ParticleAccessor;
import io.github.axolotlclient.modules.AbstractModule;
import io.github.axolotlclient.util.ClientColors;
import lombok.Getter;
import net.minecraft.class_2378;
import net.minecraft.class_2396;
import net.minecraft.class_2398;
import net.minecraft.class_703;
import org.apache.commons.lang3.StringUtils;

public class Particles extends AbstractModule {

	@Getter
	private static final Particles Instance = new Particles();

	public final HashMap<class_2396<?>, HashMap<String, Option<?>>> particleOptions = new HashMap<>();
	public final WeakHashMap<class_703, class_2396<?>> particleMap = new WeakHashMap<>();

	private final OptionCategory cat = OptionCategory.create("particles");
	private final BooleanOption enabled = new BooleanOption("enabled", false);

	@Override
	public void init() {
		cat.add(enabled);

		addParticleOptions();
		AxolotlClient.config().rendering.add(cat);
	}

	private void addParticleOptions() {
		for (class_2396<?> type : class_2378.field_11141.method_10220().sorted(new AlphabeticalComparator()).toList()) {
			if (class_2378.field_11141.method_10221(type) != null) {
				//noinspection DataFlowIssue
				OptionCategory category = OptionCategory.create(
					Arrays.stream(class_2378.field_11141.method_10221(type).method_12832().split("_"))
						.map(StringUtils::capitalize).collect(Collectors.joining(" ")));
				HashMap<String, Option<?>> optionsByKey = new LinkedHashMap<>();

				populateMap(optionsByKey, new BooleanOption("showParticle", true),
					new IntegerOption("count", 1, 1, 20),
					new BooleanOption("customColor", false),
					new ColorOption("color", "particles.color.tooltip", ClientColors.WHITE));

				if (type == class_2398.field_11205 || type == class_2398.field_11208) {
					populateMap(optionsByKey, new BooleanOption("alwaysCrit", false));
				}

				optionsByKey.values().forEach(category::add);
				particleOptions.put(type, optionsByKey);

				cat.add(category);
			}
		}
	}

	private void populateMap(HashMap<String, Option<?>> map, Option<?>... options) {
		for (Option<?> option : options) {
			map.put(option.getName(), option);
		}
	}

	public void applyOptions(class_703 particle) {
		if (enabled.get() && particleMap.containsKey(particle)) {
			class_2396<?> type = particleMap.get(particle);
			if (particleOptions.containsKey(type)) {
				HashMap<String, Option<?>> options = particleOptions.get(type);

				if (((BooleanOption) options.get("customColor")).get()) {
					Color color = ((ColorOption) options.get("color")).get();
					particle.method_3084(color.getRed() / 255F, color.getGreen() / 255F, color.getBlue() / 255F);
					((ParticleAccessor) particle).setColorAlpha(color.getAlpha() / 255F);
				}
			}
		}
	}

	public int getMultiplier(class_2396<?> type) {
		if (enabled.get() && particleOptions.containsKey(type)) {
			HashMap<String, Option<?>> options = particleOptions.get(type);

			return ((IntegerOption) options.get("count")).get();
		}
		return 1;
	}

	public boolean getAlwaysOn(class_2396<?> type) {
		return enabled.get() && particleOptions.containsKey(type)
			&& ((BooleanOption) Particles.getInstance().particleOptions.get(type).get("alwaysCrit")).get();
	}

	public boolean getShowParticle(class_2396<?> type) {
		return enabled.get() && particleOptions.containsKey(type)
			? ((BooleanOption) Particles.getInstance().particleOptions.get(type).get("showParticle")).get()
			: true;
	}

	protected static class AlphabeticalComparator implements Comparator<class_2396<?>> {

		// Function to compare
		public int compare(class_2396<?> s1, class_2396<?> s2) {
			if (getName(s1).equals(getName(s2)))
				return 0;
			String[] strings = {getName(s1), getName(s2)};
			Arrays.sort(strings, Collections.reverseOrder());

			if (strings[0].equals(getName(s1)))
				return 1;
			else
				return -1;
		}

		private String getName(class_2396<?> type) {
			if (class_2378.field_11141.method_10221(type) != null) {
				//noinspection DataFlowIssue
				return class_2378.field_11141.method_10221(type).method_12832();
			}
			return "";
		}
	}
}
