/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.screenshotUtils;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

import io.github.axolotlclient.AxolotlClientConfig.api.util.Colors;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.api.requests.FriendRequest;
import io.github.axolotlclient.api.requests.UserRequest;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import io.github.axolotlclient.util.Watcher;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1159;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_2585;
import net.minecraft.class_2588;
import net.minecraft.class_287;
import net.minecraft.class_289;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_3532;
import net.minecraft.class_364;
import net.minecraft.class_4185;
import net.minecraft.class_4265;
import net.minecraft.class_437;
import net.minecraft.class_4587;
import net.minecraft.class_5244;
import net.minecraft.class_5250;
import org.jetbrains.annotations.NotNull;

public class GalleryScreen extends class_437 {

	public static final Path SCREENSHOTS_DIR = FabricLoader.getInstance().getGameDir().resolve("screenshots");

	private record Tab<T>(class_2561 title, Callable<List<T>> list, Map<T, ImageInstance> loadingCache,
						  Loader<T> loader) {

		private static <T> Tab<T> of(class_2561 title, Callable<List<T>> list, Loader<T> loader) {
			var cache = new HashMap<T, ImageInstance>();
			return new Tab<>(title, list, cache, o -> {
				if (cache.containsKey(o)) {
					return cache.get(o);
				}
				var val = loader.load(o);
				cache.put(o, val);
				return val;
			});
		}

		private static final Tab<Path> LOCAL = of(new class_2588("gallery.title.local"), () -> {
			try (Stream<Path> screenshots = Files.list(SCREENSHOTS_DIR)) {
				return screenshots.filter(Files::isRegularFile).sorted(Comparator.<Path>comparingLong(p -> {
					try {
						return Files.getLastModifiedTime(p).toMillis();
					} catch (IOException e) {
						return 0L;
					}
				}).reversed()).toList();
			}
		}, ImageInstance.LocalImpl::new);

		private static final Tab<String> SHARED = of(new class_2588("gallery.title.shared"), () ->
			FriendRequest.getInstance().getFriendUuids()
				.thenApply(res -> res.stream().map(UserRequest::getUploadedImages)
					.map(CompletableFuture::join)
					.filter(Optional::isPresent)
					.map(Optional::get)
					.reduce(new ArrayList<>(), (l1, l2) -> {
						l1.addAll(l2);
						return l1;
					})).join(), url -> ImageShare.getInstance().downloadImage(url).join());

		interface Loader<T> {
			ImageInstance load(T obj) throws Exception;
		}
	}

	private Tab<?> current;

	private final class_437 parent;
	private final Watcher watcher;

	public GalleryScreen(class_437 parent) {
		super(new class_2588("gallery.title"));
		this.parent = parent;
		this.current = Tab.LOCAL;
		this.watcher = Watcher.createSelfTicking(SCREENSHOTS_DIR, () -> {
			if (current == Tab.LOCAL) {
				method_25423(field_22787, field_22789, field_22790);
			}
		});
	}

	private static final int entrySpacing = 4,
		entryWidth = 100,
		entryHeight = 75,
		marginLeftRight = 10;

	private boolean isError, online;
	private ImageList area;

	@Override
	public void method_25394(class_4587 graphics, int mouseX, int mouseY, float delta) {
		method_25420(graphics);
		area.method_25394(graphics, mouseX, mouseY, delta);
		super.method_25394(graphics, mouseX, mouseY, delta);

		if (online) {
			method_27534(graphics, field_22793, method_25440(), field_22789 / 2, 40 / 2 - 2 - field_22793.field_2000, -1);
			method_27534(graphics, field_22793, current.title(), field_22789 / 2, 40 / 2 + 2, -1);
		} else {
			method_27534(graphics, field_22793, method_25440(), field_22789 / 2, 40 / 2 - field_22793.field_2000 / 2, -1);
		}

		if (isError) {
			method_27534(graphics, field_22793, new class_2588("gallery.error.loading"), field_22789 / 2, 36, -1);
		}
	}

	@Override
	protected void method_25426() {
		online = API.getInstance().isAuthenticated();

		int columnCount = (field_22789 - (marginLeftRight * 2) + entrySpacing - 13) / (entryWidth + entrySpacing); // -13 to always have enough space for the scrollbar

		area = new ImageList(field_22787, field_22789, field_22790, 33, field_22790 - 40, entryHeight + entrySpacing, columnCount);
		method_25429(area);

		method_20085(area);
		CompletableFuture.runAsync(() -> {
			try {
				loadTab(current, columnCount, area);
			} catch (Exception e) {
				isError = true;
				method_20085(method_25411(new class_4185(field_22789 / 2 - 75, 36 + field_22793.field_2000 + 8, 150, 20, new class_2588("gallery.reload"), b -> method_25423(field_22787, field_22789, field_22790))));
			}
		});

		int buttonWidth = columnCount <= 5 && online ? 100 : 150;
		int footerButtonX = online ? field_22789 / 2 - buttonWidth - buttonWidth / 2 - 4 : field_22789 / 2 - buttonWidth - 2;
		int footerButtonY = field_22790 - 33 / 2 - 10;
		if (online) {
			class_4185 switchTab;
			if (current == Tab.SHARED) {
				switchTab = new class_4185(footerButtonX, footerButtonY, buttonWidth, 20, new class_2588("gallery.tab.local"), b -> setTab(Tab.LOCAL));
			} else {
				switchTab = new class_4185(footerButtonX, footerButtonY, buttonWidth, 20, new class_2588("gallery.tab.shared"), b -> setTab(Tab.SHARED));
			}
			method_25411(switchTab);
			footerButtonX += buttonWidth + 4;
		}
		method_25411(new class_4185(footerButtonX, footerButtonY, buttonWidth, 20, new class_2588("gallery.download_external"), b -> field_22787.method_1507(new DownloadImageScreen(this))));
		footerButtonX += buttonWidth + 4;
		method_25411(new class_4185(footerButtonX, footerButtonY, buttonWidth, 20, class_5244.field_24339, b -> method_25419()));
	}

	@Override
	public void method_25419() {
		Tab.LOCAL.loadingCache().forEach((path, instance) -> {
			if (instance != null) {
				field_22787.method_1531().method_4615(instance.id());
			}
		});
		Tab.LOCAL.loadingCache().clear();
		Tab.SHARED.loadingCache().forEach((s, instance) -> {
			if (instance != null) {
				field_22787.method_1531().method_4615(instance.id());
			}
		});
		Tab.SHARED.loadingCache().clear();
		Watcher.close(watcher);
		field_22787.method_1507(parent);
	}

	private void setTab(Tab<?> tab) {
		current = tab;
		method_25423(field_22787, field_22789, field_22790);
	}

	private <T> void loadTab(Tab<T> tab, int columnCount, ImageList area) throws Exception {
		List<T> images = tab.list.call();
		int size = images.size();
		for (int i = 0; i < size; i += columnCount) {
			ImageListEntry row = new ImageListEntry(columnCount, area);
			area.addEntry(row);
			for (int x = 0; x < columnCount; x++) {
				if (i + x >= size) {
					break;
				}
				T p = images.get(i + x);
				var entry = new ImageEntry(entryWidth, entryHeight, () -> tab.loader.load(p), row);
				row.add(entry);
			}
		}
	}

	// This image list is loading its entries lazily! :)
	private class ImageEntry extends class_4185 {

		private static final int bgColor = Colors.DARK_GRAY.toInt();
		private static final int accent = Colors.GRAY.withBrightness(0.5f).withAlpha(128).toInt();

		private final class_327 font;
		private final Callable<ImageInstance> instanceSupplier;
		private final ImageListEntry row;
		private long loadStart;
		private CompletableFuture<ImageInstance> future;

		protected ImageEntry(int width, int height, Callable<ImageInstance> instanceSupplier, ImageListEntry row) {
			super(0, 0, width, height, class_2585.field_24366, b -> {
			});
			this.instanceSupplier = instanceSupplier;
			this.row = row;
			this.font = class_310.method_1551().field_1772;
		}

		private CompletableFuture<ImageInstance> load() {
			if (future == null) {
				loadStart = class_156.method_658();
				future = CompletableFuture.supplyAsync(() -> {
					try {
						var instance = instanceSupplier.call();
						method_25355(new class_2585(instance.filename()));
						return instance;
					} catch (Exception e) {
						field_22787.method_20493(() -> row.remove(this));
						return null;
					}
				});
			}
			return future;
		}

		@Override
		public void method_25306() {
			field_22787.method_1507(ImageScreen.create(GalleryScreen.this, load(), false));
		}

		@Override
		public void method_25359(class_4587 guiGraphics, int mouseX, int mouseY, float partialTick) {
			if (load().isDone() && load().join() != null) {
				field_22787.method_1531().method_22813(load().join().id());
				method_25290(guiGraphics, field_22760, field_22761, 0, 0, method_25368(), method_25364() - font.field_2000 - 2, method_25368(), method_25364() - font.field_2000 - 2);
				drawScrollingText(guiGraphics, font, 2, -1);
			} else {
				float delta = (float) easeInOutCubic((class_156.method_658() - loadStart) % 1000f / 1000f);

				method_25294(guiGraphics, getX() + 2, getY() + 2, getXEnd() - 2, getYEnd() - font.field_2000 - 2, bgColor);
				drawHorizontalGradient(guiGraphics, getX() + 2, getY() + 2, getYEnd() - font.field_2000 - 2, lerp(delta, getX() + 2, getXEnd() - 2));

				method_25294(guiGraphics, getX() + 2, getYEnd() - font.field_2000 - 1, getXEnd() - 2, getYEnd() - 2, bgColor);
				drawHorizontalGradient(guiGraphics, getX() + 2, getYEnd() - font.field_2000 - 1, getYEnd() - 2, lerp(delta, getX() + 2, getXEnd() - 2));
			}
			DrawUtil.outlineRect(guiGraphics, getX(), getY(), method_25368(), method_25364(), row.list.isInListContent(mouseX, mouseY) && method_25367() ? -1 : bgColor);
		}

		private int getX() {
			return field_22760;
		}

		private int getY() {
			return field_22761;
		}

		protected int getYEnd() {
			return field_22761 + method_25364();
		}

		protected int getXEnd() {
			return field_22760 + method_25368();
		}

		private void drawHorizontalGradient(class_4587 guiGraphics, int x1, int y1, int y2, int x2) {
			class_287 consumer = class_289.method_1348().method_1349();
			class_1159 matrix4f = guiGraphics.method_23760().method_23761();
			consumer.method_22918(matrix4f, x1, y1, 0).method_1336(bgColor >> 16 & 255, bgColor >> 8 & 255, bgColor & 255, bgColor >> 24 & 255);
			consumer.method_22918(matrix4f, x1, y2, 0).method_1336(bgColor >> 16 & 255, bgColor >> 8 & 255, bgColor & 255, bgColor >> 24 & 255);
			consumer.method_22918(matrix4f, x2, y2, 0).method_1336(accent >> 16 & 255, accent >> 8 & 255, accent & 255, accent >> 24 & 255);
			consumer.method_22918(matrix4f, x2, y1, 0).method_1336(accent >> 16 & 255, accent >> 8 & 255, accent & 255, accent >> 24 & 255);
			class_289.method_1348().method_1350();
		}

		private double easeInOutCubic(double x) {
			return x < 0.5 ? 4 * x * x * x : 1 - Math.pow(-2 * x + 2, 3) / 2;
		}

		private int lerp(float delta, int start, int end) {
			return (int) class_3532.method_15363(class_3532.method_16439(delta, start, end), start, end);
		}

		@Override
		protected @NotNull class_5250 method_25360() {
			return new class_2588("gui.narrate.button", new class_2588("gallery.image.view"));
		}

		protected void drawScrollingText(class_4587 guiGraphics, class_327 font, int offset, int color) {
			int i = this.field_22760 + offset;
			int j = this.field_22760 + this.method_25368() - offset;
			DrawUtil.drawScrollableText(guiGraphics, font, this.method_25369(), i, this.field_22761 + method_25364() - font.field_2000 - 1, j, this.field_22761 + this.method_25364(), color);
		}
	}

	private static class ImageListEntry extends class_4265.class_4266<ImageListEntry> {

		private final List<ImageEntry> buttons;
		private final int size;
		private final ImageList list;

		public ImageListEntry(int size, ImageList list) {
			this.size = size;
			buttons = new ArrayList<>(size);
			this.list = list;
		}

		@Override
		public void method_25343(class_4587 guiGraphics, int index, int top, int left, int width, int height, int mouseX, int mouseY, boolean hovering, float partialTick) {
			if (Math.max(left, list.method_25342()) <= Math.min(left + width, list.method_25342() + list.getWidth()) - 1 &&
				Math.max(top - height, list.getY()) <= Math.min(top + height * 2, list.getY() + list.getHeight()) - 1) {
				buttons.forEach(e -> {
					e.field_22761 = top;
					e.method_25394(guiGraphics, mouseX, mouseY, partialTick);
				});
			} else {
				buttons.forEach(e -> {
					e.field_22761 = top;
				});
			}
		}

		public void add(ImageEntry e) {
			buttons.add(e);
			repositionButtons();
		}

		public void remove(ImageEntry e) {
			buttons.remove(e);
			if (buttons.isEmpty()) {
				list.removeEntry(this);
			} else if (buttons.size() < size) {
				list.shiftEntries(this);
			}
			repositionButtons();
		}

		private void repositionButtons() {
			int x = list.method_25342();
			for (ImageEntry e : buttons) {
				e.field_22760 = x;
				x += e.method_25368() + entrySpacing;
			}
		}

		public ImageEntry pop() {
			var entry = buttons.remove(0);
			if (buttons.isEmpty()) {
				list.removeEntry(this);
				list.method_25307(list.method_25341());
			} else if (buttons.size() < size) {
				list.shiftEntries(this);
			}
			repositionButtons();
			return entry;
		}

		@Override
		public @NotNull List<? extends class_364> method_25396() {
			return buttons;
		}
	}

	private static class ImageList extends class_4265<ImageListEntry> {

		private final int rowWidth;

		public ImageList(class_310 minecraft, int screenWidth, int screenHeight, int top, int bottom, int entryHeight, int columns) {
			super(minecraft, screenWidth, screenHeight, top, bottom, entryHeight);
			this.rowWidth = columns * (entryWidth + entrySpacing) - entrySpacing;
		}

		@Override
		public int addEntry(ImageListEntry entry) {
			return super.method_25321(entry);
		}

		@Override
		public int method_25322() {
			return rowWidth;
		}

		@Override
		public int method_25342() {
			return this.field_19088 + this.field_22742 / 2 - this.method_25322() / 2;
		}

		@Override
		public boolean removeEntry(ImageListEntry entry) {
			return super.method_25330(entry);
		}

		public void shiftEntries(ImageListEntry origin) {
			int originIndex = method_25396().indexOf(origin);
			int lastIndex = method_25396().size() - 1;
			if (originIndex == lastIndex) {
				return;
			}
			ImageListEntry next = method_25326(originIndex + 1);
			origin.add(next.pop());
		}

		@Override
		protected int method_25329() {
			return method_25342() + method_25322() + 6 + 2;
		}

		public boolean isInListContent(int x, int y) {
			return x >= field_19088 && x < field_19087 && y >= field_19085 && y < field_19086;
		}

		public int getY() {
			return field_19085;
		}

		public int getWidth() {
			return field_22742;
		}

		public int getHeight() {
			return field_19086 - field_19085;
		}
	}
}
