/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.screenshotUtils;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import net.minecraft.class_156;
import net.minecraft.class_2585;
import net.minecraft.class_2588;
import net.minecraft.class_310;
import net.minecraft.class_339;
import net.minecraft.class_4185;
import net.minecraft.class_437;
import net.minecraft.class_4587;
import net.minecraft.class_5244;
import net.minecraft.class_5250;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientCommon;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.api.util.UUIDHelper;
import io.github.axolotlclient.util.GsonHelper;
import io.github.axolotlclient.util.notifications.Notifications;

public class ImageScreen extends class_437 {

	private final class_437 parent;
	private final ImageInstance image;
	private final boolean freeOnClose;
	private final boolean isRemote;
	private final CompletableFuture<String> uploader;

	static class_437 create(class_437 parent, CompletableFuture<ImageInstance> future, boolean freeOnClose) {
		if (future.isDone()) {
			if (future.join() != null) {
				return new ImageScreen(parent, future.join(), freeOnClose);
			} else {
				return parent;
			}
		}
		return new LoadingImageScreen(parent, future.thenAccept(i -> {
			if (i != null) {
				class_310.method_1551().execute(() -> class_310.method_1551().method_1507(new ImageScreen(parent, i, freeOnClose)));
			} else {
				class_310.method_1551().execute(() -> class_310.method_1551().method_1507(parent));
			}
		}), freeOnClose);
	}

	private ImageScreen(class_437 parent, ImageInstance instance, boolean freeOnClose) {
		super(new class_2585(instance.filename()));
		this.parent = parent;
		this.image = instance;
		this.freeOnClose = freeOnClose;
		this.isRemote = image instanceof ImageInstance.Remote;
		if (isRemote) {
			this.uploader = UUIDHelper.tryGetUsernameAsync(((ImageInstance.Remote) image).uploader());
		} else {
			this.uploader = null;
		}
	}

	@Override
	public void method_25394(class_4587 graphics, int mouseX, int mouseY, float delta) {
		method_25420(graphics);
		super.method_25394(graphics, mouseX, mouseY, delta);
		if (isRemote && uploader.isDone()) {
			ImageInstance.Remote r = (ImageInstance.Remote) image;
			method_27534(graphics, field_22793, method_25440(), field_22789 / 2, 38 / 2 - field_22793.field_2000 - 2, -1);
			method_27534(graphics, field_22793,
				new class_2588("gallery.image.upload_details", uploader.join(),
					r.sharedAt().atZone(ZoneId.systemDefault()).format(AxolotlClientCommon.getInstance().getConfig().getDateTimeFormatter())),
				field_22789 / 2, 38 / 2 + 2, -1);

		} else {
			method_27534(graphics, field_22793, method_25440(), field_22789 / 2, 33 / 2 - field_22793.field_2000 / 2, -1);
		}
	}

	@Override
	protected void method_25426() {

		int contentsHeight = field_22790 - ((isRemote ? 38 : 33) + 33);
		int buttonWidth = 75;
		double imgAspectRatio = image.image().method_4307() / (double) image.image().method_4323();
		int imageWidth = Math.min((int) (contentsHeight * imgAspectRatio), field_22789 - buttonWidth - 4 - 20);
		int imageHeight = (int) (imageWidth / imgAspectRatio);


		var element = method_25411(new ImageElement(imageWidth, imageHeight));
		if (field_22789 / 2 > (imageWidth / 2) + buttonWidth + 4) {
			element.setPosition(field_22789 / 2 - imageWidth / 2, 36);
		} else {
			element.setPosition(field_22789 / 2 - imageWidth / 2 - buttonWidth / 2 - 2, 36);
		}
		int actionX = element.field_22760 + imageWidth + 4;
		var actions = new ArrayList<class_4185>();
		if (image instanceof ImageInstance.Local local) {
			if (API.getInstance().isAuthenticated() && !(image instanceof ImageInstance.Remote)) {
				actions.add(new class_4185(0, 0, buttonWidth, 20, new class_2588("gallery.image.upload"), b -> {
					b.field_22763 = false;
					ImageShare.getInstance().upload(local.location()).thenAccept(s -> {
						if (s.isEmpty()) {
							Notifications.getInstance().addStatus("gallery.image.upload.failure", "gallery.image.upload.failure.description");
						} else {
							client.execute(() -> client.openScreen(new ImageScreen(parent, local.toShared(s, API.getInstance().getSelf().getUuid(), Instant.now()), freeOnClose)));
							client.keyboard.setClipboard(s);
							Notifications.getInstance().addStatus("gallery.image.upload.success", "gallery.image.upload.success.description", s);
						}
					});
				}));
			}
			actions.add(new class_4185(0, 0, buttonWidth, 20, new class_2588("gallery.image.copy"), b -> ScreenshotCopying.copy(local.location())));
			actions.add(new class_4185(0, 0, buttonWidth, 20, new class_2588("gallery.image.open.external"), b -> class_156.method_668().method_673(local.location().toUri())));
		}
		if (image instanceof ImageInstance.Remote remote) {
			if (!(image instanceof ImageInstance.Local)) {
				actions.add(new class_4185(0, 0, buttonWidth, 20, new class_2588("gallery.image.save"), b -> {
					b.field_22763 = false;
					try {
						Path out = saveSharedImage(remote);
						field_22787.method_1507(new ImageScreen(parent, remote.toShared(out), freeOnClose));
					} catch (IOException e) {
						Notifications.getInstance().addStatus("gallery.image.save.failure", "gallery.image.save.failure.description", e.getMessage());
						AxolotlClient.LOGGER.warn("Failed to save shared image!", e);
					}
				}));
				actions.add(new class_4185(0, 0, buttonWidth, 20, new class_2588("gallery.image.copy"), b -> {
					try {
						ScreenshotCopying.copy(image.image().method_24036());
					} catch (IOException e) {
						Notifications.getInstance().addStatus("gallery.image.copy.failure", "gallery.image.copy.failure.description", e.getMessage());
						AxolotlClient.LOGGER.warn("Failed to copy shared image!", e);
					}
				}));
			}
			actions.add(new class_4185(0, 0, buttonWidth, 20, new class_2588("gallery.image.open.external.browser"), b -> class_156.method_668().method_670(remote.url())));
			actions.add(new class_4185(0, 0, buttonWidth, 20, new class_2588("gallery.image.copy_url"), b -> field_22787.field_1774.method_1455(remote.url())));
		}
		int actionY = element.field_22761;
		for (class_4185 w : actions) {
			w.field_22760 = actionX;
			w.field_22761 = actionY;
			method_25411(w);
			actionY += 4 + w.method_25364();
		}

		method_25411(new class_4185(field_22789 / 2 - 75, field_22790 - 33 + 33 / 2 - 10, 150, 20, class_5244.field_24339, b -> method_25419()));
	}

	@Override
	public void method_25419() {
		if (freeOnClose) {
			field_22787.method_1531().method_4615(image.id());
		}
		field_22787.method_1507(parent);
	}

	private Path saveSharedImage(ImageInstance.Remote remote) throws IOException {
		Path out = GalleryScreen.SCREENSHOTS_DIR.resolve("shared")
			.resolve(remote.uploader())
			.resolve(remote.filename());
		Path infoJson = out.resolveSibling(remote.filename() + ".json");

		Files.createDirectories(out.getParent());
		remote.image().method_4314(out);
		Object json = Map.of("uploader", remote.uploader(), "shared_at", remote.sharedAt());
		Files.writeString(infoJson, GsonHelper.GSON.toJson(json));
		return out;
	}

	private class ImageElement extends class_339 {

		public ImageElement(int width, int height) {
			super(0, 0, width, height, class_2585.field_24366);
			field_22763 = false;
		}

		@Override
		public void method_25359(class_4587 guiGraphics, int mouseX, int mouseY, float partialTick) {
			field_22787.method_1531().method_22813(image.id());
			method_25290(guiGraphics, field_22760, field_22761, 0, 0, method_25368(), method_25364(), method_25368(), method_25364());
		}

		@Override
		protected class_5250 method_25360() {
			return class_2585.field_24366.method_27662();
		}

		public void setPosition(int x, int y) {
			this.field_22760 = x;
			this.field_22761 = y;
		}
	}
}
