/*
 * Copyright © 2023 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.sky;

import java.util.Locale;
import java.util.Objects;
import net.minecraft.class_1159;
import net.minecraft.class_1160;
import net.minecraft.class_287;
import net.minecraft.class_289;
import net.minecraft.class_290;
import net.minecraft.class_291;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3532;
import net.minecraft.class_4493;
import net.minecraft.class_4587;
import net.minecraft.class_758;
import com.google.gson.JsonObject;
import com.mojang.blaze3d.systems.RenderSystem;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.mixin.WorldRendererAccessor;
import io.github.axolotlclient.util.Util;
import org.lwjgl.opengl.GL14;

/**
 * This implementation of custom skies is based on the FabricSkyBoxes mod by AMereBagatelle
 * <a href="https://github.com/AMereBagatelle/FabricSkyBoxes">Github Link.</a>
 *
 * <p>License: MIT</p>
 **/

// TODO fix rotation & blending issues with shooting stars, implement more missing features like worlds, weather, biomes...

@SuppressWarnings("deprecation")
public abstract class SkyboxInstance {

	protected final class_2960 MOON_PHASES = new class_2960("textures/environment/moon_phases.png");
	protected final class_2960 SUN = new class_2960("textures/environment/sun.png");
	protected int blendMode = 1;
	// ! These are the options variables.  Do not mess with these.
	protected boolean alwaysOn;
	protected float maxAlpha = 1f;
	protected boolean manualBlend = false;
	protected int blendSrcFactor = 1;
	protected int blendDstFactor = 1;
	protected int blendEquation;
	protected boolean rotate = false;
	protected float rotationSpeed = 1F;
	protected float[] rotationStatic = new float[]{0, 0, 0};
	protected float[] rotationAxis = new float[]{0, 0, 0};
	protected boolean showSun = true;
	protected boolean showMoon = true;
	protected boolean showStars = true;
	JsonObject object;
	float alpha = 1F;
	class_2960[] textures = new class_2960[6];
	int[] fade = new int[4];

	public SkyboxInstance(JsonObject json) {
		this.object = json;
	}

	public float getAlpha() {
		if (alwaysOn) {
			return 1F;
		}

		int currentTime = (int) Objects.requireNonNull(class_310.method_1551().field_1687).method_8532() % 24000; // modulo so that it's bound to 24000
		int durationIn = Util.getTicksBetween(fade[0], fade[1]);
		int durationOut = Util.getTicksBetween(fade[2], fade[3]);

		int startFadeIn = fade[0] % 24000;
		int endFadeIn = fade[1] % 24000;

		if (endFadeIn < startFadeIn) {
			endFadeIn += 24000;
		}

		int startFadeOut = fade[2] % 24000;
		int endFadeOut = fade[3] % 24000;

		if (startFadeOut < endFadeIn) {
			startFadeOut += 24000;
		}

		if (endFadeOut < startFadeOut) {
			endFadeOut += 24000;
		}

		int tempInTime = currentTime;

		if (tempInTime < startFadeIn) {
			tempInTime += 24000;
		}

		int tempFullTime = currentTime;

		if (tempFullTime < endFadeIn) {
			tempFullTime += 24000;
		}

		int tempOutTime = currentTime;

		if (tempOutTime < startFadeOut) {
			tempOutTime += 24000;
		}

		float maxPossibleAlpha;

		if (startFadeIn < tempInTime && endFadeIn >= tempInTime) {
			maxPossibleAlpha = 1f - (((float) (endFadeIn - tempInTime)) / durationIn); // fading in
		} else if (endFadeIn < tempFullTime && startFadeOut >= tempFullTime) {
			maxPossibleAlpha = 1f; // fully faded in
		} else if (startFadeOut < tempOutTime && endFadeOut >= tempOutTime) {
			maxPossibleAlpha = (float) (endFadeOut - tempOutTime) / durationOut; // fading out
		} else {
			maxPossibleAlpha = 0f; // default not showing
		}

		return alpha = class_3532.method_15363(maxPossibleAlpha * maxAlpha, 0, 1);
	}

	protected int parseBlend(String str) {
		if (str == null) {
			return 1;
		} else {
			return switch (str.toLowerCase(Locale.ENGLISH).trim()) {
				case "alpha" -> 0;
				case "add" -> 1;
				case "subtract" -> 2;
				case "multiply" -> 3;
				case "dodge" -> 4;
				case "burn" -> 5;
				case "screen" -> 6;
				case "overlay" -> 7;
				case "replace" -> 8;
				default -> {
					AxolotlClient.LOGGER.warn("Unknown blend: " + str);
					yield 1;
				}
			};
		}
	}

	public void render(class_4587 matrices, float tickDelta) {
		//noinspection DataFlowIssue
		float brightness = class_310.method_1551().field_1687.method_8430(tickDelta);
		matrices.method_22903();
		setupBlend(brightness);
		setupRotate(matrices, tickDelta, brightness);
		renderSkybox(matrices);
		clearBlend(brightness);
		clearRotate(matrices);
		matrices.method_22909();
		renderDecorations(matrices, tickDelta);
		RenderSystem.enableTexture();
	}

	protected void setupBlend(float brightness) {
		if (manualBlend) {
			RenderSystem.enableBlend();
			RenderSystem.blendFunc(blendSrcFactor, blendDstFactor);
			GL14.glBlendEquation(blendEquation);
			RenderSystem.enableTexture();
			return;
		}

		switch (blendMode) {
			case 0:

				class_4493.method_22056();
				class_4493.method_21984(770, 771);
				break;

			case 1:

				class_4493.method_22056();
				class_4493.method_21984(770, 1);
				break;

			case 2:
				class_4493.method_22056();
				class_4493.method_21984(775, 0);
				break;

			case 3:
				class_4493.method_22056();
				class_4493.method_21984(774, 771);
				break;

			case 4:
				class_4493.method_22056();
				class_4493.method_21984(1, 1);
				break;

			case 5:
				class_4493.method_22056();
				class_4493.method_21984(0, 769);
				break;

			case 6:
				class_4493.method_22056();
				class_4493.method_21984(1, 769);
				break;

			case 7:
				class_4493.method_22056();
				class_4493.method_21984(774, 768);
				break;

			case 8:
				class_4493.method_22053();
				break;
		}
		RenderSystem.color4f(1.0F, 1.0F, 1.0F, brightness);

		RenderSystem.enableTexture();
	}

	protected void setupRotate(class_4587 matrices, float delta, float brightness) {
		matrices.method_22907(class_1160.field_20703.method_23214(rotationStatic[0]));
		matrices.method_22907(class_1160.field_20705.method_23214(rotationStatic[1]));
		matrices.method_22907(class_1160.field_20707.method_23214(rotationStatic[2]));
		if (rotate) {
			RenderSystem.color4f(1.0F, 1.0F, 1.0F, brightness);
			matrices.method_22907(class_1160.field_20703.method_23214(rotationAxis[0]));
			matrices.method_22907(class_1160.field_20705.method_23214(rotationAxis[1]));
			matrices.method_22907(class_1160.field_20707.method_23214(rotationAxis[2]));
			matrices.method_22907(class_1160.field_20705.method_23214(-90.0F));
			//noinspection DataFlowIssue
			matrices.method_22907(class_1160.field_20702.method_23214(
				class_310.method_1551().field_1687.method_30274(delta) * 360F * rotationSpeed));
			matrices.method_22907(class_1160.field_20706.method_23214(rotationAxis[0]));
			matrices.method_22907(class_1160.field_20704.method_23214(rotationAxis[1]));
			matrices.method_22907(class_1160.field_20702.method_23214(rotationAxis[2]));
		}
	}

	public abstract void renderSkybox(class_4587 matrices);

	protected void clearBlend(float brightness) {
		RenderSystem.enableBlend();
		RenderSystem.blendFunc(770, 1);
		RenderSystem.color4f(1.0F, 1.0F, 1.0F, brightness);
	}

	protected void clearRotate(class_4587 matrices) {
		matrices.method_22907(class_1160.field_20702.method_23214(rotationStatic[0]));
		matrices.method_22907(class_1160.field_20704.method_23214(rotationStatic[1]));
		matrices.method_22907(class_1160.field_20706.method_23214(rotationStatic[2]));
	}

	protected void renderDecorations(class_4587 matrices, float delta) {
		WorldRendererAccessor worldRendererAccessor = (WorldRendererAccessor) class_310
			.method_1551().field_1769;
		RenderSystem.enableTexture();
		RenderSystem.blendFuncSeparate(class_4493.class_4535.field_22541, class_4493.class_4534.field_22518,
			class_4493.class_4535.field_22534, class_4493.class_4534.field_22527);

		matrices.method_22903();
		//noinspection DataFlowIssue
		float i = 1.0F - class_310.method_1551().field_1687.method_8430(delta);
		RenderSystem.color4f(1.0F, 1.0F, 1.0F, i);
		matrices.method_22907(class_1160.field_20705.method_23214(-90.0F));
		matrices.method_22907(
			class_1160.field_20703.method_23214(class_310.method_1551().field_1687.method_30274(delta) * 360.0F));
		class_1159 matrix4f2 = matrices.method_23760().method_23761();
		float k = 30.0F;

		class_287 bufferBuilder = class_289.method_1348().method_1349();

		if (showSun) {
			class_310.method_1551().method_1531().method_22813(SUN);
			bufferBuilder.method_1328(7, class_290.field_1585);
			bufferBuilder.method_22918(matrix4f2, -k, 100.0F, -k).method_22913(0.0F, 0.0F).method_1344();
			bufferBuilder.method_22918(matrix4f2, k, 100.0F, -k).method_22913(1.0F, 0.0F).method_1344();
			bufferBuilder.method_22918(matrix4f2, k, 100.0F, k).method_22913(1.0F, 1.0F).method_1344();
			bufferBuilder.method_22918(matrix4f2, -k, 100.0F, k).method_22913(0.0F, 1.0F).method_1344();
			class_289.method_1348().method_1350();
		}
		if (showMoon) {
			k = 20.0F;
			class_310.method_1551().method_1531().method_22813(MOON_PHASES);
			int r = class_310.method_1551().field_1687.method_30273();
			int s = r % 4;
			int m = r / 4 % 2;
			float t = (float) (s) / 4.0F;
			float o = (float) (m) / 2.0F;
			float p = (float) (s + 1) / 4.0F;
			float q = (float) (m + 1) / 2.0F;
			bufferBuilder.method_1328(7, class_290.field_1585);
			bufferBuilder.method_22918(matrix4f2, -k, -100.0F, k).method_22913(p, q).method_1344();
			bufferBuilder.method_22918(matrix4f2, k, -100.0F, k).method_22913(t, q).method_1344();
			bufferBuilder.method_22918(matrix4f2, k, -100.0F, -k).method_22913(t, o).method_1344();
			bufferBuilder.method_22918(matrix4f2, -k, -100.0F, -k).method_22913(p, o).method_1344();
			class_289.method_1348().method_1350();
		}
		if (showStars) {
			RenderSystem.disableTexture();
			float u = class_310.method_1551().field_1687.method_8442(delta) * i;
			if (u > 0.0F) {
				RenderSystem.color4f(u, u, u, u);
				class_758.method_23792();
				worldRendererAccessor.getStarsBuffer().method_1353();
				worldRendererAccessor.getStarsBuffer().method_1351(matrices.method_23760().method_23761(), 7);
				class_291.method_1354();
			}

			RenderSystem.color4f(1.0F, 1.0F, 1.0F, 1.0F);
			RenderSystem.disableBlend();
			matrices.method_22909();
		}

		RenderSystem.enableTexture();
		RenderSystem.depthMask(true);
	}
}
