/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.hud;

import java.util.List;
import java.util.Optional;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_4185;
import net.minecraft.class_437;
import net.minecraft.class_5244;
import io.github.axolotlclient.AxolotlClient;
import io.github.axolotlclient.AxolotlClientConfig.api.AxolotlClientConfig;
import io.github.axolotlclient.AxolotlClientConfig.api.options.OptionCategory;
import io.github.axolotlclient.AxolotlClientConfig.impl.options.BooleanOption;
import io.github.axolotlclient.AxolotlClientConfig.impl.util.ConfigStyles;
import io.github.axolotlclient.modules.hud.gui.component.HudEntry;
import io.github.axolotlclient.modules.hud.snapping.SnappingHelper;
import io.github.axolotlclient.modules.hud.util.DrawPosition;
import io.github.axolotlclient.modules.hud.util.Rectangle;
import org.lwjgl.glfw.GLFW;

/**
 * This implementation of Hud modules is based on KronHUD.
 * <a href="https://github.com/DarkKronicle/KronHUD">Github Link.</a>
 *
 * <p>License: GPL-3.0</p>
 */

public class HudEditScreen extends class_437 {

	private static final BooleanOption snapping = new BooleanOption("snapping", true);
	private static final OptionCategory hudEditScreenCategory = OptionCategory.create("hudEditScreen");
	private static final int GRAB_TOLERANCE = 5;
	private static final long MOVE_CURSOR = GLFW.glfwCreateStandardCursor(GLFW.GLFW_RESIZE_ALL_CURSOR);
	private static final long DEFAULT_CURSOR = GLFW.glfwCreateStandardCursor(GLFW.GLFW_ARROW_CURSOR);
	private static final long NWSE_RESIZE_CURSOR = GLFW.glfwCreateStandardCursor(GLFW.GLFW_RESIZE_NWSE_CURSOR),
		NESW_RESIZE_CURSOR = GLFW.glfwCreateStandardCursor(GLFW.GLFW_RESIZE_NESW_CURSOR);

	public static boolean isSnappingEnabled() {
		return snapping.get();
	}

	public static void toggleSnapping() {
		snapping.toggle();
	}

	static {
		hudEditScreenCategory.add(snapping);
		AxolotlClient.config().hidden.add(hudEditScreenCategory);
	}

	private final class_437 parent;
	private HudEntry current;
	private DrawPosition offset = null;
	private boolean mouseDown;
	private SnappingHelper snap;
	private long currentCursor;
	private ModificationMode mode = ModificationMode.NONE;

	public HudEditScreen() {
		this(null);
	}

	public HudEditScreen(class_437 parent) {
		super(class_2561.method_43473());
		updateSnapState();
		mouseDown = false;
		this.parent = parent;
	}

	private void updateSnapState() {
		if (snapping.get() && current != null) {
			List<Rectangle> bounds = HudManager.getInstance().getAllBounds();
			bounds.remove(current.getTrueBounds());
			snap = new SnappingHelper(bounds, current.getTrueBounds());
		} else if (snap != null) {
			snap = null;
		}
	}

	private void setCursor(long cursor) {
		if (cursor > 0 && cursor != currentCursor) {
			currentCursor = cursor;
			GLFW.glfwSetCursor(class_310.method_1551().method_22683().method_4490(), cursor);
		}
	}

	@Override
	public void method_25394(class_332 graphics, int mouseX, int mouseY, float delta) {
		super.method_25394(graphics, mouseX, mouseY, delta);

		Optional<HudEntry> entry;
		if (current != null && mode != ModificationMode.NONE) {
			current.setHovered(true);
			entry = Optional.of(current);
		} else {
			entry = HudManager.getInstance().getEntryXY(mouseX, mouseY);
			entry.ifPresent(abstractHudEntry -> abstractHudEntry.setHovered(true));
		}
		HudManager.getInstance().renderPlaceholder(graphics, delta);
		if (entry.isPresent()) {
			var bounds = entry.get().getTrueBounds();
			if (mode == ModificationMode.NONE && bounds.isMouseOver(mouseX, mouseY)) {
				var supportsScaling = entry.get().supportsScaling();
				var xBound = Math.max(0, mouseX - bounds.x());
				var yBound = Math.max(0, mouseY - bounds.y());
				var tolerance = GRAB_TOLERANCE;
				if (supportsScaling && xBound < tolerance && yBound < tolerance) {
					// top-left
					setCursor(NWSE_RESIZE_CURSOR);
				} else if (supportsScaling && Math.abs(xBound - bounds.width()) < tolerance && Math.abs(yBound - bounds.height()) < tolerance) {
					// bottom-right
					setCursor(NWSE_RESIZE_CURSOR);
				} else if (supportsScaling && xBound < tolerance && Math.abs(yBound - bounds.height()) < tolerance) {
					// bottom-left
					setCursor(NESW_RESIZE_CURSOR);
				} else if (supportsScaling && yBound < tolerance && Math.abs(xBound - bounds.width()) < tolerance) {
					// top-right
					setCursor(NESW_RESIZE_CURSOR);
				} else {
					setCursor(MOVE_CURSOR);
				}
			}
		} else if (current == null) {
			setCursor(DEFAULT_CURSOR);
			mode = ModificationMode.NONE;
		}
		if (mouseDown && snap != null) {
			snap.renderSnaps(graphics);
		}
	}

	@Override
	public void method_25432() {
		setCursor(DEFAULT_CURSOR);
		mode = ModificationMode.NONE;
		super.method_25432();
	}

	@Override
	public void method_25426() {
		mode = ModificationMode.NONE;

		HudManager.getInstance().getMoveableEntries().forEach(e -> method_37063(new HudEntryWidget(e)));

		this.method_37063(new class_4185.class_7840(class_2561.method_43471("hud.snapping").method_27693(": ")
			.method_10852(class_2561.method_43471(snapping.get() ? "options.on" : "options.off")),
			buttonWidget -> {
				snapping.toggle();
				buttonWidget.method_25355(class_2561.method_43471("hud.snapping").method_27693(": ")
					.method_10852(class_2561.method_43471(snapping.get() ? "options.on" : "options.off")));
				AxolotlClient.getInstance().getConfigManager().save();
			}).method_46434(field_22789 / 2 - 50, field_22790 / 2 + 12, 100, 20).method_46431());

		this.method_37063(new class_4185.class_7840(class_2561.method_43471("hud.clientOptions"),
			buttonWidget -> {
				class_437 screen = ConfigStyles.createScreen(this, AxolotlClient.getInstance().getConfigManager().getRoot());
				class_310.method_1551().method_1507(screen);
			}).method_46434(field_22789 / 2 - 75, field_22790 / 2 - 10, 150, 20).method_46431());

		if (parent != null)
			method_37063(new class_4185.class_7840(class_5244.field_24339, buttonWidget -> class_310.method_1551().method_1507(parent))
				.method_46434(field_22789 / 2 - 75, field_22790 - 50 + 22, 150, 20).method_46431());
		else
			method_37063(new class_4185.class_7840(class_2561.method_43471("close"),
				buttonWidget -> class_310.method_1551().method_1507(null))
				.method_46434(field_22789 / 2 - 75, field_22790 - 50 + 22, 150, 20).method_46431());
	}

	@Override
	public boolean method_25402(double mouseX, double mouseY, int button) {
		super.method_25402(mouseX, mouseY, button);
		Optional<HudEntry> entry = HudManager.getInstance().getEntryXY((int) Math.round(mouseX),
			(int) Math.round(mouseY));
		if (button == 0) {
			mouseDown = true;
			if (entry.isPresent()) {
				current = entry.get();
				offset = new DrawPosition((int) Math.round(mouseX - current.getTruePos().x()),
					(int) Math.round(mouseY - current.getTruePos().y()));
				var bounds = entry.get().getTrueBounds();
				var xBound = Math.max(0, mouseX - bounds.x());
				var yBound = Math.max(0, mouseY - bounds.y());
				if (currentCursor == NWSE_RESIZE_CURSOR) {
					if (xBound < bounds.width() / 2f && yBound < bounds.height() / 2f) {
						// top-left corner
						mode = ModificationMode.TOP_LEFT;
					} else if (xBound - bounds.width() / 2f > 0 && yBound - bounds.height() / 2f > 0) {
						// bottom-right corner
						mode = ModificationMode.BOTTOM_RIGHT;
					}
				} else if (currentCursor == NESW_RESIZE_CURSOR) {
					if (xBound < bounds.width() / 2f && yBound - bounds.height() / 2f > 0) {
						// bottom-left corner
						mode = ModificationMode.BOTTOM_LEFT;
					} else if (xBound - bounds.width() / 2f > 0 && yBound < bounds.height() / 2f) {
						// top-right corner
						mode = ModificationMode.TOP_RIGHT;
					}
				} else if (currentCursor == MOVE_CURSOR) {
					updateSnapState();
					mode = ModificationMode.MOVE;
				}
				return true;
			} else {
				mode = ModificationMode.NONE;
				current = null;
			}
		} else if (button == 1) {
			entry.ifPresent(abstractHudEntry -> {
				class_437 screen = ConfigStyles.createScreen(this, abstractHudEntry.getCategory());
				class_310.method_1551().method_1507(screen);
			});
		}
		return false;
	}

	@Override
	public boolean method_25406(double mouseX, double mouseY, int button) {
		if (current != null) {
			AxolotlClientConfig.getInstance().getConfigManager(current.getCategory()).save();
		}
		current = null;
		snap = null;
		mouseDown = false;
		mode = ModificationMode.NONE;
		setCursor(DEFAULT_CURSOR);
		return super.method_25406(mouseX, mouseY, button);
	}

	@Override
	public boolean method_25403(double mouseX, double mouseY, int button, double deltaX, double deltaY) {
		if (current != null) {
			if (mode == ModificationMode.MOVE) {
				current.setX((int) mouseX - offset.x() + current.offsetTrueWidth());
				current.setY((int) mouseY - offset.y() + current.offsetTrueHeight());
				if (snap != null) {
					Integer snapX, snapY;
					snap.setCurrent(current.getTrueBounds());
					if ((snapX = snap.getCurrentXSnap()) != null) {
						current.setX(snapX + current.offsetTrueWidth());
					}
					if ((snapY = snap.getCurrentYSnap()) != null) {
						current.setY(snapY + current.offsetTrueHeight());
					}
				}
			} else {
				var bounds = current.getTrueBounds();
				double newWidth, newHeight;
				if (mode == ModificationMode.TOP_LEFT) {
					// top-left corner
					newWidth = mouseX - bounds.xEnd();
					newHeight = mouseY - bounds.yEnd();
				} else if (mode == ModificationMode.BOTTOM_LEFT) {
					// bottom-left corner
					newWidth = mouseX - bounds.xEnd();
					newHeight = mouseY - bounds.y();
				} else if (mode == ModificationMode.TOP_RIGHT) {
					// top-right corner
					newWidth = mouseX - bounds.x();
					newHeight = mouseY - bounds.yEnd();
				} else if (mode == ModificationMode.BOTTOM_RIGHT) {
					// bottom-right corner
					newWidth = mouseX - bounds.x();
					newHeight = mouseY - bounds.y();
				} else {
					newWidth = bounds.width();
					newHeight = bounds.height();
				}
				float newScale = current.getScale() * Math.max((float) Math.abs(newWidth) / bounds.width(), (float) Math.abs(newHeight) / bounds.height());
				current.setScale(Math.max(0.1f, newScale));
				if (mode == ModificationMode.TOP_LEFT) {
					// top-left corner
					current.setX(bounds.xEnd() - current.getTrueWidth() + current.offsetTrueWidth());
					current.setY(bounds.yEnd() - current.getTrueHeight() + current.offsetTrueHeight());
				} else if (mode == ModificationMode.BOTTOM_LEFT) {
					// bottom-left corner
					current.setX(bounds.xEnd() - current.getTrueWidth() + current.offsetTrueWidth());
				} else if (mode == ModificationMode.TOP_RIGHT) {
					// top-right corner
					current.setY(bounds.yEnd() - current.getTrueHeight() + current.offsetTrueHeight());
				}
			}
			if (current.tickable()) {
				current.tick();
			}
			return true;
		}
		return false;
	}

	private enum ModificationMode {
		NONE,
		MOVE,
		TOP_LEFT,
		TOP_RIGHT,
		BOTTOM_LEFT,
		BOTTOM_RIGHT
	}
}
