/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.modules.screenshotUtils;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Colors;
import io.github.axolotlclient.api.API;
import io.github.axolotlclient.api.requests.FriendRequest;
import io.github.axolotlclient.api.requests.UserRequest;
import io.github.axolotlclient.modules.hud.util.DrawUtil;
import io.github.axolotlclient.util.Watcher;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_156;
import net.minecraft.class_1921;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_318;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_364;
import net.minecraft.class_4185;
import net.minecraft.class_4265;
import net.minecraft.class_437;
import net.minecraft.class_4588;
import net.minecraft.class_5244;
import net.minecraft.class_5250;
import net.minecraft.class_6379;
import net.minecraft.class_7842;
import net.minecraft.class_7847;
import net.minecraft.class_8132;
import net.minecraft.class_8667;
import org.jetbrains.annotations.NotNull;
import org.joml.Matrix4f;

public class GalleryScreen extends class_437 {

	public static final Path SCREENSHOTS_DIR = FabricLoader.getInstance().getGameDir().resolve(class_318.field_41337);

	private record Tab<T>(class_2561 title, Callable<List<T>> list, Map<T, ImageInstance> loadingCache,
						  Loader<T> loader) {

		private static <T> Tab<T> of(class_2561 title, Callable<List<T>> list, Loader<T> loader) {
			var cache = new HashMap<T, ImageInstance>();
			return new Tab<>(title, list, cache, o -> {
				if (cache.containsKey(o)) {
					return cache.get(o);
				}
				var val = loader.load(o);
				cache.put(o, val);
				return val;
			});
		}

		private static final Tab<Path> LOCAL = of(class_2561.method_43471("gallery.title.local"), () -> {
			try (Stream<Path> screenshots = Files.list(SCREENSHOTS_DIR)) {
				return screenshots.filter(Files::isRegularFile).sorted(Comparator.<Path>comparingLong(p -> {
					try {
						return Files.getLastModifiedTime(p).toMillis();
					} catch (IOException e) {
						return 0L;
					}
				}).reversed()).toList();
			}
		}, ImageInstance.LocalImpl::new);

		private static final Tab<String> SHARED = of(class_2561.method_43471("gallery.title.shared"), () ->
			FriendRequest.getInstance().getFriendUuids()
				.thenApply(res -> res.stream().map(UserRequest::getUploadedImages)
					.map(CompletableFuture::join)
					.filter(Optional::isPresent)
					.map(Optional::get)
					.reduce(new ArrayList<>(), (l1, l2) -> {
						l1.addAll(l2);
						return l1;
					})).join(), url -> ImageShare.getInstance().downloadImage(url).join());

		interface Loader<T> {
			ImageInstance load(T obj) throws Exception;
		}
	}

	private Tab<?> current;

	private final class_437 parent;
	private final Watcher watcher;

	public GalleryScreen(class_437 parent) {
		super(class_2561.method_43471("gallery.title"));
		this.parent = parent;
		this.current = Tab.LOCAL;
		this.watcher = Watcher.createSelfTicking(SCREENSHOTS_DIR, () -> {
			if (current == Tab.LOCAL) {
				method_41843();
			}
		});
	}

	private static final int entrySpacing = 4,
		entryWidth = 100,
		entryHeight = 75,
		marginLeftRight = 10;

	@Override
	protected void method_25426() {
		boolean online = API.getInstance().isAuthenticated();
		class_8132 layout = new class_8132(this);
		layout.method_48995(40);
		class_8667 header = layout.method_48992(class_8667.method_52741().method_52735(4));
		header.method_52739().method_46467();
		header.method_52736(new class_7842(field_22785, field_22793));
		if (online) {
			header.method_52736(new class_7842(current.title(), field_22793));
		}

		int columnCount = (field_22789 - (marginLeftRight * 2) + entrySpacing - 13) / (entryWidth + entrySpacing); // -13 to always have enough space for the scrollbar

		final var area = new ImageList(field_22787, layout.method_25368(), layout.method_57727(), layout.method_48998(), entryHeight + entrySpacing, columnCount);

		layout.method_49000(area, class_7847::method_46461);
		method_48265(area);
		CompletableFuture.runAsync(() -> {
			try {
				loadTab(current, columnCount, area);
			} catch (Exception e) {
				class_8667 error = class_8667.method_52741().method_52735(8);
				error.method_52739().method_46474();
				error.method_52736(new class_7842(class_2561.method_43471("gallery.error.loading"), field_22793));
				method_48265(error.method_52736(class_4185.method_46430(class_2561.method_43471("gallery.reload"), b -> method_41843()).method_46431()));
				layout.method_48999(error);
			}
		});

		var footer = layout.method_48996(class_8667.method_52742()).method_52735(4);
		footer.method_52739().method_46467();
		int buttonWidth = columnCount <= 5 && online ? 100 : 150;
		if (online) {
			class_4185.class_7840 switchTab;
			if (current == Tab.SHARED) {
				switchTab = class_4185.method_46430(class_2561.method_43471("gallery.tab.local"), b -> setTab(Tab.LOCAL));
			} else {
				switchTab = class_4185.method_46430(class_2561.method_43471("gallery.tab.shared"), b -> setTab(Tab.SHARED));
			}
			footer.method_52736(switchTab.method_46432(buttonWidth).method_46431());
		}
		footer.method_52736(class_4185.method_46430(class_2561.method_43471("gallery.download_external"), b -> field_22787.method_1507(new DownloadImageScreen(this)))
			.method_46432(buttonWidth).method_46431());
		footer.method_52736(class_4185.method_46430(class_5244.field_24339, b -> method_25419())
			.method_46432(buttonWidth).method_46431());

		layout.method_48222();
		layout.method_48206(this::method_37063);
	}

	@Override
	public void method_25419() {
		Tab.LOCAL.loadingCache().forEach((path, instance) -> {
			if (instance != null) {
				field_22787.method_1531().method_4615(instance.id());
			}
		});
		Tab.LOCAL.loadingCache().clear();
		Tab.SHARED.loadingCache().forEach((s, instance) -> {
			if (instance != null) {
				field_22787.method_1531().method_4615(instance.id());
			}
		});
		Tab.SHARED.loadingCache().clear();
		Watcher.close(watcher);
		field_22787.method_1507(parent);
	}

	private void setTab(Tab<?> tab) {
		current = tab;
		method_41843();
	}

	private <T> void loadTab(Tab<T> tab, int columnCount, ImageList area) throws Exception {
		List<T> images = tab.list.call();
		int size = images.size();
		for (int i = 0; i < size; i += columnCount) {
			ImageListEntry row = new ImageListEntry(columnCount, area);
			area.addEntry(row);
			for (int x = 0; x < columnCount; x++) {
				if (i + x >= size) {
					break;
				}
				T p = images.get(i + x);
				var entry = new ImageEntry(entryWidth, entryHeight, () -> tab.loader.load(p), row);
				row.add(entry);
			}
		}
	}

	// This image list is loading its entries lazily! :)
	private class ImageEntry extends class_4185 {

		private static final int bgColor = Colors.DARK_GRAY.toInt();
		private static final int accent = Colors.GRAY.withBrightness(0.5f).withAlpha(128).toInt();

		private final class_327 font;
		private final Callable<ImageInstance> instanceSupplier;
		private final ImageListEntry row;
		private long loadStart;
		private CompletableFuture<ImageInstance> future;

		protected ImageEntry(int width, int height, Callable<ImageInstance> instanceSupplier, ImageListEntry row) {
			super(0, 0, width, height, class_2561.method_43473(), b -> {
			}, field_40754);
			this.instanceSupplier = instanceSupplier;
			this.row = row;
			this.font = class_310.method_1551().field_1772;
		}

		private CompletableFuture<ImageInstance> load() {
			if (future == null) {
				loadStart = class_156.method_658();
				future = CompletableFuture.supplyAsync(() -> {
					try {
						var instance = instanceSupplier.call();
						method_25355(class_2561.method_43470(instance.filename()));
						return instance;
					} catch (Exception e) {
						field_22787.execute(() -> row.remove(this));
						return null;
					}
				});
			}
			return future;
		}

		@Override
		public void method_25306() {
			field_22787.method_1507(ImageScreen.create(GalleryScreen.this, load(), false));
		}

		@Override
		protected void method_48579(class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
			if (load().isDone() && load().join() != null) {
				guiGraphics.method_25290(load().join().id(), method_46426(), method_46427(), 0, 0, method_25368(), method_25364() - font.field_2000 - 2, method_25368(), method_25364() - font.field_2000 - 2);
				method_48589(guiGraphics, font, -1);
			} else {
				float delta = (float) easeInOutCubic((class_156.method_658() - loadStart) % 1000f / 1000f);

				guiGraphics.method_25294(method_46426() + 2, method_46427() + 2, method_55442() - 2, method_55443() - font.field_2000 - 2, bgColor);
				drawHorizontalGradient(guiGraphics, method_46426() + 2, method_46427() + 2, method_55443() - font.field_2000 - 2, lerp(delta, method_46426() + 2, method_55442() - 2));

				guiGraphics.method_25294(method_46426() + 2, method_55443() - font.field_2000 - 1, method_55442() - 2, method_55443() - 2, bgColor);
				drawHorizontalGradient(guiGraphics, method_46426() + 2, method_55443() - font.field_2000 - 1, method_55443() - 2, lerp(delta, method_46426() + 2, method_55442() - 2));
			}
			DrawUtil.outlineRect(guiGraphics, method_46426(), method_46427(), method_25368(), method_25364(), method_25367() ? -1 : bgColor);
		}

		private void drawHorizontalGradient(class_332 guiGraphics, int x1, int y1, int y2, int x2) {
			class_4588 consumer = field_22787.method_22940().method_23000().getBuffer(class_1921.method_51784());
			Matrix4f matrix4f = guiGraphics.method_51448().method_23760().method_23761();
			consumer.method_22918(matrix4f, x1, y1, 0).method_39415(ImageEntry.bgColor);
			consumer.method_22918(matrix4f, x1, y2, 0).method_39415(ImageEntry.bgColor);
			consumer.method_22918(matrix4f, x2, y2, 0).method_39415(ImageEntry.accent);
			consumer.method_22918(matrix4f, x2, y1, 0).method_39415(ImageEntry.accent);
		}

		private double easeInOutCubic(double x) {
			return x < 0.5 ? 4 * x * x * x : 1 - Math.pow(-2 * x + 2, 3) / 2;
		}

		private int lerp(float delta, int start, int end) {
			return class_3532.method_15340(class_3532.method_48781(delta, start, end), start, end);
		}

		@Override
		protected @NotNull class_5250 method_25360() {
			return method_32602(class_2561.method_43471("gallery.image.view"));
		}

		@Override
		protected void method_49604(class_332 guiGraphics, class_327 font, int width, int color) {
			int i = this.method_46426() + width;
			int j = this.method_46426() + this.method_25368() - width;
			method_52718(guiGraphics, font, this.method_25369(), i, this.method_46427() + method_25364() - font.field_2000 - 1, j, this.method_46427() + this.method_25364(), color);
		}
	}

	private static class ImageListEntry extends class_4265.class_4266<ImageListEntry> {

		private final List<ImageEntry> buttons;
		private final int size;
		private final ImageList list;

		public ImageListEntry(int size, ImageList list) {
			this.size = size;
			buttons = new ArrayList<>(size);
			this.list = list;
		}

		@Override
		public @NotNull List<? extends class_6379> method_37025() {
			return buttons;
		}

		@Override
		public void method_25343(class_332 guiGraphics, int index, int top, int left, int width, int height, int mouseX, int mouseY, boolean hovering, float partialTick) {
			if (Math.max(left, list.method_46426()) <= Math.min(left + width, list.method_46426() + list.method_25368()) - 1 &&
				Math.max(top - height, list.method_46427()) <= Math.min(top + height * 2, list.method_46427() + list.method_25364()) - 1) {
				buttons.forEach(e -> {
					e.method_46419(top);
					e.method_25394(guiGraphics, mouseX, mouseY, partialTick);
				});
			} else {
				buttons.forEach(e -> e.method_46419(top));
			}
		}

		public void add(ImageEntry e) {
			buttons.add(e);
			repositionButtons();
		}

		public void remove(ImageEntry e) {
			buttons.remove(e);
			if (buttons.isEmpty()) {
				list.removeEntry(this);
			} else if (buttons.size() < size) {
				list.shiftEntries(this);
			}
			repositionButtons();
		}

		private void repositionButtons() {
			int x = list.method_25342();
			for (ImageEntry e : buttons) {
				e.method_46421(x);
				x += e.method_25368() + entrySpacing;
			}
		}

		public ImageEntry pop() {
			var entry = buttons.removeFirst();
			if (buttons.isEmpty()) {
				list.removeEntry(this);
				list.method_60322();
			} else if (buttons.size() < size) {
				list.shiftEntries(this);
			}
			repositionButtons();
			return entry;
		}

		@Override
		public @NotNull List<? extends class_364> method_25396() {
			return buttons;
		}
	}

	private static class ImageList extends class_4265<ImageListEntry> {

		private final int rowWidth;

		public ImageList(class_310 minecraft, int i, int j, int k, int l, int columns) {
			super(minecraft, i, j, k, l);
			this.rowWidth = columns * (entryWidth + entrySpacing) - entrySpacing;
		}

		@Override
		public int addEntry(ImageListEntry entry) {
			return super.method_25321(entry);
		}

		@Override
		public int method_25322() {
			return rowWidth;
		}

		@Override
		public int method_25342() {
			return this.method_46426() + this.field_22758 / 2 - this.method_25322() / 2;
		}

		@Override
		public boolean removeEntry(ImageListEntry entry) {
			return super.method_25330(entry);
		}

		public void shiftEntries(ImageListEntry origin) {
			int originIndex = method_25396().indexOf(origin);
			int lastIndex = method_25396().size() - 1;
			if (originIndex == lastIndex) {
				return;
			}
			ImageListEntry next = method_25326(originIndex + 1);
			origin.add(next.pop());
		}
	}
}
