/*
 * Copyright © 2024 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.api.chat;

import java.util.List;

import io.github.axolotlclient.api.API;
import io.github.axolotlclient.api.ContextMenu;
import io.github.axolotlclient.api.requests.ChannelRequest;
import io.github.axolotlclient.api.requests.FriendRequest;
import io.github.axolotlclient.api.types.Channel;
import io.github.axolotlclient.api.types.Relation;
import io.github.axolotlclient.api.types.User;
import io.github.axolotlclient.api.util.AlphabeticalComparator;
import io.github.axolotlclient.modules.auth.Auth;
import lombok.Getter;
import net.minecraft.class_11909;
import net.minecraft.class_11910;
import net.minecraft.class_156;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_4280;
import net.minecraft.class_7532;

public class ChatUserListWidget extends class_4280<ChatUserListWidget.UserListEntry> {

	private final ChatScreen screen;

	public ChatUserListWidget(ChatScreen screen, class_310 client, int width, int height, int top, int bottom, int entryHeight) {
		super(client, width, bottom - top, top, entryHeight);
		this.screen = screen;
	}

	public void setUsers(List<User> users, Channel channel) {
		users.stream().sorted((u1, u2) -> new AlphabeticalComparator().compare(u1.getName(), u2.getName())).forEach(user -> addEntry(new UserListEntry(user, channel)));
	}

	@Override
	public int method_25322() {
		return field_22758 - 5;
	}

	public int addEntry(UserListEntry entry) {
		return super.method_25321(entry.init(screen));
	}

	@Override
	protected int method_65507() {
		return method_25342() + field_22758 - 8;
	}

	@Override
	public boolean method_25370() {
		return this.screen.method_25399() == this;
	}

	@Override
	protected boolean method_25351(class_11910 index) {
		return true;
	}

	public class UserListEntry extends Entry<UserListEntry> {

		@Getter
		private final User user;
		private final class_310 client;
		private final Channel channel;
		private ChatScreen screen;

		public UserListEntry(User user, Channel channel) {
			this.client = class_310.method_1551();
			this.user = user;
			this.channel = channel;
		}

		public UserListEntry init(ChatScreen screen) {
			this.screen = screen;
			return this;
		}

		protected static void drawScrollableText(class_332 graphics, class_327 textRenderer, class_2561 text, int left, int top, int right, int bottom, int color) {
			int i = textRenderer.method_27525(text);
			int j = (top + bottom - 9) / 2 + 1;
			int k = right - left;
			if (i > k) {
				int l = i - k;
				double d = (double) class_156.method_658() / 1000.0;
				double e = Math.max((double) l * 0.5, 3.0);
				double f = Math.sin((Math.PI / 2) * Math.cos((Math.PI * 2) * d / e)) / 2.0 + 0.5;
				double g = class_3532.method_16436(f, 0.0, l);
				graphics.method_44379(left, top, right, bottom);
				graphics.method_27535(textRenderer, text, left - (int) g, j, color);
				graphics.method_44380();
			} else {
				graphics.method_27535(textRenderer, text, left, j, color);
			}
		}


		@Override
		public class_2561 getNarration() {
			return class_2561.method_43470(user.getName());
		}

		@Override
		public void renderContent(class_332 graphics, int mouseX, int mouseY, boolean hovered, float tickDelta) {
			var x = getContentX();
			var y = getContentY();
			var entryWidth = getContentWidth();
			var entryHeight = getContentHeight();
			if (hovered && !screen.hasContextMenu()) {
				graphics.method_25294(x - 2, y - 1, x + entryWidth - 3, y + entryHeight + 1, 0x55ffffff);
			}
			drawScrollableText(graphics, client.field_1772, class_2561.method_43470(user.getName()), x + 3 + entryHeight, y + 1,
				x + entryWidth - 6, y + 1 + client.field_1772.field_2000 + 2, -1
			);
			drawScrollableText(graphics, client.field_1772, class_2561.method_43470(user.getStatus().getTitle()), x + 3 + entryHeight,
				y + 12, x + entryWidth - 6, y + 12 + client.field_1772.field_2000 + 2, 8421504);

			class_2960 texture = Auth.getInstance().getSkinTexture(user.getUuid());
			class_7532.method_44445(graphics, texture, x, y, entryHeight, true, false, -1);
		}

		@Override
		public boolean mouseClicked(class_11909 event, boolean doubleClick) {
			ChatUserListWidget.this.method_25313(this);
			if (event.method_74245() == 1) { // right click

				if (!user.equals(API.getInstance().getSelf())) {
					ContextMenu.Builder menu = ContextMenu.builder().title(class_2561.method_43470(user.getName())).spacer();
					if (!channel.isDM()) {
						menu.entry(class_2561.method_43471("api.friends.chat"), buttonWidget -> {
							ChannelRequest.getOrCreateDM(user).whenCompleteAsync((channel, throwable) -> client.execute(
								() -> client.method_1507(new ChatScreen(screen.getParent(), channel))));
						}).spacer();
					}
					if (user.getRelation() != Relation.BLOCKED) {
						if (user.getRelation() != Relation.FRIEND) {
							menu.entry(class_2561.method_43471("api.friends.add"), b -> FriendRequest.getInstance().addFriend(user.getUuid())).spacer();
						}
						menu.entry(class_2561.method_43471("api.users.block"),
							buttonWidget -> FriendRequest.getInstance().blockUser(user)
						);
					} else {
						menu.entry(class_2561.method_43471("api.users.unblock"),
							buttonWidget -> FriendRequest.getInstance().unblockUser(user)
						);
					}
					if (channel.getOwner().equals(API.getInstance().getSelf())) {
						menu.spacer().entry(class_2561.method_43471("api.channel.remove_user"), b -> ChannelRequest.removeUserFromChannel(channel, user));
					}
					screen.setContextMenu(menu.build());
					return true;
				}
			}

			return false;
		}
	}
}
